"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlowLogDestination = exports.FlowLogFileFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const s3_buckets_1 = require("../../s3-buckets");
const formatting_1 = require("../../utils/formatting");
/**
 * The file format options for flow log files delivered to S3.
 *
 * @see [FlowLog DestinationOptions](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-destinationoptions)
 * @see [Flow log files](https://docs.aws.amazon.com/vpc/latest/tgw/flow-logs-s3.html#flow-logs-s3-path)
 */
var FlowLogFileFormat;
(function (FlowLogFileFormat) {
    /**
       * Apache Parquet is a columnar data format. Queries on data in Parquet
       * format are 10 to 100 times faster compared to queries on data in plain
       * text. Data in Parquet format with Gzip compression takes 20 percent less
       * storage space than plain text with Gzip compression.
       */
    FlowLogFileFormat["PARQUET"] = " parquet";
    /**
       * Plain text. This is the default format.
       */
    FlowLogFileFormat["PLAIN_TEXT"] = "plain-text";
})(FlowLogFileFormat = exports.FlowLogFileFormat || (exports.FlowLogFileFormat = {}));
/**
 * Represents a resource that can act as a deliver endpoint for captured flow
 * logs.
 */
class FlowLogDestination {
    /**
       * Represents a CloudWatch log group that will serve as the endpoint where
       * flow logs should be delivered.
       *
       * @see [Publish flow logs to CloudWatch Logs](https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs-cwl.html)
       *
       * @param logGroup The CloudWatch LogGroup where flow logs should be
       * delivered.
       * @param role An IAM role that allows Amazon EC2 to publish flow logs to a
       * CloudWatch Logs log group in your account.
       * @returns A configuration object containing details on how to set up
       * logging to the log group.
       */
    static toCloudWatchLogs(logGroup, role) {
        return {
            bind: (scope) => {
                const resolvedLogGroup = logGroup ?? new aws_logs_1.LogGroup(scope, 'log-group', {
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                    retention: aws_logs_1.RetentionDays.TWO_WEEKS,
                });
                const resolvedRole = role ?? new aws_iam_1.Role(scope, 'role', {
                    assumedBy: new aws_iam_1.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
                    roleName: aws_cdk_lib_1.PhysicalName.GENERATE_IF_NEEDED,
                });
                resolvedRole.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                    actions: [
                        'logs:CreateLogStream',
                        'logs:DescribeLogStreams',
                        'logs:PutLogEvents',
                    ],
                    // TODO - Handle condition for extra security
                    // See: https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs-cwl.html#flow-logs-iam
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: [
                        resolvedLogGroup.logGroupArn,
                    ],
                }));
                return {
                    role: resolvedRole,
                    destinationType: aws_ec2_1.FlowLogDestinationType.CLOUD_WATCH_LOGS,
                    logGroup: logGroup,
                };
            },
        };
    }
    /**
       * Represents a CloudWatch log group that will serve as the endpoint where
       * flow logs should be delivered.
       *
       * @see [Publish flow logs to Amazon S3](https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs-s3.html)
       *
       * @param bucket The S3 Bucket where flow logs should be delivered.
       * @param options Configuration options controlling how flow logs will be
       * written to S3.
       * @returns A configuration object containing details on how to set up
       * logging to the bucket.
       */
    static toS3(bucket, options) {
        return {
            bind: (scope) => {
                const resolvedBucket = bucket ?? new s3_buckets_1.FlowLogsBucket(scope, 'bucket');
                resolvedBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                    actions: [
                        's3:PutObject',
                    ],
                    conditions: {
                        ArnLike: {
                            'aws:SourceArn': aws_cdk_lib_1.Stack.of(scope).formatArn({
                                arnFormat: aws_cdk_lib_1.ArnFormat.NO_RESOURCE_NAME,
                                resource: '*',
                                service: 'logs',
                            }),
                        },
                        StringEquals: {
                            'aws:SourceAccount': aws_cdk_lib_1.Stack.of(scope).account,
                            's3:x-amz-acl': 'bucket-owner-full-control',
                        },
                    },
                    effect: aws_iam_1.Effect.ALLOW,
                    principals: [
                        new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com'),
                    ],
                    resources: [
                        resolvedBucket.arnForObjects('*'),
                    ],
                }));
                resolvedBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                    actions: [
                        's3:GetBucketAcl',
                    ],
                    conditions: {
                        ArnLike: {
                            'aws:SourceArn': aws_cdk_lib_1.Stack.of(scope).formatArn({
                                arnFormat: aws_cdk_lib_1.ArnFormat.NO_RESOURCE_NAME,
                                resource: '*',
                                service: 'logs',
                            }),
                        },
                        StringEquals: {
                            'aws:SourceAccount': aws_cdk_lib_1.Stack.of(scope).account,
                        },
                    },
                    effect: aws_iam_1.Effect.ALLOW,
                    principals: [
                        new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com'),
                    ],
                    resources: [
                        resolvedBucket.bucketArn,
                    ],
                }));
                return {
                    bucket: resolvedBucket,
                    destinationOptions: formatting_1.undefinedIfNoKeys({
                        FileFormat: options?.fileFormat,
                        HiveCompatiblePartitions: options?.hiveCompatiblePartitions,
                        PerHourPartition: options?.perHourPartition,
                    }),
                    destinationType: aws_ec2_1.FlowLogDestinationType.S3,
                    s3Path: options?.keyPrefix ? resolvedBucket.arnForObjects(options?.keyPrefix) : resolvedBucket.bucketArn,
                };
            },
        };
    }
}
exports.FlowLogDestination = FlowLogDestination;
_a = JSII_RTTI_SYMBOL_1;
FlowLogDestination[_a] = { fqn: "cdk-extensions.ec2.FlowLogDestination", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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