"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AlbLogsTable = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const region_info_1 = require("aws-cdk-lib/region-info");
const athena_1 = require("../athena");
const glue_1 = require("../glue");
const data_format_1 = require("../glue/lib/data-format");
class AlbLogsTable extends glue_1.Table {
    /**
       * Creates a new instance of the AlbLogsTable class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope, id, props) {
        const projectionYear = new Date().getFullYear() - 1;
        super(scope, id, {
            columns: [
                new glue_1.BasicColumn({
                    name: 'type',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'time',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'elb',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'client_ip',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'client_port',
                    type: 'int',
                }),
                new glue_1.BasicColumn({
                    name: 'target_ip',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'target_port',
                    type: 'int',
                }),
                new glue_1.BasicColumn({
                    name: 'request_processing_time',
                    type: 'double',
                }),
                new glue_1.BasicColumn({
                    name: 'target_processing_time',
                    type: 'double',
                }),
                new glue_1.BasicColumn({
                    name: 'response_processing_time',
                    type: 'double',
                }),
                new glue_1.BasicColumn({
                    name: 'elb_status_code',
                    type: 'int',
                }),
                new glue_1.BasicColumn({
                    name: 'target_status_code',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'received_bytes',
                    type: 'bigint',
                }),
                new glue_1.BasicColumn({
                    name: 'sent_bytes',
                    type: 'bigint',
                }),
                new glue_1.BasicColumn({
                    name: 'request_verb',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'request_url',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'request_proto',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'user_agent',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'ssl_cipher',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'ssl_protocol',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'target_group_arn',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'trace_id',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'domain_name',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'chosen_cert_arn',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'matched_rule_priority',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'request_creation_time',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'actions_executed',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'redirect_url',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'lambda_error_reason',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'target_port_list',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'target_status_code_list',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'classification',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'classification_reason',
                    type: 'string',
                }),
            ],
            compressed: false,
            dataFormat: {
                inputFormat: data_format_1.InputFormat.TEXT,
                outputFormat: data_format_1.OutputFormat.HIVE_IGNORE_KEY_TEXT,
                serializationLibrary: data_format_1.SerializationLibrary.REGEXP,
            },
            database: props.database,
            description: 'Table used for querying ALB access logs.',
            location: `s3://${props.bucket.bucketName}/${props.s3Prefix ?? ''}AWSLogs/`,
            name: props.name,
            owner: 'hadoop',
            parameters: {
                'EXTERNAL': 'TRUE',
                'projection.day.format': 'yyyy/MM/dd',
                'projection.day.interval': '1',
                'projection.day.range': `${projectionYear}/01/01,NOW`,
                'projection.day.type': 'date',
                'projection.day.interval.unit': 'DAYS',
                'projection.logname.type': 'enum',
                'projection.logname.values': 'elasticloadbalancing',
                'projection.regionname.type': 'enum',
                'projection.regionname.values': region_info_1.RegionInfo.regions.map((x) => {
                    return x.name;
                }).join(','),
                'projection.source.type': 'enum',
                'projection.source.values': aws_cdk_lib_1.Lazy.string({
                    produce: () => {
                        if (aws_cdk_lib_1.Token.isUnresolved(this.stack.account)) {
                            aws_cdk_lib_1.Annotations.of(this).addWarning([
                                `ALB logs table stack for ${this.node.path} is environment agnostic.`,
                                'Cross account partition projection cannot be configured on environment',
                                'agnostic stacks. To enable partition projection please specify an account',
                                `for the ${this.stack.node.path} stack.`,
                            ].join(' '));
                            return this.stack.account;
                        }
                        else {
                            return [...new Set(this.node.root.node.findAll().reduce((prev, cur) => {
                                    if ((cur instanceof aws_cdk_lib_1.Stage || cur instanceof aws_cdk_lib_1.Stack) && cur.account && !aws_cdk_lib_1.Token.isUnresolved(cur.account)) {
                                        prev.push(cur.account);
                                    }
                                    return prev;
                                }, []))].join(',');
                        }
                    },
                }),
                'projection.enabled': 'true',
                'storage.location.template': `s3://${props.bucket.bucketName}/${props.s3Prefix ?? ''}AWSLogs/\${source}/\${logname}/\${regionname}/\${day}`,
            },
            partitionKeys: [
                new glue_1.BasicColumn({
                    name: 'source',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'logname',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'regionname',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'day',
                    type: 'string',
                }),
            ],
            serdeParameters: {
                'input.regex': '([^ ]*) ([^ ]*) ([^ ]*) ([^ ]*):([0-9]*) ([^ ]*)[:-]([0-9]*) ([-.0-9]*) ([-.0-9]*) ([-.0-9]*) (|[-0-9]*) (-|[-0-9]*) ([-0-9]*) ([-0-9]*) \"([^ ]*) (.*) (- |[^ ]*)\" \"([^\"]*)\" ([A-Z0-9-_]+) ([A-Za-z0-9.-]*) ([^ ]*) \"([^\"]*)\" \"([^\"]*)\" \"([^\"]*)\" ([-.0-9]*) ([^ ]*) \"([^\"]*)\" \"([^\"]*)\" \"([^ ]*)\" \"([^\s]+?)\" \"([^\s]+)\" \"([^ ]*)\" \"([^ ]*)\"',
                'serialization.format': '1',
            },
            storedAsSubDirectories: false,
            tableType: glue_1.TableType.EXTERNAL_TABLE,
        });
        this.createQueries = props.createQueries ?? true;
        this.friendlyQueryNames = props.friendlyQueryNames ?? false;
        if (this.createQueries) {
            this.topIpsNamedQuery = new athena_1.NamedQuery(this, 'top-ips-named-query', {
                database: this.database,
                description: 'Gets the 100 most actvie IP addresses by request count.',
                name: this.friendlyQueryNames ? 'alb-top-ips' : undefined,
                queryString: [
                    'SELECT client_ip,',
                    '    COUNT(*) AS requests,',
                    '    COUNT_IF(elb_status_code BETWEEN 400 AND 499) AS errors_4xx,',
                    '    COUNT_IF(elb_status_code BETWEEN 500 AND 599) AS errors_5xx,',
                    '    SUM(sent_bytes) AS sent,',
                    '    SUM(received_bytes) AS received,',
                    '    SUM(sent_bytes + received_bytes) AS total,',
                    '    ARBITRARY(user_agent) as user_agent',
                    `FROM ${this.tableName}`,
                    "WHERE day >= DATE_FORMAT(NOW() - PARSE_DURATION('1d'), '%Y/%m/%d')",
                    "    AND FROM_ISO8601_TIMESTAMP(time) >= NOW() - PARSE_DURATION('1d')",
                    'GROUP BY client_ip',
                    'ORDER by total DESC LIMIT 100;',
                ].join('\n'),
            });
            this.status5xxNamedQuery = new athena_1.NamedQuery(this, 'status-5xx-named-query', {
                database: this.database,
                description: 'Gets the 100 most recent ELB 5XX responses.',
                name: this.friendlyQueryNames ? 'alb-5xx-errors' : undefined,
                queryString: [
                    "SELECT FROM_ISO8601_TIMESTAMP(time) AT TIME ZONE 'UTC' AS time,",
                    "    CONCAT(client_ip, ':', CAST(client_port AS varchar)) AS client,",
                    "    CONCAT(target_ip, ':', CAST(target_port AS varchar)) AS target,",
                    '    CASE',
                    "        WHEN CAST(elb_status_code AS varchar) != target_status_code THEN CONCAT(CAST(elb_status_code AS varchar), ' (', target_status_code, ')')",
                    '        ELSE CAST(elb_status_code AS varchar)',
                    '    END as status,',
                    '    request_verb AS verb,',
                    '    request_url AS url,',
                    '    source AS account,',
                    '    elb',
                    `FROM ${this.tableName}`,
                    'WHERE elb_status_code BETWEEN 500 AND 599',
                    "    AND day >= DATE_FORMAT(NOW() - PARSE_DURATION('1d'), '%Y/%m/%d')",
                    "    AND FROM_ISO8601_TIMESTAMP(time) >= NOW() - PARSE_DURATION('1d')",
                    'ORDER BY time DESC LIMIT 100;',
                ].join('\n'),
            });
        }
    }
}
exports.AlbLogsTable = AlbLogsTable;
_a = JSII_RTTI_SYMBOL_1;
AlbLogsTable[_a] = { fqn: "cdk-extensions.glue_tables.AlbLogsTable", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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