"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlowLogsTable = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const region_info_1 = require("aws-cdk-lib/region-info");
const athena_1 = require("../athena");
const ec2_1 = require("../ec2");
const glue_1 = require("../glue");
const formatting_1 = require("../utils/formatting");
class FlowLogsTable extends glue_1.Table {
    /**
       * Creates a new instance of the FlowLogsTable class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope, id, props) {
        const projectionYear = new Date().getFullYear() - 1;
        super(scope, id, {
            compressed: false,
            dataFormat: {
                inputFormat: glue_1.InputFormat.TEXT,
                outputFormat: glue_1.OutputFormat.HIVE_IGNORE_KEY_TEXT,
                serializationLibrary: glue_1.SerializationLibrary.LAZY_SIMPLE,
            },
            database: props.database,
            description: 'Table used for querying network traffic flow.',
            location: `s3://${props.bucket.bucketName}/${props.s3Prefix ?? ''}AWSLogs/`,
            name: props.name,
            owner: 'hadoop',
            parameters: {
                'EXTERNAL': 'TRUE',
                'projection.day.format': 'yyyy/MM/dd',
                'projection.day.interval': '1',
                'projection.day.range': `${projectionYear}/01/01,NOW`,
                'projection.day.type': 'date',
                'projection.day.interval.unit': 'DAYS',
                'projection.logname.type': 'enum',
                'projection.logname.values': 'vpcflowlogs',
                'projection.regionname.type': 'enum',
                'projection.regionname.values': region_info_1.RegionInfo.regions.map((x) => {
                    return x.name;
                }).join(','),
                'projection.source.type': 'enum',
                'projection.source.values': aws_cdk_lib_1.Lazy.string({
                    produce: () => {
                        if (aws_cdk_lib_1.Token.isUnresolved(this.stack.account)) {
                            aws_cdk_lib_1.Annotations.of(this).addWarning([
                                `Flow logs table stack for ${this.node.path} is environment agnostic.`,
                                'Cross account partition projection cannot be configured on environment',
                                'agnostic stacks. To enable partition projection please specify an account',
                                `for the ${this.stack.node.path} stack.`,
                            ].join(' '));
                            return this.stack.account;
                        }
                        else {
                            return [...new Set(this.node.root.node.findAll().reduce((prev, cur) => {
                                    if ((cur instanceof aws_cdk_lib_1.Stage || cur instanceof aws_cdk_lib_1.Stack) && cur.account && !aws_cdk_lib_1.Token.isUnresolved(cur.account)) {
                                        prev.push(cur.account);
                                    }
                                    return prev;
                                }, []))].join(',');
                        }
                    },
                }),
                'projection.enabled': 'true',
                'skip.header.line.count': '1',
                'storage.location.template': `s3://${props.bucket.bucketName}/${props.s3Prefix ?? ''}AWSLogs/\${source}/\${logname}/\${regionname}/\${day}`,
            },
            partitionKeys: [
                new glue_1.BasicColumn({
                    name: 'source',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'logname',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'regionname',
                    type: 'string',
                }),
                new glue_1.BasicColumn({
                    name: 'day',
                    type: 'string',
                }),
            ],
            serdeParameters: {
                'field.delim': ' ',
                'serialization.format': ' ',
            },
            storedAsSubDirectories: false,
            tableType: glue_1.TableType.EXTERNAL_TABLE,
        });
        this.createQueries = props.createQueries ?? true;
        this.format = props.format ?? ec2_1.FlowLogFormat.V2;
        this.friendlyQueryNames = props.friendlyQueryNames ?? false;
        const internalRejectedFields = [
            ec2_1.FlowLogField.ACTION,
            ec2_1.FlowLogField.DSTADDR,
            ec2_1.FlowLogField.DSTPORT,
            ec2_1.FlowLogField.PROTOCOL,
            ec2_1.FlowLogField.SRCADDR,
            ec2_1.FlowLogField.SRCPORT,
            ec2_1.FlowLogField.START,
        ];
        if (this.createQueries) {
            if (formatting_1.includesAll(this.format.fields, internalRejectedFields)) {
                this.internalRejectedNamedQuery = new athena_1.NamedQuery(this, 'internal-rejected-named-query', {
                    database: this.database,
                    description: 'Gets the 100 most recent rejected packets that stayed within the private network ranges.',
                    name: this.friendlyQueryNames ? 'flow-logs-internal-rejected' : undefined,
                    queryString: [
                        'SELECT FROM_UNIXTIME("start", \'UTC\') AS "timestamp",',
                        '    CASE',
                        "        WHEN protocol = 1 THEN 'ICMP'",
                        "        WHEN protocol = 6 THEN 'TCP'",
                        "        WHEN protocol = 17 THEN 'UDP'",
                        '        ELSE CAST(protocol AS varchar)',
                        '    END AS proto,',
                        "    IF(protocol IN (6, 17), CONCAT(srcaddr, ':', CAST(srcport AS varchar)), srcaddr) AS source,",
                        "    IF(protocol IN (6, 17), CONCAT(dstaddr, ':', CAST(dstport AS varchar)), dstaddr) AS destination,",
                        '    action',
                        `FROM ${this.tableName}`,
                        "WHERE REGEXP_LIKE(srcaddr, '^(10\.|192\.168\.|172\.(1[6-9]|2[0-9]|3[0-1])\.).*')",
                        "    AND REGEXP_LIKE(dstaddr, '^(10\.|192\.168\.|172\.(1[6-9]|2[0-9]|3[0-1])\.).*')",
                        "    AND day >= DATE_FORMAT(NOW() - PARSE_DURATION('1d'), '%Y/%m/%d')",
                        "    AND start >= TO_UNIXTIME(NOW() - PARSE_DURATION('1d'))",
                        "    AND action = 'REJECT'",
                        'ORDER BY start DESC LIMIT 100;',
                    ].join('\n'),
                });
            }
        }
    }
    renderStorageDescriptor() {
        this.format.fields.forEach((x) => {
            this.addColumn(new glue_1.BasicColumn({
                name: x.name.toLowerCase().replace(/[^a-z0-9]+/g, ''),
                type: x.type === ec2_1.FlowLogDataType.STRING ? 'string' : 'int',
            }));
        });
        return super.renderStorageDescriptor();
    }
}
exports.FlowLogsTable = FlowLogsTable;
_a = JSII_RTTI_SYMBOL_1;
FlowLogsTable[_a] = { fqn: "cdk-extensions.glue_tables.FlowLogsTable", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmxvdy1sb2dzLXRhYmxlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2dsdWUtdGFibGVzL2Zsb3ctbG9ncy10YWJsZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDZDQUFvRjtBQUdwRix5REFBcUQ7QUFFckQsc0NBQXVDO0FBQ3ZDLGdDQUFzRTtBQUN0RSxrQ0FBbUg7QUFDbkgsb0RBQWtEO0FBMkNsRCxNQUFhLGFBQWMsU0FBUSxZQUFLO0lBcUJ0Qzs7Ozs7OztTQU9LO0lBQ0wsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF5QjtRQUNqRSxNQUFNLGNBQWMsR0FBRyxJQUFJLElBQUksRUFBRSxDQUFDLFdBQVcsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUVwRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNmLFVBQVUsRUFBRSxLQUFLO1lBQ2pCLFVBQVUsRUFBRTtnQkFDVixXQUFXLEVBQUUsa0JBQVcsQ0FBQyxJQUFJO2dCQUM3QixZQUFZLEVBQUUsbUJBQVksQ0FBQyxvQkFBb0I7Z0JBQy9DLG9CQUFvQixFQUFFLDJCQUFvQixDQUFDLFdBQVc7YUFDdkQ7WUFDRCxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7WUFDeEIsV0FBVyxFQUFFLCtDQUErQztZQUM1RCxRQUFRLEVBQUUsUUFBUSxLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVUsSUFBSSxLQUFLLENBQUMsUUFBUSxJQUFJLEVBQUUsVUFBVTtZQUMzRSxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUk7WUFDaEIsS0FBSyxFQUFFLFFBQVE7WUFDZixVQUFVLEVBQUU7Z0JBQ1YsVUFBVSxFQUFFLE1BQU07Z0JBQ2xCLHVCQUF1QixFQUFFLFlBQVk7Z0JBQ3JDLHlCQUF5QixFQUFFLEdBQUc7Z0JBQzlCLHNCQUFzQixFQUFFLEdBQUcsY0FBYyxZQUFZO2dCQUNyRCxxQkFBcUIsRUFBRSxNQUFNO2dCQUM3Qiw4QkFBOEIsRUFBRSxNQUFNO2dCQUN0Qyx5QkFBeUIsRUFBRSxNQUFNO2dCQUNqQywyQkFBMkIsRUFBRSxhQUFhO2dCQUMxQyw0QkFBNEIsRUFBRSxNQUFNO2dCQUNwQyw4QkFBOEIsRUFBRSx3QkFBVSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRTtvQkFDM0QsT0FBTyxDQUFDLENBQUMsSUFBSSxDQUFDO2dCQUNoQixDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDO2dCQUNaLHdCQUF3QixFQUFFLE1BQU07Z0JBQ2hDLDBCQUEwQixFQUFFLGtCQUFJLENBQUMsTUFBTSxDQUFDO29CQUN0QyxPQUFPLEVBQUUsR0FBRyxFQUFFO3dCQUNaLElBQUksbUJBQUssQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsRUFBRTs0QkFDMUMseUJBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsVUFBVSxDQUFDO2dDQUM5Qiw2QkFBNkIsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLDJCQUEyQjtnQ0FDdEUsd0VBQXdFO2dDQUN4RSwyRUFBMkU7Z0NBQzNFLFdBQVcsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxTQUFTOzZCQUN6QyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDOzRCQUNiLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUM7eUJBQzNCOzZCQUFNOzRCQUNMLE9BQU8sQ0FBQyxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLEVBQUUsR0FBRyxFQUFFLEVBQUU7b0NBQ3BFLElBQUksQ0FBQyxHQUFHLFlBQVksbUJBQUssSUFBSSxHQUFHLFlBQVksbUJBQUssQ0FBQyxJQUFJLEdBQUcsQ0FBQyxPQUFPLElBQUksQ0FBQyxtQkFBSyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLEVBQUU7d0NBQ3JHLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxDQUFDO3FDQUN4QjtvQ0FFRCxPQUFPLElBQUksQ0FBQztnQ0FDZCxDQUFDLEVBQUUsRUFBYyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQzt5QkFDaEM7b0JBQ0gsQ0FBQztpQkFDRixDQUFDO2dCQUNGLG9CQUFvQixFQUFFLE1BQU07Z0JBQzVCLHdCQUF3QixFQUFFLEdBQUc7Z0JBQzdCLDJCQUEyQixFQUFFLFFBQVEsS0FBSyxDQUFDLE1BQU0sQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDLFFBQVEsSUFBSSxFQUFFLHVEQUF1RDthQUM1STtZQUNELGFBQWEsRUFBRTtnQkFDYixJQUFJLGtCQUFXLENBQUM7b0JBQ2QsSUFBSSxFQUFFLFFBQVE7b0JBQ2QsSUFBSSxFQUFFLFFBQVE7aUJBQ2YsQ0FBQztnQkFDRixJQUFJLGtCQUFXLENBQUM7b0JBQ2QsSUFBSSxFQUFFLFNBQVM7b0JBQ2YsSUFBSSxFQUFFLFFBQVE7aUJBQ2YsQ0FBQztnQkFDRixJQUFJLGtCQUFXLENBQUM7b0JBQ2QsSUFBSSxFQUFFLFlBQVk7b0JBQ2xCLElBQUksRUFBRSxRQUFRO2lCQUNmLENBQUM7Z0JBQ0YsSUFBSSxrQkFBVyxDQUFDO29CQUNkLElBQUksRUFBRSxLQUFLO29CQUNYLElBQUksRUFBRSxRQUFRO2lCQUNmLENBQUM7YUFDSDtZQUNELGVBQWUsRUFBRTtnQkFDZixhQUFhLEVBQUUsR0FBRztnQkFDbEIsc0JBQXNCLEVBQUUsR0FBRzthQUM1QjtZQUNELHNCQUFzQixFQUFFLEtBQUs7WUFDN0IsU0FBUyxFQUFFLGdCQUFTLENBQUMsY0FBYztTQUNwQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDO1FBQ2pELElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sSUFBSSxtQkFBYSxDQUFDLEVBQUUsQ0FBQztRQUMvQyxJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixJQUFJLEtBQUssQ0FBQztRQUU1RCxNQUFNLHNCQUFzQixHQUFHO1lBQzdCLGtCQUFZLENBQUMsTUFBTTtZQUNuQixrQkFBWSxDQUFDLE9BQU87WUFDcEIsa0JBQVksQ0FBQyxPQUFPO1lBQ3BCLGtCQUFZLENBQUMsUUFBUTtZQUNyQixrQkFBWSxDQUFDLE9BQU87WUFDcEIsa0JBQVksQ0FBQyxPQUFPO1lBQ3BCLGtCQUFZLENBQUMsS0FBSztTQUNuQixDQUFDO1FBRUYsSUFBSSxJQUFJLENBQUMsYUFBYSxFQUFFO1lBQ3RCLElBQUksd0JBQVcsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxzQkFBc0IsQ0FBQyxFQUFFO2dCQUMzRCxJQUFJLENBQUMsMEJBQTBCLEdBQUcsSUFBSSxtQkFBVSxDQUFDLElBQUksRUFBRSwrQkFBK0IsRUFBRTtvQkFDdEYsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO29CQUN2QixXQUFXLEVBQUUsMEZBQTBGO29CQUN2RyxJQUFJLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDLENBQUMsU0FBUztvQkFDekUsV0FBVyxFQUFFO3dCQUNYLHdEQUF3RDt3QkFDeEQsVUFBVTt3QkFDVix1Q0FBdUM7d0JBQ3ZDLHNDQUFzQzt3QkFDdEMsdUNBQXVDO3dCQUN2Qyx3Q0FBd0M7d0JBQ3hDLG1CQUFtQjt3QkFDbkIsaUdBQWlHO3dCQUNqRyxzR0FBc0c7d0JBQ3RHLFlBQVk7d0JBQ1osUUFBUSxJQUFJLENBQUMsU0FBUyxFQUFFO3dCQUN4QixrRkFBa0Y7d0JBQ2xGLG9GQUFvRjt3QkFDcEYsc0VBQXNFO3dCQUN0RSw0REFBNEQ7d0JBQzVELDJCQUEyQjt3QkFDM0IsZ0NBQWdDO3FCQUNqQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7aUJBQ2IsQ0FBQyxDQUFDO2FBQ0o7U0FDRjtJQUNILENBQUM7SUFFUyx1QkFBdUI7UUFDL0IsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUU7WUFDL0IsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLGtCQUFXLENBQUM7Z0JBQzdCLElBQUksRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLE9BQU8sQ0FBQyxhQUFhLEVBQUUsRUFBRSxDQUFDO2dCQUNyRCxJQUFJLEVBQUUsQ0FBQyxDQUFDLElBQUksS0FBSyxxQkFBZSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxLQUFLO2FBQzNELENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPLEtBQUssQ0FBQyx1QkFBdUIsRUFBRSxDQUFDO0lBQ3pDLENBQUM7O0FBbEtILHNDQW1LQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFubm90YXRpb25zLCBMYXp5LCBSZXNvdXJjZVByb3BzLCBTdGFjaywgU3RhZ2UsIFRva2VuIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgQ2ZuVGFibGUgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZ2x1ZSc7XG5pbXBvcnQgeyBJQnVja2V0IH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLXMzJztcbmltcG9ydCB7IFJlZ2lvbkluZm8gfSBmcm9tICdhd3MtY2RrLWxpYi9yZWdpb24taW5mbyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IE5hbWVkUXVlcnkgfSBmcm9tICcuLi9hdGhlbmEnO1xuaW1wb3J0IHsgRmxvd0xvZ0RhdGFUeXBlLCBGbG93TG9nRmllbGQsIEZsb3dMb2dGb3JtYXQgfSBmcm9tICcuLi9lYzInO1xuaW1wb3J0IHsgQmFzaWNDb2x1bW4sIERhdGFiYXNlLCBJbnB1dEZvcm1hdCwgT3V0cHV0Rm9ybWF0LCBTZXJpYWxpemF0aW9uTGlicmFyeSwgVGFibGUsIFRhYmxlVHlwZSB9IGZyb20gJy4uL2dsdWUnO1xuaW1wb3J0IHsgaW5jbHVkZXNBbGwgfSBmcm9tICcuLi91dGlscy9mb3JtYXR0aW5nJztcblxuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIEZsb3dMb2dzVGFibGVcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBGbG93TG9nc1RhYmxlUHJvcHMgZXh0ZW5kcyBSZXNvdXJjZVByb3BzIHtcbiAgLyoqXG4gICAqIEEgYnVja2V0IHdoZXJlIGxvZ3Mgd2lsbCBiZSBzdG9yZWRcbiAgICpcbiAgICogQHNlZSBbQVdTIFMzIGlCdWNrZXRdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jZGsvYXBpL3YyL2RvY3MvYXdzLWNkay1saWIuYXdzX3MzLklCdWNrZXQuaHRtbClcbiAgICovXG4gIHJlYWRvbmx5IGJ1Y2tldDogSUJ1Y2tldDtcbiAgLyoqXG4gICAqIEJvb2xlYW4gaW5kaWNhdGluZyB3aGV0aGVyIHRvIGNyZWF0ZSBkZWZhdWx0IEF0aGVuYSBxdWVyaWVzIGZvciB0aGUgRmxvdyBMb2dzXG4gICAqXG4gICAqIEBzZWUgW2BDZm5OYW1lZFF1ZXJpZXNgXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2RrL2FwaS92MS9weXRob24vYXdzX2Nkay5hd3NfYXRoZW5hL0Nmbk5hbWVkUXVlcnkuaHRtbClcbiAgICovXG4gIHJlYWRvbmx5IGNyZWF0ZVF1ZXJpZXM/OiBib29sZWFuO1xuICAvKipcbiAgICogQSBjZGstZXh0ZW5zaW9ucy9nbHVlIHtAbGluayBhd3MtZ2x1ZSFEYXRhYmFzZSB9IG9iamVjdCB0aGF0IHRoZSB0YWJsZSBzaG91bGQgYmUgY3JlYXRlZCBpbi5cbiAgICpcbiAgICogQHNlZSBbQVdTOjpHbHVlOjpEYXRhYmFzZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWdsdWUtZGF0YWJhc2UuaHRtbClcbiAgICovXG4gIHJlYWRvbmx5IGRhdGFiYXNlOiBEYXRhYmFzZTtcbiAgLyoqXG4gICAqIEEgY2RrLWV4dGVudGlvbnMvZWMyIHtAbGluayBhd3MtZWMyIUZsb3dMb2dGb3JtYXQgfSBvYmplY3QgZGVmaW5pbmcgdGhlIGRlc2lyZWQgZm9ybWF0dGluZyBmb3IgRmxvdyBMb2dzXG4gICAqL1xuICByZWFkb25seSBmb3JtYXQ/OiBGbG93TG9nRm9ybWF0O1xuICAvKipcbiAgICogQm9vbGVhbiBmb3IgYWRkaW5nIFwiZnJpZW5kbHkgbmFtZXNcIiBmb3IgdGhlIGNyZWF0ZWQgQXRoZW5hIHF1ZXJpZXMuXG4gICAqL1xuICByZWFkb25seSBmcmllbmRseVF1ZXJ5TmFtZXM/OiBib29sZWFuO1xuICAvKipcbiAgICogTmFtZSBmb3IgRmxvdyBMb2dzIFRhYmxlXG4gICAqL1xuICByZWFkb25seSBuYW1lPzogc3RyaW5nO1xuICAvKipcbiAgICogU2V0IGEgY3VzdG9tIHByZWZpeCBmb3IgdGhlIFMzIEJ1Y2tldFxuICAgKi9cbiAgcmVhZG9ubHkgczNQcmVmaXg/OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBjbGFzcyBGbG93TG9nc1RhYmxlIGV4dGVuZHMgVGFibGUge1xuICAvLyBJbnB1dCBwcm9wZXJ0aWVzXG4gIC8qKlxuICAgKiBCb29sZWFuIGluZGljYXRpbmcgd2hldGhlciB0byBjcmVhdGUgZGVmYXVsdCBBdGhlbmEgcXVlcmllcyBmb3IgdGhlIEZsb3cgTG9nc1xuICAgKlxuICAgKiBAc2VlIFtgQ2ZuTmFtZWRRdWVyaWVzYF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nkay9hcGkvdjEvcHl0aG9uL2F3c19jZGsuYXdzX2F0aGVuYS9DZm5OYW1lZFF1ZXJ5Lmh0bWwpXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY3JlYXRlUXVlcmllczogYm9vbGVhbjtcbiAgLyoqXG4gICAqIEEgY2RrLWV4dGVudGlvbnMvZWMyIHtAbGluayBhd3MtZWMyIUZsb3dMb2dGb3JtYXQgfSBvYmplY3QgZGVmaW5pbmcgdGhlIGRlc2lyZWQgZm9ybWF0dGluZyBmb3IgRmxvdyBMb2dzXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZm9ybWF0OiBGbG93TG9nRm9ybWF0O1xuICAvKipcbiAgICogQm9vbGVhbiBmb3IgYWRkaW5nIFwiZnJpZW5kbHkgbmFtZXNcIiBmb3IgdGhlIGNyZWF0ZWQgQXRoZW5hIHF1ZXJpZXMuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZnJpZW5kbHlRdWVyeU5hbWVzOiBib29sZWFuO1xuXG4gIC8vIFJlc291cmNlIHByb3BlcnRpZXNcbiAgcHVibGljIHJlYWRvbmx5IGludGVybmFsUmVqZWN0ZWROYW1lZFF1ZXJ5PzogTmFtZWRRdWVyeTtcblxuXG4gIC8qKlxuICAgICAqIENyZWF0ZXMgYSBuZXcgaW5zdGFuY2Ugb2YgdGhlIEZsb3dMb2dzVGFibGUgY2xhc3MuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gc2NvcGUgQSBDREsgQ29uc3RydWN0IHRoYXQgd2lsbCBzZXJ2ZSBhcyB0aGlzIHN0YWNrJ3MgcGFyZW50IGluIHRoZSBjb25zdHJ1Y3QgdHJlZS5cbiAgICAgKiBAcGFyYW0gaWQgQSBuYW1lIHRvIGJlIGFzc29jaWF0ZWQgd2l0aCB0aGUgc3RhY2sgYW5kIHVzZWQgaW4gcmVzb3VyY2UgbmFtaW5nLiBNdXN0IGJlIHVuaXF1ZVxuICAgICAqIHdpdGhpbiB0aGUgY29udGV4dCBvZiAnc2NvcGUnLlxuICAgICAqIEBwYXJhbSBwcm9wcyBBcmd1bWVudHMgcmVsYXRlZCB0byB0aGUgY29uZmlndXJhdGlvbiBvZiB0aGUgcmVzb3VyY2UuXG4gICAgICovXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBGbG93TG9nc1RhYmxlUHJvcHMpIHtcbiAgICBjb25zdCBwcm9qZWN0aW9uWWVhciA9IG5ldyBEYXRlKCkuZ2V0RnVsbFllYXIoKSAtIDE7XG5cbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIGNvbXByZXNzZWQ6IGZhbHNlLFxuICAgICAgZGF0YUZvcm1hdDoge1xuICAgICAgICBpbnB1dEZvcm1hdDogSW5wdXRGb3JtYXQuVEVYVCxcbiAgICAgICAgb3V0cHV0Rm9ybWF0OiBPdXRwdXRGb3JtYXQuSElWRV9JR05PUkVfS0VZX1RFWFQsXG4gICAgICAgIHNlcmlhbGl6YXRpb25MaWJyYXJ5OiBTZXJpYWxpemF0aW9uTGlicmFyeS5MQVpZX1NJTVBMRSxcbiAgICAgIH0sXG4gICAgICBkYXRhYmFzZTogcHJvcHMuZGF0YWJhc2UsXG4gICAgICBkZXNjcmlwdGlvbjogJ1RhYmxlIHVzZWQgZm9yIHF1ZXJ5aW5nIG5ldHdvcmsgdHJhZmZpYyBmbG93LicsXG4gICAgICBsb2NhdGlvbjogYHMzOi8vJHtwcm9wcy5idWNrZXQuYnVja2V0TmFtZX0vJHtwcm9wcy5zM1ByZWZpeCA/PyAnJ31BV1NMb2dzL2AsXG4gICAgICBuYW1lOiBwcm9wcy5uYW1lLFxuICAgICAgb3duZXI6ICdoYWRvb3AnLFxuICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAnRVhURVJOQUwnOiAnVFJVRScsXG4gICAgICAgICdwcm9qZWN0aW9uLmRheS5mb3JtYXQnOiAneXl5eS9NTS9kZCcsXG4gICAgICAgICdwcm9qZWN0aW9uLmRheS5pbnRlcnZhbCc6ICcxJyxcbiAgICAgICAgJ3Byb2plY3Rpb24uZGF5LnJhbmdlJzogYCR7cHJvamVjdGlvblllYXJ9LzAxLzAxLE5PV2AsXG4gICAgICAgICdwcm9qZWN0aW9uLmRheS50eXBlJzogJ2RhdGUnLFxuICAgICAgICAncHJvamVjdGlvbi5kYXkuaW50ZXJ2YWwudW5pdCc6ICdEQVlTJyxcbiAgICAgICAgJ3Byb2plY3Rpb24ubG9nbmFtZS50eXBlJzogJ2VudW0nLFxuICAgICAgICAncHJvamVjdGlvbi5sb2duYW1lLnZhbHVlcyc6ICd2cGNmbG93bG9ncycsXG4gICAgICAgICdwcm9qZWN0aW9uLnJlZ2lvbm5hbWUudHlwZSc6ICdlbnVtJyxcbiAgICAgICAgJ3Byb2plY3Rpb24ucmVnaW9ubmFtZS52YWx1ZXMnOiBSZWdpb25JbmZvLnJlZ2lvbnMubWFwKCh4KSA9PiB7XG4gICAgICAgICAgcmV0dXJuIHgubmFtZTtcbiAgICAgICAgfSkuam9pbignLCcpLFxuICAgICAgICAncHJvamVjdGlvbi5zb3VyY2UudHlwZSc6ICdlbnVtJyxcbiAgICAgICAgJ3Byb2plY3Rpb24uc291cmNlLnZhbHVlcyc6IExhenkuc3RyaW5nKHtcbiAgICAgICAgICBwcm9kdWNlOiAoKSA9PiB7XG4gICAgICAgICAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKHRoaXMuc3RhY2suYWNjb3VudCkpIHtcbiAgICAgICAgICAgICAgQW5ub3RhdGlvbnMub2YodGhpcykuYWRkV2FybmluZyhbXG4gICAgICAgICAgICAgICAgYEZsb3cgbG9ncyB0YWJsZSBzdGFjayBmb3IgJHt0aGlzLm5vZGUucGF0aH0gaXMgZW52aXJvbm1lbnQgYWdub3N0aWMuYCxcbiAgICAgICAgICAgICAgICAnQ3Jvc3MgYWNjb3VudCBwYXJ0aXRpb24gcHJvamVjdGlvbiBjYW5ub3QgYmUgY29uZmlndXJlZCBvbiBlbnZpcm9ubWVudCcsXG4gICAgICAgICAgICAgICAgJ2Fnbm9zdGljIHN0YWNrcy4gVG8gZW5hYmxlIHBhcnRpdGlvbiBwcm9qZWN0aW9uIHBsZWFzZSBzcGVjaWZ5IGFuIGFjY291bnQnLFxuICAgICAgICAgICAgICAgIGBmb3IgdGhlICR7dGhpcy5zdGFjay5ub2RlLnBhdGh9IHN0YWNrLmAsXG4gICAgICAgICAgICAgIF0uam9pbignICcpKTtcbiAgICAgICAgICAgICAgcmV0dXJuIHRoaXMuc3RhY2suYWNjb3VudDtcbiAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgIHJldHVybiBbLi4ubmV3IFNldCh0aGlzLm5vZGUucm9vdC5ub2RlLmZpbmRBbGwoKS5yZWR1Y2UoKHByZXYsIGN1cikgPT4ge1xuICAgICAgICAgICAgICAgIGlmICgoY3VyIGluc3RhbmNlb2YgU3RhZ2UgfHwgY3VyIGluc3RhbmNlb2YgU3RhY2spICYmIGN1ci5hY2NvdW50ICYmICFUb2tlbi5pc1VucmVzb2x2ZWQoY3VyLmFjY291bnQpKSB7XG4gICAgICAgICAgICAgICAgICBwcmV2LnB1c2goY3VyLmFjY291bnQpO1xuICAgICAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgICAgIHJldHVybiBwcmV2O1xuICAgICAgICAgICAgICB9LCBbXSBhcyBzdHJpbmdbXSkpXS5qb2luKCcsJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSxcbiAgICAgICAgfSksXG4gICAgICAgICdwcm9qZWN0aW9uLmVuYWJsZWQnOiAndHJ1ZScsXG4gICAgICAgICdza2lwLmhlYWRlci5saW5lLmNvdW50JzogJzEnLFxuICAgICAgICAnc3RvcmFnZS5sb2NhdGlvbi50ZW1wbGF0ZSc6IGBzMzovLyR7cHJvcHMuYnVja2V0LmJ1Y2tldE5hbWV9LyR7cHJvcHMuczNQcmVmaXggPz8gJyd9QVdTTG9ncy9cXCR7c291cmNlfS9cXCR7bG9nbmFtZX0vXFwke3JlZ2lvbm5hbWV9L1xcJHtkYXl9YCxcbiAgICAgIH0sXG4gICAgICBwYXJ0aXRpb25LZXlzOiBbXG4gICAgICAgIG5ldyBCYXNpY0NvbHVtbih7XG4gICAgICAgICAgbmFtZTogJ3NvdXJjZScsXG4gICAgICAgICAgdHlwZTogJ3N0cmluZycsXG4gICAgICAgIH0pLFxuICAgICAgICBuZXcgQmFzaWNDb2x1bW4oe1xuICAgICAgICAgIG5hbWU6ICdsb2duYW1lJyxcbiAgICAgICAgICB0eXBlOiAnc3RyaW5nJyxcbiAgICAgICAgfSksXG4gICAgICAgIG5ldyBCYXNpY0NvbHVtbih7XG4gICAgICAgICAgbmFtZTogJ3JlZ2lvbm5hbWUnLFxuICAgICAgICAgIHR5cGU6ICdzdHJpbmcnLFxuICAgICAgICB9KSxcbiAgICAgICAgbmV3IEJhc2ljQ29sdW1uKHtcbiAgICAgICAgICBuYW1lOiAnZGF5JyxcbiAgICAgICAgICB0eXBlOiAnc3RyaW5nJyxcbiAgICAgICAgfSksXG4gICAgICBdLFxuICAgICAgc2VyZGVQYXJhbWV0ZXJzOiB7XG4gICAgICAgICdmaWVsZC5kZWxpbSc6ICcgJyxcbiAgICAgICAgJ3NlcmlhbGl6YXRpb24uZm9ybWF0JzogJyAnLFxuICAgICAgfSxcbiAgICAgIHN0b3JlZEFzU3ViRGlyZWN0b3JpZXM6IGZhbHNlLFxuICAgICAgdGFibGVUeXBlOiBUYWJsZVR5cGUuRVhURVJOQUxfVEFCTEUsXG4gICAgfSk7XG5cbiAgICB0aGlzLmNyZWF0ZVF1ZXJpZXMgPSBwcm9wcy5jcmVhdGVRdWVyaWVzID8/IHRydWU7XG4gICAgdGhpcy5mb3JtYXQgPSBwcm9wcy5mb3JtYXQgPz8gRmxvd0xvZ0Zvcm1hdC5WMjtcbiAgICB0aGlzLmZyaWVuZGx5UXVlcnlOYW1lcyA9IHByb3BzLmZyaWVuZGx5UXVlcnlOYW1lcyA/PyBmYWxzZTtcblxuICAgIGNvbnN0IGludGVybmFsUmVqZWN0ZWRGaWVsZHMgPSBbXG4gICAgICBGbG93TG9nRmllbGQuQUNUSU9OLFxuICAgICAgRmxvd0xvZ0ZpZWxkLkRTVEFERFIsXG4gICAgICBGbG93TG9nRmllbGQuRFNUUE9SVCxcbiAgICAgIEZsb3dMb2dGaWVsZC5QUk9UT0NPTCxcbiAgICAgIEZsb3dMb2dGaWVsZC5TUkNBRERSLFxuICAgICAgRmxvd0xvZ0ZpZWxkLlNSQ1BPUlQsXG4gICAgICBGbG93TG9nRmllbGQuU1RBUlQsXG4gICAgXTtcblxuICAgIGlmICh0aGlzLmNyZWF0ZVF1ZXJpZXMpIHtcbiAgICAgIGlmIChpbmNsdWRlc0FsbCh0aGlzLmZvcm1hdC5maWVsZHMsIGludGVybmFsUmVqZWN0ZWRGaWVsZHMpKSB7XG4gICAgICAgIHRoaXMuaW50ZXJuYWxSZWplY3RlZE5hbWVkUXVlcnkgPSBuZXcgTmFtZWRRdWVyeSh0aGlzLCAnaW50ZXJuYWwtcmVqZWN0ZWQtbmFtZWQtcXVlcnknLCB7XG4gICAgICAgICAgZGF0YWJhc2U6IHRoaXMuZGF0YWJhc2UsXG4gICAgICAgICAgZGVzY3JpcHRpb246ICdHZXRzIHRoZSAxMDAgbW9zdCByZWNlbnQgcmVqZWN0ZWQgcGFja2V0cyB0aGF0IHN0YXllZCB3aXRoaW4gdGhlIHByaXZhdGUgbmV0d29yayByYW5nZXMuJyxcbiAgICAgICAgICBuYW1lOiB0aGlzLmZyaWVuZGx5UXVlcnlOYW1lcyA/ICdmbG93LWxvZ3MtaW50ZXJuYWwtcmVqZWN0ZWQnIDogdW5kZWZpbmVkLFxuICAgICAgICAgIHF1ZXJ5U3RyaW5nOiBbXG4gICAgICAgICAgICAnU0VMRUNUIEZST01fVU5JWFRJTUUoXCJzdGFydFwiLCBcXCdVVENcXCcpIEFTIFwidGltZXN0YW1wXCIsJyxcbiAgICAgICAgICAgICcgICAgQ0FTRScsXG4gICAgICAgICAgICBcIiAgICAgICAgV0hFTiBwcm90b2NvbCA9IDEgVEhFTiAnSUNNUCdcIixcbiAgICAgICAgICAgIFwiICAgICAgICBXSEVOIHByb3RvY29sID0gNiBUSEVOICdUQ1AnXCIsXG4gICAgICAgICAgICBcIiAgICAgICAgV0hFTiBwcm90b2NvbCA9IDE3IFRIRU4gJ1VEUCdcIixcbiAgICAgICAgICAgICcgICAgICAgIEVMU0UgQ0FTVChwcm90b2NvbCBBUyB2YXJjaGFyKScsXG4gICAgICAgICAgICAnICAgIEVORCBBUyBwcm90bywnLFxuICAgICAgICAgICAgXCIgICAgSUYocHJvdG9jb2wgSU4gKDYsIDE3KSwgQ09OQ0FUKHNyY2FkZHIsICc6JywgQ0FTVChzcmNwb3J0IEFTIHZhcmNoYXIpKSwgc3JjYWRkcikgQVMgc291cmNlLFwiLFxuICAgICAgICAgICAgXCIgICAgSUYocHJvdG9jb2wgSU4gKDYsIDE3KSwgQ09OQ0FUKGRzdGFkZHIsICc6JywgQ0FTVChkc3Rwb3J0IEFTIHZhcmNoYXIpKSwgZHN0YWRkcikgQVMgZGVzdGluYXRpb24sXCIsXG4gICAgICAgICAgICAnICAgIGFjdGlvbicsXG4gICAgICAgICAgICBgRlJPTSAke3RoaXMudGFibGVOYW1lfWAsXG4gICAgICAgICAgICBcIldIRVJFIFJFR0VYUF9MSUtFKHNyY2FkZHIsICdeKDEwXFwufDE5MlxcLjE2OFxcLnwxNzJcXC4oMVs2LTldfDJbMC05XXwzWzAtMV0pXFwuKS4qJylcIixcbiAgICAgICAgICAgIFwiICAgIEFORCBSRUdFWFBfTElLRShkc3RhZGRyLCAnXigxMFxcLnwxOTJcXC4xNjhcXC58MTcyXFwuKDFbNi05XXwyWzAtOV18M1swLTFdKVxcLikuKicpXCIsXG4gICAgICAgICAgICBcIiAgICBBTkQgZGF5ID49IERBVEVfRk9STUFUKE5PVygpIC0gUEFSU0VfRFVSQVRJT04oJzFkJyksICclWS8lbS8lZCcpXCIsXG4gICAgICAgICAgICBcIiAgICBBTkQgc3RhcnQgPj0gVE9fVU5JWFRJTUUoTk9XKCkgLSBQQVJTRV9EVVJBVElPTignMWQnKSlcIixcbiAgICAgICAgICAgIFwiICAgIEFORCBhY3Rpb24gPSAnUkVKRUNUJ1wiLFxuICAgICAgICAgICAgJ09SREVSIEJZIHN0YXJ0IERFU0MgTElNSVQgMTAwOycsXG4gICAgICAgICAgXS5qb2luKCdcXG4nKSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgcHJvdGVjdGVkIHJlbmRlclN0b3JhZ2VEZXNjcmlwdG9yKCk6IENmblRhYmxlLlN0b3JhZ2VEZXNjcmlwdG9yUHJvcGVydHkgfCB1bmRlZmluZWQge1xuICAgIHRoaXMuZm9ybWF0LmZpZWxkcy5mb3JFYWNoKCh4KSA9PiB7XG4gICAgICB0aGlzLmFkZENvbHVtbihuZXcgQmFzaWNDb2x1bW4oe1xuICAgICAgICBuYW1lOiB4Lm5hbWUudG9Mb3dlckNhc2UoKS5yZXBsYWNlKC9bXmEtejAtOV0rL2csICcnKSxcbiAgICAgICAgdHlwZTogeC50eXBlID09PSBGbG93TG9nRGF0YVR5cGUuU1RSSU5HID8gJ3N0cmluZycgOiAnaW50JyxcbiAgICAgIH0pKTtcbiAgICB9KTtcblxuICAgIHJldHVybiBzdXBlci5yZW5kZXJTdG9yYWdlRGVzY3JpcHRvcigpO1xuICB9XG59XG4iXX0=