"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Connection = exports.ConnectionType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
const formatting_1 = require("../utils/formatting");
var ConnectionType;
(function (ConnectionType) {
    /**
     * JDBC - Designates a connection to a database through Java Database Connectivity (JDBC).
     */
    ConnectionType["JDBC"] = "JDBC";
    /**
     * KAFKA - Designates a connection to an Apache Kafka streaming platform.
     */
    ConnectionType["KAFKA"] = "KAFKA";
    /**
     * MONGODB - Designates a connection to a MongoDB document database.
     */
    ConnectionType["MONGODB"] = "MONGODB";
    /**
     * NETWORK - Designates a network connection to a data source within an Amazon Virtual Private Cloud environment (Amazon VPC).
     */
    ConnectionType["NETWORK"] = "NETWORK";
})(ConnectionType = exports.ConnectionType || (exports.ConnectionType = {}));
/**
 * Creates a resource specifying a Glue Connection to a data source.
 *
 * @see [AWS::Glue::Connection](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-connection.html)
 */
class Connection extends aws_cdk_lib_1.Resource {
    /**
   * Creates a new instance of the Connection class
   *
   * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
   */
    constructor(scope, id, props) {
        super(scope, id, props);
        // Internal properties
        this._matchCriteria = [];
        this._properties = {};
        this.connectionType = props.connectionType;
        this.description = props.description;
        this.name = props.name;
        this.securityGroups = props.securityGroups ?? [];
        this.subnets = props.subnets ?? {
            onePerAz: true,
        };
        this.vpc = props.vpc;
        const properties = props?.properties ?? {};
        Object.keys(properties).forEach((x) => {
            this.addProperty(x, properties[x]);
        });
        const subnet = this.vpc?.selectSubnets(this.subnets).subnets[0];
        if (this.vpc && this.securityGroups.length === 0) {
            this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'security-group', {
                description: 'Provides connectivity for Glue Connection',
                vpc: this.vpc,
            });
            this.securityGroups.push(this.securityGroup);
        }
        this.connections = new aws_ec2_1.Connections({
            securityGroups: this.securityGroups,
        });
        this.connections.allowFromAnyIpv4(aws_ec2_1.Port.allTraffic(), 'Required by Glue Connections');
        this.resource = new aws_glue_1.CfnConnection(this, 'Resource', {
            catalogId: this.stack.account,
            connectionInput: {
                connectionProperties: aws_cdk_lib_1.Lazy.uncachedAny({
                    produce: () => {
                        return !!Object.keys(this._properties).length ? this._properties : undefined;
                    },
                }),
                connectionType: this.connectionType,
                description: this.description,
                matchCriteria: aws_cdk_lib_1.Lazy.uncachedList({
                    produce: () => {
                        return this._matchCriteria;
                    },
                }, {
                    omitEmpty: true,
                }),
                name: this.name,
                physicalConnectionRequirements: formatting_1.undefinedIfNoKeys({
                    availabilityZone: subnet?.availabilityZone,
                    securityGroupIdList: !!!this.vpc ? undefined : this.securityGroups.map((x) => {
                        return x.securityGroupId;
                    }),
                    subnetId: subnet?.subnetId,
                }),
            },
        });
        this.connectionArn = this.stack.formatArn({
            resource: 'connection',
            resourceName: this.resource.ref,
            service: 'glue',
        });
        this.connectionName = this.resource.ref;
    }
    addMatchCriteria(value) {
        this._matchCriteria.push(value);
    }
    addProperty(key, value) {
        this._properties[key] = value;
    }
}
exports.Connection = Connection;
_a = JSII_RTTI_SYMBOL_1;
Connection[_a] = { fqn: "cdk-extensions.glue.Connection", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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