"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Crawler = exports.UpdateBehavior = exports.TableUpdateBehavior = exports.TableGroupingPolicy = exports.RecrawlBehavior = exports.PartitionUpdateBehavior = exports.DeleteBehavior = exports.ConfigurationVersion = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const formatting_1 = require("../utils/formatting");
var ConfigurationVersion;
(function (ConfigurationVersion) {
    ConfigurationVersion[ConfigurationVersion["V1_0"] = 1] = "V1_0";
})(ConfigurationVersion = exports.ConfigurationVersion || (exports.ConfigurationVersion = {}));
var DeleteBehavior;
(function (DeleteBehavior) {
    DeleteBehavior["DELETE_FROM_DATABASE"] = "DELETE_FROM_DATABASE";
    DeleteBehavior["DEPRECATE_IN_DATABASE"] = "DEPRECATE_IN_DATABASE";
    DeleteBehavior["LOG"] = "LOG";
})(DeleteBehavior = exports.DeleteBehavior || (exports.DeleteBehavior = {}));
var PartitionUpdateBehavior;
(function (PartitionUpdateBehavior) {
    PartitionUpdateBehavior["INHERIT_FROM_TABLE"] = "InheritFromTable";
})(PartitionUpdateBehavior = exports.PartitionUpdateBehavior || (exports.PartitionUpdateBehavior = {}));
var RecrawlBehavior;
(function (RecrawlBehavior) {
    RecrawlBehavior["EVENT_MODE"] = "CRAWL_EVENT_MODE";
    RecrawlBehavior["EVERYTHING"] = "CRAWL_EVERYTHING";
    RecrawlBehavior["NEW_FOLDERS_ONLY"] = "CRAWL_NEW_FOLDERS_ONLY";
})(RecrawlBehavior = exports.RecrawlBehavior || (exports.RecrawlBehavior = {}));
var TableGroupingPolicy;
(function (TableGroupingPolicy) {
    TableGroupingPolicy["COMBINE_COMPATIBLE_SCHEMAS"] = "CombineCompatibleSchemas";
})(TableGroupingPolicy = exports.TableGroupingPolicy || (exports.TableGroupingPolicy = {}));
var TableUpdateBehavior;
(function (TableUpdateBehavior) {
    TableUpdateBehavior["MERGE_NEW_COLUMNS"] = "MergeNewColumns";
})(TableUpdateBehavior = exports.TableUpdateBehavior || (exports.TableUpdateBehavior = {}));
var UpdateBehavior;
(function (UpdateBehavior) {
    UpdateBehavior["UPDATE_IN_DATABASE"] = "UPDATE_IN_DATABASE";
    UpdateBehavior["LOG"] = "LOG";
})(UpdateBehavior = exports.UpdateBehavior || (exports.UpdateBehavior = {}));
class CrawlerBase extends aws_cdk_lib_1.Resource {
}
/**
 * Create a Crawler resource to pull information from the provided resource.
 *
 * @see [AWS::Glue::Crawler](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-crawler.html)
 */
class Crawler extends CrawlerBase {
    /**
     * Creates a new instance of the Crawler class.
     *
     * @param scope A CDK Construct that will serve as this resource's parent in
     * the construct tree.
     * @param id A name to be associated with the stack and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param props Arguments related to the configuration of the resource.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        // Internal properties
        this._classifiers = [];
        this._targets = [];
        this.configuration = props.configuration;
        this.database = props.database;
        this.deleteBehavior = props.deleteBehavior;
        this.description = props.description;
        this.name = props.name;
        this.recrawlBehavior = props.recrawlBehavior;
        this.scheduleExpression = props.scheduleExpression;
        this.securityConfiguration = props.securityConfiguration;
        this.tablePrefix = props.tablePrefix;
        this.updateBehavior = props.updateBehavior;
        props.targets?.forEach((x) => {
            this.addTarget(x);
        });
        this.role = new aws_iam_1.Role(this, 'role', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole'),
            ],
        });
        this.resource = new aws_glue_1.CfnCrawler(this, 'Resource', {
            classifiers: aws_cdk_lib_1.Lazy.uncachedList({
                produce: () => {
                    return this._classifiers;
                },
            }, {
                omitEmpty: true,
            }),
            configuration: this.renderConfiguration(),
            crawlerSecurityConfiguration: this.securityConfiguration?.securityConfigurationName,
            databaseName: this.database?.databaseName,
            description: this.description,
            name: this.name,
            recrawlPolicy: formatting_1.undefinedIfNoKeys({
                recrawlBehavior: this.recrawlBehavior,
            }),
            role: this.role.roleArn,
            schedule: formatting_1.undefinedIfNoKeys({
                scheduleExpression: this.scheduleExpression?.expressionString,
            }),
            schemaChangePolicy: formatting_1.undefinedIfNoKeys({
                deleteBehavior: this.deleteBehavior,
                updateBehavior: this.updateBehavior,
            }),
            tablePrefix: this.tablePrefix,
            targets: aws_cdk_lib_1.Lazy.uncachedAny({
                produce: () => {
                    return this.renderTargets();
                },
            }),
        });
        this.crawlerArn = this.stack.formatArn({
            resource: 'crawler',
            resourceName: this.resource.ref,
            service: 'glue',
        });
        this.crawlerName = this.resource.ref;
    }
    /**
     * Imports an existing crawler using its Amazon Resource Name (ARN).
     *
     * @param scope A CDK Construct that will serve as this resource's parent in
     * the construct tree.
     * @param id A name to be associated with the stack and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param crawlerArn The ARN of the crawler to import.
     * @returns An object representing the crawler that was imported.
     */
    static fromCrawlerArn(scope, id, crawlerArn) {
        class Import extends CrawlerBase {
            constructor() {
                super(...arguments);
                this.crawlerArn = crawlerArn;
                this.crawlerName = aws_cdk_lib_1.Arn.split(crawlerArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports an existing crawler using its name.
     *
     * @param scope A CDK Construct that will serve as this resource's parent in
     * the construct tree.
     * @param id A name to be associated with the stack and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param crawlerName The name of the crawler to import.
     * @returns An object representing the crawler that was imported.
     */
    static fromCrawlerName(scope, id, crawlerName) {
        return Crawler.fromCrawlerArn(scope, id, aws_cdk_lib_1.Stack.of(scope).formatArn({
            resource: 'crawler',
            resourceName: crawlerName,
            service: 'glue',
        }));
    }
    addClassifier(classifier) {
        this._classifiers.push(classifier);
    }
    addTarget(target) {
        this._targets.push(target);
    }
    renderConfiguration() {
        const configuration = formatting_1.definedFieldsOrUndefined({
            CrawlerOutput: formatting_1.definedFieldsOrUndefined({
                Partitions: formatting_1.definedFieldsOrUndefined({
                    AddOrUpdateBehavior: this.configuration?.partitionUpdateBehavior,
                }),
                Tables: formatting_1.definedFieldsOrUndefined({
                    AddOrUpdateBehavior: this.configuration?.tableUpdateBehavior,
                }),
            }),
            Grouping: formatting_1.definedFieldsOrUndefined({
                TableGroupingPolicy: this.configuration?.tableGroupingPolicy,
                TableLevelConfiguration: this.configuration?.tableLevel,
            }),
        });
        return configuration === undefined ? undefined : this.stack.toJsonString({
            Version: this.configuration?.version ?? ConfigurationVersion.V1_0,
            ...configuration,
        });
    }
    renderTargets() {
        const resolved = this._targets.map((x) => {
            return x.bind(this);
        });
        return formatting_1.undefinedIfNoKeys({
            catalogTargets: formatting_1.flattenedOrUndefined(resolved.map((x) => {
                return x.catalogTargets ?? [];
            })),
            dynamoDbTargets: formatting_1.flattenedOrUndefined(resolved.map((x) => {
                return x.dynamoDbTargets ?? [];
            })),
            jdbcTargets: formatting_1.flattenedOrUndefined(resolved.map((x) => {
                return x.jdbcTargets ?? [];
            })),
            s3Targets: formatting_1.flattenedOrUndefined(resolved.map((x) => {
                return x.s3Targets ?? [];
            })),
        });
    }
}
exports.Crawler = Crawler;
_a = JSII_RTTI_SYMBOL_1;
Crawler[_a] = { fqn: "cdk-extensions.glue.Crawler", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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