import { Duration, Resource, ResourceProps } from 'aws-cdk-lib';
import { CfnJob } from 'aws-cdk-lib/aws-glue';
import { IRole } from 'aws-cdk-lib/aws-iam';
import { ILogGroup } from 'aws-cdk-lib/aws-logs';
import { Construct, IConstruct } from 'constructs';
import { Connection } from './connection';
import { JobExecutable } from './lib/job-executable';
import { WorkerType } from './lib/worker-type';
import { SecurityConfiguration } from './security-configuration';
/**
 * Represnets a Glue Job in AWS.
 */
export interface IJob extends IConstruct {
    /**
     * The Amazon Resource Name (ARN) of the job.
     */
    readonly jobArn: string;
    /**
      * The name of the job.
      */
    readonly jobName: string;
}
declare abstract class JobBase extends Resource implements IJob {
    /**
     * The Amazon Resource Name (ARN) of the job.
     */
    abstract readonly jobArn: string;
    /**
     * The name of the job.
     */
    abstract readonly jobName: string;
}
export interface ContinuousLoggingProps {
    /**
       * Apply the provided conversion pattern.
       *
       * This is a Log4j Conversion Pattern to customize driver and executor logs.
       *
       * @default `%d{yy/MM/dd HH:mm:ss} %p %c{1}: %m%n`
       */
    readonly conversionPattern?: string;
    /**
       * Enable continouous logging.
       */
    readonly enabled: boolean;
    /**
       * Specify a custom CloudWatch log group name.
       *
       * @default - a log group is created with name `/aws-glue/jobs/logs-v2/`.
       */
    readonly logGroup?: ILogGroup;
    /**
       * Specify a custom CloudWatch log stream prefix.
       *
       * @default - the job run ID.
       */
    readonly logStreamPrefix?: string;
    /**
       * Filter out non-useful Apache Spark driver/executor and Apache Hadoop YARN heartbeat log messages.
       *
       * @default true
       */
    readonly quiet?: boolean;
}
/**
 * Configuration for the Glue Job resource.
 */
export interface JobProps extends ResourceProps {
    /**
     * The number of capacity units that are allocated to this job.
     *
     * @see [AWS::Glue::Job](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-job.html#cfn-glue-job-allocatedcapacity)
     */
    readonly allocatedCapacity?: number;
    /**
     * List of Connections for use with this job.
     *
     * @see [AWS::Glue::Job](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-job.html#cfn-glue-job-connections)
     */
    readonly connections?: Connection[];
    /**
     * Set of properties for configuration of Continuous Logging
     */
    readonly continuousLogging?: ContinuousLoggingProps;
    /**
     * The default arguments for this job, specified as name-value pairs.
     *
     * You can specify arguments here that your own job-execution script consumes, in addition to arguments that AWS Glue itself consumes.
     *
     * @see [AWS::Glue::Job](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-job.html#cfn-glue-job-defaultarguments)
     */
    readonly defaultArguments?: {
        [key: string]: string;
    };
    /**
     * A description of the job.
     */
    readonly description?: string;
    /**
     * Boolean value for whether to enable Profiling Metrics
     */
    readonly enableProfilingMetrics?: boolean;
    /**
     * Executable properties for the Job
     */
    readonly executable: JobExecutable;
    /**
     * The number of AWS Glue data processing units (DPUs) that can be allocated when this job runs. A DPU is a relative measure of processing power that consists of 4 vCPUs of compute capacity and 16 GB of memory.
     *
     * Do not set Max Capacity if using WorkerType and NumberOfWorkers.
     *
     * The value that can be allocated for MaxCapacity depends on whether you are running a Python shell job or an Apache Spark ETL job:
     *
     *    - When you specify a Python shell job (JobCommand.Name="pythonshell"), you can allocate either 0.0625 or 1 DPU. The default is 0.0625 DPU.
     *
     *    - When you specify an Apache Spark ETL job (JobCommand.Name="glueetl"), you can allocate from 2 to 100 DPUs. The default is 10 DPUs. This job type cannot have a fractional DPU allocation.
     *
     */
    readonly maxCapacity?: number;
    /**
     * Maximum number of concurrent executions
     *
     * @see [AWS::Glue::Job ExecutionProperty](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-glue-job-executionproperty.html)
     */
    readonly maxConcurrentRuns?: number;
    /**
     * The maximum number of times to retry this job after a JobRun fails.
     *
     * @see [AWS::Glue::Job](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-job.html#cfn-glue-job-maxretries)
     */
    readonly maxRetries?: number;
    /**
     * A name for the Job
     */
    readonly name?: string;
    /**
     * After a job run starts, the number of minutes to wait before sending a job run delay notification.
     *
     * @see [AWS::Glue::Job NotificationProperty](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-glue-job-notificationproperty.html)
     */
    readonly notifyDelayAfter?: Duration;
    /**
     * The name or Amazon Resource Name (ARN) of the IAM role associated with this job.
     */
    readonly role?: IRole;
    /**
     * The Security Configuration object to be applied to the Job
     */
    readonly securityConfiguration?: SecurityConfiguration;
    /**
     * The job timeout in minutes. This is the maximum time that a job run can consume resources before it is terminated and enters TIMEOUT status. The default is 2,880 minutes (48 hours).
     *
     * @see [AWS::Glue::Job](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-job.html#cfn-glue-job-timeout)
     */
    readonly timeout?: Duration;
    /**
     * The number of worker available the Job
     */
    readonly workerCount?: number;
    /**
     * The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or G.2X.
     *
     * @see [AWS::Glue::Job](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-job.html#cfn-glue-job-workertype)
     */
    readonly workerType?: WorkerType;
}
/**
 * Creates a Glue Job
 *
 * @see [AWS::Glue::Job](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-job.html)
 */
export declare class Job extends JobBase {
    /**
     * Imports an existing job using its Amazon Resource Name (ARN).
     *
     * @param scope A CDK Construct that will serve as this resource's parent in
     * the construct tree.
     * @param id A name to be associated with the stack and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param jobArn The ARN of the job to import.
     * @returns An object representing the job that was imported.
     */
    static fromJobArn(scope: IConstruct, id: string, jobArn: string): IJob;
    /**
     * Imports an existing job using its name.
     *
     * @param scope A CDK Construct that will serve as this resource's parent in
     * the construct tree.
     * @param id A name to be associated with the stack and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param jobName The name of the job to import.
     * @returns An object representing the job that was imported.
     */
    static fromJobName(scope: IConstruct, id: string, jobName: string): IJob;
    private readonly _arguments;
    private readonly _connections;
    /**
      * {@link JobProps.allocatedCapacity }
      */
    readonly allocatedCapacity?: number;
    /**
      * {@link JobProps.connections}
      */
    readonly connections?: Connection[];
    /**
      * {@link JobProps.continuousLogging}
      */
    readonly continuousLogging?: ContinuousLoggingProps;
    /**
      * {@link JobProps.description}
      */
    readonly description?: string;
    /**
      * {@link JobProps.executable:}
      */
    readonly executable: JobExecutable;
    /**
      * {@link JobProps.maxCapacity}
      */
    readonly maxCapacity?: number;
    /**
      * {@link JobProps.maxConcurrentRuns}
      */
    readonly maxConcurrentRuns?: number;
    /**
      * {@link JobProps.maxRetries}
      */
    readonly maxRetries?: number;
    /**
      * {@link JobProps.name}
      */
    readonly name?: string;
    /**
      * {@link JobProps.notifyDelayAfter}
      */
    readonly notifyDelayAfter?: Duration;
    /**
      * {@link JobProps.securityConfiguration}
      */
    readonly securityConfiguration?: SecurityConfiguration;
    /**
      * {@link JobProps.timeout}
      */
    readonly timeout?: Duration;
    /**
      * {@link JobProps.workerCount}
      */
    readonly workerCount?: number;
    /**
      * {@link JobProps.workerType}
      */
    readonly workerType?: WorkerType;
    readonly logGroup?: ILogGroup;
    readonly resource: CfnJob;
    readonly role: IRole;
    readonly jobArn: string;
    readonly jobName: string;
    /**
   * Creates a new instance of the Job class
   *
   * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
   */
    constructor(scope: Construct, id: string, props: JobProps);
    addArgument(key: string, value: string): void;
    addConnection(connection: Connection): void;
    private buildCodeS3ObjectUrl;
    private configureContinuousLogging;
}
export {};
