"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Job = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const formatting_1 = require("../utils/formatting");
class JobBase extends aws_cdk_lib_1.Resource {
}
/**
 * Creates a Glue Job
 *
 * @see [AWS::Glue::Job](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-job.html)
 */
class Job extends JobBase {
    /**
   * Creates a new instance of the Job class
   *
   * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
   */
    constructor(scope, id, props) {
        super(scope, id, props);
        // Internal properties
        this._arguments = {};
        this._connections = [];
        this.allocatedCapacity = props.allocatedCapacity;
        this.continuousLogging = props.continuousLogging;
        this.description = props.description;
        this.executable = props.executable;
        this.maxCapacity = props.maxCapacity;
        this.maxConcurrentRuns = props.maxConcurrentRuns;
        this.maxRetries = props.maxRetries;
        this.name = props.name;
        this.notifyDelayAfter = props.notifyDelayAfter;
        this.securityConfiguration = props.securityConfiguration;
        this.timeout = props.timeout;
        this.workerCount = props.workerCount;
        this.workerType = props.workerType;
        const executable = props.executable.bind();
        if (props.enableProfilingMetrics) {
            this.addArgument('--enable-metrics', '');
        }
        if (this.continuousLogging?.enabled) {
            this.logGroup = this.configureContinuousLogging(this.continuousLogging);
        }
        if (this.workerType?.name === 'G.1X' && (this.workerCount ?? 0) > 299) {
            aws_cdk_lib_1.Annotations.of(this).addError('The maximum number of workers you can define for G.1X is 299.');
        }
        else if (this.workerType?.name === 'G.2X' && (this.workerCount ?? 0) > 149) {
            aws_cdk_lib_1.Annotations.of(this).addError('The maximum number of workers you can define for G.2X is 149.');
        }
        this.role = props.role ?? new aws_iam_1.Role(this, 'role', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole'),
            ],
        });
        this.resource = new aws_glue_1.CfnJob(this, 'Resource', {
            allocatedCapacity: this.allocatedCapacity,
            command: {
                name: executable.type.name,
                pythonVersion: executable.pythonVersion,
                scriptLocation: this.buildCodeS3ObjectUrl(executable.script),
            },
            connections: aws_cdk_lib_1.Lazy.uncachedAny({
                produce: () => {
                    return !!!this._connections.length ? undefined : {
                        connections: this._connections.map((x) => {
                            return x.connectionName;
                        }),
                    };
                },
            }),
            defaultArguments: aws_cdk_lib_1.Lazy.uncachedAny({
                produce: () => {
                    return !!Object.keys(this._arguments).length ? this._arguments : undefined;
                },
            }),
            description: this.description,
            executionProperty: formatting_1.undefinedIfNoKeys({
                maxConcurrentRuns: this.maxConcurrentRuns,
            }),
            glueVersion: executable.glueVersion.name,
            maxCapacity: this.maxCapacity,
            maxRetries: this.maxRetries,
            name: this.name,
            notificationProperty: formatting_1.undefinedIfNoKeys({
                notifyDelayAfter: this.notifyDelayAfter?.toMinutes(),
            }),
            numberOfWorkers: this.workerCount,
            role: this.role.roleArn,
            securityConfiguration: this.securityConfiguration?.securityConfigurationName,
            timeout: this.timeout?.toMinutes(),
            workerType: this.workerType?.name,
        });
        this.jobArn = this.stack.formatArn({
            resource: 'job',
            resourceName: this.resource.ref,
            service: 'glue',
        });
        this.jobName = this.resource.ref;
        const args = props.defaultArguments ?? {};
        Object.keys(args).forEach((x) => {
            this.addArgument(x, args[x]);
        });
        props.connections?.forEach((x) => {
            this.addConnection(x);
        });
    }
    /**
     * Imports an existing job using its Amazon Resource Name (ARN).
     *
     * @param scope A CDK Construct that will serve as this resource's parent in
     * the construct tree.
     * @param id A name to be associated with the stack and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param jobArn The ARN of the job to import.
     * @returns An object representing the job that was imported.
     */
    static fromJobArn(scope, id, jobArn) {
        class Import extends JobBase {
            constructor() {
                super(...arguments);
                this.jobArn = jobArn;
                this.jobName = aws_cdk_lib_1.Arn.split(jobArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports an existing job using its name.
     *
     * @param scope A CDK Construct that will serve as this resource's parent in
     * the construct tree.
     * @param id A name to be associated with the stack and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param jobName The name of the job to import.
     * @returns An object representing the job that was imported.
     */
    static fromJobName(scope, id, jobName) {
        return Job.fromJobArn(scope, id, aws_cdk_lib_1.Stack.of(scope).formatArn({
            resource: 'job',
            resourceName: jobName,
            service: 'glue',
        }));
    }
    addArgument(key, value) {
        this._arguments[key] = value;
    }
    addConnection(connection) {
        this._connections.push(connection);
    }
    buildCodeS3ObjectUrl(code) {
        const s3Location = code.bind(this, this.role).s3Location;
        return `s3://${s3Location.bucketName}/${s3Location.objectKey}`;
    }
    configureContinuousLogging(props) {
        const logGroup = props?.logGroup ?? new aws_logs_1.LogGroup(this, 'log-group', {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            retention: aws_logs_1.RetentionDays.TWO_WEEKS,
        });
        logGroup.grantWrite(this.role);
        this.addArgument('--enable-continuous-cloudwatch-log', 'true');
        this.addArgument('--enable-continuous-log-filter', `${props.quiet ?? true}`);
        this.addArgument('--continuous-log-logGroup', logGroup.logGroupName);
        if (props.logStreamPrefix) {
            this.addArgument('--continuous-log-logStreamPrefix', props.logStreamPrefix);
        }
        if (props.conversionPattern) {
            this.addArgument('--continuous-log-conversionPattern', props.conversionPattern);
        }
        return logGroup;
    }
}
exports.Job = Job;
_a = JSII_RTTI_SYMBOL_1;
Job[_a] = { fqn: "cdk-extensions.glue.Job", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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