"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityConfiguration = exports.JobBookmarksEncryptionMode = exports.CloudWatchEncryptionMode = exports.S3EncryptionMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const formatting_1 = require("../utils/formatting");
/**
 * Encryption mode for S3.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_S3Encryption.html#Glue-Type-S3Encryption-S3EncryptionMode
 */
var S3EncryptionMode;
(function (S3EncryptionMode) {
    /**
       * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
       *
       * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
       */
    S3EncryptionMode["KMS"] = "SSE-KMS";
    /**
       * Server side encryption (SSE) with an Amazon S3-managed key.
       *
       * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
       */
    S3EncryptionMode["S3_MANAGED"] = "SSE-S3";
})(S3EncryptionMode = exports.S3EncryptionMode || (exports.S3EncryptionMode = {}));
/**
 * Encryption mode for CloudWatch Logs.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_CloudWatchEncryption.html#Glue-Type-CloudWatchEncryption-CloudWatchEncryptionMode
 */
var CloudWatchEncryptionMode;
(function (CloudWatchEncryptionMode) {
    /**
       * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
       *
       * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
       */
    CloudWatchEncryptionMode["KMS"] = "SSE-KMS";
})(CloudWatchEncryptionMode = exports.CloudWatchEncryptionMode || (exports.CloudWatchEncryptionMode = {}));
/**
 * Encryption mode for Job Bookmarks.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_JobBookmarksEncryption.html#Glue-Type-JobBookmarksEncryption-JobBookmarksEncryptionMode
 */
var JobBookmarksEncryptionMode;
(function (JobBookmarksEncryptionMode) {
    /**
       * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
       *
       * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
       */
    JobBookmarksEncryptionMode["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(JobBookmarksEncryptionMode = exports.JobBookmarksEncryptionMode || (exports.JobBookmarksEncryptionMode = {}));
class SecurityConfiguration extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.name = props.name ?? aws_cdk_lib_1.Names.uniqueId(this);
        let cloudWatchKey = undefined;
        let jobBookmarksKey = undefined;
        let s3Key = undefined;
        const fetchKey = (key) => {
            return key ?? this.node.tryFindChild('key') ?? new aws_kms_1.Key(this, 'key');
        };
        if (this.cloudWatchEncryption?.mode === CloudWatchEncryptionMode.KMS) {
            cloudWatchKey = fetchKey(this.cloudWatchEncryption.kmsKey);
        }
        else if (this.jobBookmarksEncryption?.mode === JobBookmarksEncryptionMode.CLIENT_SIDE_KMS && !!!this.jobBookmarksEncryption.kmsKey) {
            jobBookmarksKey = fetchKey(this.jobBookmarksEncryption.kmsKey);
        }
        else if (this.s3Encryption?.mode === S3EncryptionMode.KMS && !!!this.s3Encryption.kmsKey) {
            s3Key = fetchKey(this.s3Encryption.kmsKey);
        }
        this.key = this.node.tryFindChild('key');
        this.resource = new aws_glue_1.CfnSecurityConfiguration(this, 'Resource', {
            encryptionConfiguration: {
                cloudWatchEncryption: formatting_1.undefinedIfNoKeys({
                    cloudWatchEncryptionMode: this.cloudWatchEncryption?.mode,
                    kmsKeyArn: cloudWatchKey?.keyArn,
                }),
                jobBookmarksEncryption: formatting_1.undefinedIfNoKeys({
                    jobBookmarksEncryptionMode: this.jobBookmarksEncryption?.mode,
                    kmsKeyArn: jobBookmarksKey?.keyArn,
                }),
                s3Encryptions: !!!this.s3Encryption ? undefined : [{
                        s3EncryptionMode: this.s3Encryption?.mode,
                        kmsKeyArn: s3Key?.keyArn,
                    }],
            },
            name: this.name,
        });
        this.securityConfigurationName = this.resource.ref;
    }
}
exports.SecurityConfiguration = SecurityConfiguration;
_a = JSII_RTTI_SYMBOL_1;
SecurityConfiguration[_a] = { fqn: "cdk-extensions.glue.SecurityConfiguration", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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