import { Resource, ResourceProps } from 'aws-cdk-lib';
import { Schedule } from 'aws-cdk-lib/aws-events';
import { CfnTrigger } from 'aws-cdk-lib/aws-glue';
import { Construct, IConstruct } from 'constructs';
import { Workflow } from './workflow';
export declare enum PredicateOperator {
    AND = "AND",
    OR = "OR"
}
export declare enum TriggerType {
    CONDITIONAL = "CONDITIONAL",
    EVENT = "EVENT",
    ON_DEMAND = "ON_DEMAND",
    SCHEDULED = "SCHEDULED"
}
/**
 * Represents an action that should be taken when a trigger is executed.
 */
export interface ITriggerAction {
    bind(scope: IConstruct): CfnTrigger.ActionProperty;
}
/**
 * Represents a precondition that must be satisfied in order for a trigger to
 * be executed.
 */
export interface ITriggerPredicate {
    bind(scope: IConstruct): CfnTrigger.ConditionProperty;
}
/**
 * Represents a Glue Trigger in AWS.
 */
export interface ITrigger extends IConstruct {
    /**
     * The Amazon Resource Name (ARN) of the trigger.
     */
    readonly triggerArn: string;
    /**
      * The name of the trigger.
      */
    readonly triggerName: string;
}
declare abstract class TriggerBase extends Resource implements ITrigger {
    /**
     * The Amazon Resource Name (ARN) of the trigger.
     */
    abstract readonly triggerArn: string;
    /**
     * The name of the trigger.
     */
    abstract readonly triggerName: string;
}
/**
 * Configuration for the GlueTrigger resource.
 */
export interface TriggerProps extends ResourceProps {
    /**
     * A list of actions initiated by this trigger.
     *
     * @see [Trigger Actions](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-trigger.html#cfn-glue-trigger-actions)
     */
    readonly actions?: ITriggerAction[];
    /**
     * A description for the trigger.
     *
     * @see [Trigger Description](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-trigger.html#cfn-glue-trigger-description)
     */
    readonly description?: string;
    /**
     * A name for the trigger.
     *
     * @see [Trigger Name](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-trigger.html#cfn-glue-trigger-name)
     */
    readonly name?: string;
    /**
     * A list of the conditions that determine when the trigger will fire.
     *
     * @see [Trigger Predicate](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-glue-trigger-predicate.html)
     */
    readonly predicateConditions?: ITriggerPredicate[];
    /**
     * Operator for chaining predicate conditions if multiple are given.
     *
     * @see [Trigger Predicate.Logical](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-glue-trigger-predicate.html#cfn-glue-trigger-predicate-logical)
     */
    readonly predicateOperator?: PredicateOperator;
    /**
     * A cron expression used to specify the schedule.
     *
     * @see [Trigger Schedule](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-trigger.html#cfn-glue-trigger-schedule)
     * @see [Time-Based Schedules for Jobs and Crawlers](https://docs.aws.amazon.com/glue/latest/dg/monitor-data-warehouse-schedule.html)
     */
    readonly schedule?: Schedule;
    /**
     * Set to true to start SCHEDULED and CONDITIONAL triggers when created. True
     * is not supported for ON_DEMAND triggers.
     *
     * @see [Trigger StartOnCreation](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-trigger.html#cfn-glue-trigger-startoncreation)
     */
    readonly startOnCreation?: boolean;
    /**
     * The type of trigger that this is.
     *
     * @see [Trigger Type](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-trigger.html#cfn-glue-trigger-type)
     */
    readonly type: TriggerType;
    /**
     * The name of the workflow associated with the trigger.
     *
     * @see [Trigger WorkflowName](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-trigger.html#cfn-glue-trigger-workflowname)
     */
    readonly workflow?: Workflow;
}
export declare class Trigger extends TriggerBase {
    /**
     * Imports an existing trigger using its Amazon Resource Name (ARN).
     *
     * @param scope A CDK Construct that will serve as this resource's parent in
     * the construct tree.
     * @param id A name to be associated with the stack and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param triggerArn The ARN of the trigger to import.
     * @returns An object representing the trigger that was imported.
     */
    static fromTriggerArn(scope: IConstruct, id: string, triggerArn: string): ITrigger;
    /**
     * Imports an existing trigger using its name.
     *
     * @param scope A CDK Construct that will serve as this resource's parent in
     * the construct tree.
     * @param id A name to be associated with the stack and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param triggerName The name of the trigger to import.
     * @returns An object representing the trigger that was imported.
     */
    static fromTriggerName(scope: IConstruct, id: string, triggerName: string): ITrigger;
    /**
     * Internal collection tracking the actions which should be run by this
     * trigger.
     */
    private readonly _actions;
    /**
     * Internal collection tracking the predicates that serve as preconditions
     * for when this trigger should run.
     */
    private readonly _predicates;
    /**
     * A description for the trigger.
     *
     * @see [Trigger Description](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-trigger.html#cfn-glue-trigger-description)
     *
     * @group Inputs
     */
    readonly description?: string;
    /**
     * A name for the trigger.
     *
     * @see [Trigger Name](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-trigger.html#cfn-glue-trigger-name)
     *
     * @group Inputs
     */
    readonly name?: string;
    /**
     * Operator for chaining predicate conditions if multiple are given.
     *
     * @see [Trigger Predicate.Logical](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-glue-trigger-predicate.html#cfn-glue-trigger-predicate-logical)
     *
     * @group Inputs
     */
    readonly predicateOperator: PredicateOperator;
    /**
     * A cron expression used to specify the schedule.
     *
     * @see [Trigger Schedule](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-trigger.html#cfn-glue-trigger-schedule)
     * @see [Time-Based Schedules for Jobs and Crawlers](https://docs.aws.amazon.com/glue/latest/dg/monitor-data-warehouse-schedule.html)
     *
     * @group Inputs
     */
    readonly schedule?: Schedule;
    /**
     * Set to true to start SCHEDULED and CONDITIONAL triggers when created. True
     * is not supported for ON_DEMAND triggers.
     *
     * @see [Trigger StartOnCreation](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-trigger.html#cfn-glue-trigger-startoncreation)
     *
     * @group Inputs
     */
    readonly startOnCreation?: boolean;
    /**
     * The type of trigger that this is.
     *
     * @see [Trigger Type](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-trigger.html#cfn-glue-trigger-type)
     *
     * @group Inputs
     */
    readonly type: TriggerType;
    /**
     * The name of the workflow associated with the trigger.
     *
     * @see [Trigger WorkflowName](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-trigger.html#cfn-glue-trigger-workflowname)
     *
     * @group Inputs
     */
    readonly workflow?: Workflow;
    /**
     * The underlying Trigger CloudFormation resource.
     *
     * @see [AWS::Glue::Trigger](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-glue-trigger.html)
     *
     * @group Resources
     */
    readonly resource: CfnTrigger;
    /**
     * The Amazon Resource Name (ARN) of the trigger.
     */
    readonly triggerArn: string;
    /**
     * The name of the trigger.
     */
    readonly triggerName: string;
    /**
     * Creates a new instance of the Trigger class.
     *
     * @param scope A CDK Construct that will serve as this resource's parent in
     * the construct tree.
     * @param id A name to be associated with the stack and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param props Arguments related to the configuration of the resource.
     */
    constructor(scope: Construct, id: string, props: TriggerProps);
    /**
     * Registers an action with the trigger. All actions associated with the
     * trigger are run when the conditions to trigger the trigger are met.
     *
     * @param action The action to be run by this trigger.
     * @returns The trigger to which the action was added.
     */
    addAction(action: ITriggerAction): Trigger;
    /**
     * Registers a predicate with the trigger. Triggers with predicates must meet
     * the conditions they specify in order to run.
     *
     * @param predicate The predicate to be added to the trigger.
     * @returns The trigger to which the predicate was added.
     */
    addPredicate(predicate: ITriggerPredicate): Trigger;
}
export {};
