"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Trigger = exports.TriggerType = exports.PredicateOperator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
var PredicateOperator;
(function (PredicateOperator) {
    PredicateOperator["AND"] = "AND";
    PredicateOperator["OR"] = "OR";
})(PredicateOperator = exports.PredicateOperator || (exports.PredicateOperator = {}));
var TriggerType;
(function (TriggerType) {
    TriggerType["CONDITIONAL"] = "CONDITIONAL";
    TriggerType["EVENT"] = "EVENT";
    TriggerType["ON_DEMAND"] = "ON_DEMAND";
    TriggerType["SCHEDULED"] = "SCHEDULED";
})(TriggerType = exports.TriggerType || (exports.TriggerType = {}));
class TriggerBase extends aws_cdk_lib_1.Resource {
}
class Trigger extends TriggerBase {
    /**
     * Creates a new instance of the Trigger class.
     *
     * @param scope A CDK Construct that will serve as this resource's parent in
     * the construct tree.
     * @param id A name to be associated with the stack and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param props Arguments related to the configuration of the resource.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        /**
         * Internal collection tracking the actions which should be run by this
         * trigger.
         */
        this._actions = [];
        /**
         * Internal collection tracking the predicates that serve as preconditions
         * for when this trigger should run.
         */
        this._predicates = [];
        this.description = props.description;
        this.name = props.name;
        this.predicateOperator = props.predicateOperator ?? PredicateOperator.AND;
        this.schedule = props.schedule;
        this.startOnCreation = props.startOnCreation ?? (props.type === TriggerType.ON_DEMAND ? false : true);
        this.type = props.type;
        this.workflow = props.workflow;
        props.actions?.forEach((x) => {
            this.addAction(x);
        });
        props.predicateConditions?.forEach((x) => {
            this.addPredicate(x);
        });
        this.resource = new aws_glue_1.CfnTrigger(this, 'Resource', {
            actions: aws_cdk_lib_1.Lazy.uncachedAny({
                produce: () => {
                    return this._actions.map((x) => {
                        return x.bind(this);
                    });
                },
            }, {
                omitEmptyArray: true,
            }),
            description: this.description,
            name: this.name,
            predicate: aws_cdk_lib_1.Lazy.uncachedAny({
                produce: () => {
                    return !!!this._predicates.length ? undefined : {
                        conditions: this._predicates.map((x) => {
                            return x.bind(this);
                        }),
                        logical: this._predicates.length > 1 ? this.predicateOperator : undefined,
                    };
                },
            }),
            schedule: this.schedule?.expressionString,
            startOnCreation: this.startOnCreation,
            type: this.type,
            workflowName: this.workflow?.workflowName,
        });
        this.triggerArn = this.stack.formatArn({
            resource: 'trigger',
            resourceName: this.resource.ref,
            service: 'glue',
        });
        this.triggerName = this.resource.ref;
    }
    /**
     * Imports an existing trigger using its Amazon Resource Name (ARN).
     *
     * @param scope A CDK Construct that will serve as this resource's parent in
     * the construct tree.
     * @param id A name to be associated with the stack and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param triggerArn The ARN of the trigger to import.
     * @returns An object representing the trigger that was imported.
     */
    static fromTriggerArn(scope, id, triggerArn) {
        class Import extends TriggerBase {
            constructor() {
                super(...arguments);
                this.triggerArn = triggerArn;
                this.triggerName = aws_cdk_lib_1.Arn.split(triggerArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports an existing trigger using its name.
     *
     * @param scope A CDK Construct that will serve as this resource's parent in
     * the construct tree.
     * @param id A name to be associated with the stack and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param triggerName The name of the trigger to import.
     * @returns An object representing the trigger that was imported.
     */
    static fromTriggerName(scope, id, triggerName) {
        return Trigger.fromTriggerArn(scope, id, aws_cdk_lib_1.Stack.of(scope).formatArn({
            resource: 'trigger',
            resourceName: triggerName,
            service: 'glue',
        }));
    }
    /**
     * Registers an action with the trigger. All actions associated with the
     * trigger are run when the conditions to trigger the trigger are met.
     *
     * @param action The action to be run by this trigger.
     * @returns The trigger to which the action was added.
     */
    addAction(action) {
        this._actions.push(action);
        return this;
    }
    /**
     * Registers a predicate with the trigger. Triggers with predicates must meet
     * the conditions they specify in order to run.
     *
     * @param predicate The predicate to be added to the trigger.
     * @returns The trigger to which the predicate was added.
     */
    addPredicate(predicate) {
        this._predicates.push(predicate);
        return this;
    }
}
exports.Trigger = Trigger;
_a = JSII_RTTI_SYMBOL_1;
Trigger[_a] = { fqn: "cdk-extensions.glue.Trigger", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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