"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AdotCollector = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
class AdotCollector extends aws_cdk_lib_1.Resource {
    /**
     * Creates a new instance of the AdotCollector class.
     *
     * @param scope A CDK Construct that will serve as this resource's parent
     * in the construct tree.
     * @param id A name to be associated with the resource and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param props Arguments related to the configuration of the resource.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.cluster = props.cluster;
        this.createNamespace = props.createNamespace ?? true;
        this.namespace = props.namespace ?? AdotCollector.DEFAULT_NAMESPACE;
        this.serviceAccount = new aws_eks_1.ServiceAccount(this, 'service-account', {
            cluster: this.cluster,
            namespace: this.namespace,
            name: `sa${aws_cdk_lib_1.Names.uniqueId(this).slice(-61).toLowerCase()}`,
        });
        this.serviceAccount.role.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchAgentServerPolicy'));
        this.manifest = new aws_eks_1.KubernetesManifest(this, 'Resource', {
            cluster: this.cluster,
            manifest: [
                ...(!this.createNamespace ? [] : [{
                        apiVersion: 'v1',
                        kind: 'Namespace',
                        metadata: {
                            name: this.namespace,
                        },
                    }]),
                {
                    kind: 'ClusterRole',
                    apiVersion: 'rbac.authorization.k8s.io/v1',
                    metadata: {
                        name: 'adotcol-admin-role',
                    },
                    rules: [
                        {
                            apiGroups: [
                                '',
                            ],
                            resources: [
                                'nodes',
                                'nodes/proxy',
                                'nodes/metrics',
                                'services',
                                'endpoints',
                                'pods',
                                'pods/proxy',
                            ],
                            verbs: [
                                'get',
                                'list',
                                'watch',
                            ],
                        },
                        {
                            nonResourceURLs: [
                                '/metrics/cadvisor',
                            ],
                            verbs: [
                                'get',
                                'list',
                                'watch',
                            ],
                        },
                    ],
                },
                {
                    kind: 'ClusterRoleBinding',
                    apiVersion: 'rbac.authorization.k8s.io/v1',
                    metadata: {
                        name: 'adotcol-admin-role-binding',
                    },
                    subjects: [
                        {
                            kind: 'ServiceAccount',
                            name: this.serviceAccount.serviceAccountName,
                            namespace: this.serviceAccount.serviceAccountNamespace,
                        },
                    ],
                    roleRef: {
                        kind: 'ClusterRole',
                        name: 'adotcol-admin-role',
                        apiGroup: 'rbac.authorization.k8s.io',
                    },
                },
                {
                    apiVersion: 'v1',
                    kind: 'ConfigMap',
                    metadata: {
                        name: 'adot-collector-config',
                        namespace: this.namespace,
                        labels: {
                            app: 'aws-adot',
                            component: 'adot-collector-config',
                        },
                    },
                    data: {
                        'adot-collector-config': this.stack.toJsonString({
                            receivers: {
                                prometheus: {
                                    config: {
                                        global: {
                                            scrape_interval: '1m',
                                            scrape_timeout: '40s',
                                        },
                                        scrape_configs: [
                                            {
                                                job_name: 'kubelets-cadvisor-metrics',
                                                sample_limit: 10000,
                                                scheme: 'https',
                                                kubernetes_sd_configs: [
                                                    {
                                                        role: 'node',
                                                    },
                                                ],
                                                tls_config: {
                                                    ca_file: '/var/run/secrets/kubernetes.io/serviceaccount/ca.crt',
                                                },
                                                bearer_token_file: '/var/run/secrets/kubernetes.io/serviceaccount/token',
                                                relabel_configs: [
                                                    {
                                                        action: 'labelmap',
                                                        regex: '__meta_kubernetes_node_label_(.+)',
                                                    },
                                                    {
                                                        target_label: '__address__',
                                                        replacement: 'kubernetes.default.svc:443',
                                                    },
                                                    {
                                                        source_labels: [
                                                            '__meta_kubernetes_node_name',
                                                        ],
                                                        regex: '(.+)',
                                                        target_label: '__metrics_path__',
                                                        replacement: '/api/v1/nodes/$${1}/proxy/metrics/cadvisor',
                                                    },
                                                ],
                                                metric_relabel_configs: [
                                                    {
                                                        action: 'replace',
                                                        source_labels: [
                                                            'id',
                                                        ],
                                                        regex: '^/machine\\.slice/machine-rkt\\\\x2d([^\\\\]+)\\\\.+/([^/]+)\\.service$',
                                                        target_label: 'rkt_container_name',
                                                        replacement: '$${2}-$${1}',
                                                    },
                                                    {
                                                        action: 'replace',
                                                        source_labels: [
                                                            'id',
                                                        ],
                                                        regex: '^/system\\.slice/(.+)\\.service$',
                                                        target_label: 'systemd_service_name',
                                                        replacement: '$${1}',
                                                    },
                                                ],
                                            },
                                        ],
                                    },
                                },
                            },
                            processors: {
                                'metricstransform/label_1': {
                                    transforms: [
                                        {
                                            include: '.*',
                                            match_type: 'regexp',
                                            action: 'update',
                                            operations: [
                                                {
                                                    action: 'update_label',
                                                    label: 'name',
                                                    new_label: 'container_id',
                                                },
                                                {
                                                    action: 'update_label',
                                                    label: 'kubernetes_io_hostname',
                                                    new_label: 'NodeName',
                                                },
                                                {
                                                    action: 'update_label',
                                                    label: 'eks_amazonaws_com_compute_type',
                                                    new_label: 'LaunchType',
                                                },
                                            ],
                                        },
                                    ],
                                },
                                'metricstransform/rename': {
                                    transforms: [
                                        {
                                            include: 'container_spec_cpu_quota',
                                            new_name: 'new_container_cpu_limit_raw',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_spec_cpu_shares',
                                            new_name: 'new_container_cpu_request',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_cpu_usage_seconds_total',
                                            new_name: 'new_container_cpu_usage_seconds_total',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_spec_memory_limit_bytes',
                                            new_name: 'new_container_memory_limit',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_cache',
                                            new_name: 'new_container_memory_cache',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_max_usage_bytes',
                                            new_name: 'new_container_memory_max_usage',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_usage_bytes',
                                            new_name: 'new_container_memory_usage',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_working_set_bytes',
                                            new_name: 'new_container_memory_working_set',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_rss',
                                            new_name: 'new_container_memory_rss',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_swap',
                                            new_name: 'new_container_memory_swap',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_failcnt',
                                            new_name: 'new_container_memory_failcnt',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_failures_total',
                                            new_name: 'new_container_memory_hierarchical_pgfault',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                                failure_type: 'pgfault',
                                                scope: 'hierarchy',
                                            },
                                        },
                                        {
                                            include: 'container_memory_failures_total',
                                            new_name: 'new_container_memory_hierarchical_pgmajfault',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                                failure_type: 'pgmajfault',
                                                scope: 'hierarchy',
                                            },
                                        },
                                        {
                                            include: 'container_memory_failures_total',
                                            new_name: 'new_container_memory_pgfault',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                                failure_type: 'pgfault',
                                                scope: 'container',
                                            },
                                        },
                                        {
                                            include: 'container_memory_failures_total',
                                            new_name: 'new_container_memory_pgmajfault',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                                failure_type: 'pgmajfault',
                                                scope: 'container',
                                            },
                                        },
                                        {
                                            include: 'container_fs_limit_bytes',
                                            new_name: 'new_container_filesystem_capacity',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_fs_usage_bytes',
                                            new_name: 'new_container_filesystem_usage',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                container: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_spec_cpu_quota',
                                            new_name: 'pod_cpu_limit_raw',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                image: '^$',
                                                container: '^$',
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_spec_cpu_shares',
                                            new_name: 'pod_cpu_request',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                image: '^$',
                                                container: '^$',
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_cpu_usage_seconds_total',
                                            new_name: 'pod_cpu_usage_seconds_total',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                image: '^$',
                                                container: '^$',
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_spec_memory_limit_bytes',
                                            new_name: 'pod_memory_limit',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                image: '^$',
                                                container: '^$',
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_cache',
                                            new_name: 'pod_memory_cache',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                image: '^$',
                                                container: '^$',
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_max_usage_bytes',
                                            new_name: 'pod_memory_max_usage',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                image: '^$',
                                                container: '^$',
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_usage_bytes',
                                            new_name: 'pod_memory_usage',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                image: '^$',
                                                container: '^$',
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_working_set_bytes',
                                            new_name: 'pod_memory_working_set',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                image: '^$',
                                                container: '^$',
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_rss',
                                            new_name: 'pod_memory_rss',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                image: '^$',
                                                container: '^$',
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_swap',
                                            new_name: 'pod_memory_swap',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                image: '^$',
                                                container: '^$',
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_failcnt',
                                            new_name: 'pod_memory_failcnt',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                image: '^$',
                                                container: '^$',
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_memory_failures_total',
                                            new_name: 'pod_memory_hierarchical_pgfault',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                image: '^$',
                                                container: '^$',
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                                failure_type: 'pgfault',
                                                scope: 'hierarchy',
                                            },
                                        },
                                        {
                                            include: 'container_memory_failures_total',
                                            new_name: 'pod_memory_hierarchical_pgmajfault',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                image: '^$',
                                                container: '^$',
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                                failure_type: 'pgmajfault',
                                                scope: 'hierarchy',
                                            },
                                        },
                                        {
                                            include: 'container_memory_failures_total',
                                            new_name: 'pod_memory_pgfault',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                image: '^$',
                                                container: '^$',
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                                failure_type: 'pgfault',
                                                scope: 'container',
                                            },
                                        },
                                        {
                                            include: 'container_memory_failures_total',
                                            new_name: 'pod_memory_pgmajfault',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                image: '^$',
                                                container: '^$',
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                                failure_type: 'pgmajfault',
                                                scope: 'container',
                                            },
                                        },
                                        {
                                            include: 'container_network_receive_bytes_total',
                                            new_name: 'pod_network_rx_bytes',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_network_receive_packets_dropped_total',
                                            new_name: 'pod_network_rx_dropped',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_network_receive_errors_total',
                                            new_name: 'pod_network_rx_errors',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_network_receive_packets_total',
                                            new_name: 'pod_network_rx_packets',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_network_transmit_bytes_total',
                                            new_name: 'pod_network_tx_bytes',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_network_transmit_packets_dropped_total',
                                            new_name: 'pod_network_tx_dropped',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_network_transmit_errors_total',
                                            new_name: 'pod_network_tx_errors',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                        {
                                            include: 'container_network_transmit_packets_total',
                                            new_name: 'pod_network_tx_packets',
                                            action: 'insert',
                                            match_type: 'regexp',
                                            experimental_match_labels: {
                                                pod: '\\S',
                                                LaunchType: 'fargate',
                                            },
                                        },
                                    ],
                                },
                                'filter': {
                                    metrics: {
                                        include: {
                                            match_type: 'regexp',
                                            metric_names: [
                                                'new_container_.*',
                                                'pod_.*',
                                            ],
                                        },
                                    },
                                },
                                'cumulativetodelta': {
                                    metrics: [
                                        'new_container_cpu_usage_seconds_total',
                                        'pod_cpu_usage_seconds_total',
                                        'pod_memory_pgfault',
                                        'pod_memory_pgmajfault',
                                        'pod_memory_hierarchical_pgfault',
                                        'pod_memory_hierarchical_pgmajfault',
                                        'pod_network_rx_bytes',
                                        'pod_network_rx_dropped',
                                        'pod_network_rx_errors',
                                        'pod_network_rx_packets',
                                        'pod_network_tx_bytes',
                                        'pod_network_tx_dropped',
                                        'pod_network_tx_errors',
                                        'pod_network_tx_packets',
                                        'new_container_memory_pgfault',
                                        'new_container_memory_pgmajfault',
                                        'new_container_memory_hierarchical_pgfault',
                                        'new_container_memory_hierarchical_pgmajfault',
                                    ],
                                },
                                'deltatorate': {
                                    metrics: [
                                        'new_container_cpu_usage_seconds_total',
                                        'pod_cpu_usage_seconds_total',
                                        'pod_memory_pgfault',
                                        'pod_memory_pgmajfault',
                                        'pod_memory_hierarchical_pgfault',
                                        'pod_memory_hierarchical_pgmajfault',
                                        'pod_network_rx_bytes',
                                        'pod_network_rx_dropped',
                                        'pod_network_rx_errors',
                                        'pod_network_rx_packets',
                                        'pod_network_tx_bytes',
                                        'pod_network_tx_dropped',
                                        'pod_network_tx_errors',
                                        'pod_network_tx_packets',
                                        'new_container_memory_pgfault',
                                        'new_container_memory_pgmajfault',
                                        'new_container_memory_hierarchical_pgfault',
                                        'new_container_memory_hierarchical_pgmajfault',
                                    ],
                                },
                                'experimental_metricsgeneration/1': {
                                    rules: [
                                        {
                                            name: 'pod_network_total_bytes',
                                            unit: 'Bytes/Second',
                                            type: 'calculate',
                                            metric1: 'pod_network_rx_bytes',
                                            metric2: 'pod_network_tx_bytes',
                                            operation: 'add',
                                        },
                                        {
                                            name: 'pod_memory_utilization_over_pod_limit',
                                            unit: 'Percent',
                                            type: 'calculate',
                                            metric1: 'pod_memory_working_set',
                                            metric2: 'pod_memory_limit',
                                            operation: 'percent',
                                        },
                                        {
                                            name: 'pod_cpu_usage_total',
                                            unit: 'Millicore',
                                            type: 'scale',
                                            metric1: 'pod_cpu_usage_seconds_total',
                                            operation: 'multiply',
                                            scale_by: 1000,
                                        },
                                        {
                                            name: 'pod_cpu_limit',
                                            unit: 'Millicore',
                                            type: 'scale',
                                            metric1: 'pod_cpu_limit_raw',
                                            operation: 'divide',
                                            scale_by: 100,
                                        },
                                    ],
                                },
                                'experimental_metricsgeneration/2': {
                                    rules: [
                                        {
                                            name: 'pod_cpu_utilization_over_pod_limit',
                                            type: 'calculate',
                                            unit: 'Percent',
                                            metric1: 'pod_cpu_usage_total',
                                            metric2: 'pod_cpu_limit',
                                            operation: 'percent',
                                        },
                                    ],
                                },
                                'metricstransform/label_2': {
                                    transforms: [
                                        {
                                            include: 'pod_.*',
                                            match_type: 'regexp',
                                            action: 'update',
                                            operations: [
                                                {
                                                    action: 'add_label',
                                                    new_label: 'Type',
                                                    new_value: 'Pod',
                                                },
                                            ],
                                        },
                                        {
                                            include: 'new_container_.*',
                                            match_type: 'regexp',
                                            action: 'update',
                                            operations: [
                                                {
                                                    action: 'add_label',
                                                    new_label: 'Type',
                                                    new_value: 'Container',
                                                },
                                            ],
                                        },
                                        {
                                            include: '.*',
                                            match_type: 'regexp',
                                            action: 'update',
                                            operations: [
                                                {
                                                    action: 'update_label',
                                                    label: 'namespace',
                                                    new_label: 'Namespace',
                                                },
                                                {
                                                    action: 'update_label',
                                                    label: 'pod',
                                                    new_label: 'PodName',
                                                },
                                            ],
                                        },
                                        {
                                            include: '^new_container_(.*)$$',
                                            match_type: 'regexp',
                                            action: 'update',
                                            new_name: 'container_$$1',
                                        },
                                    ],
                                },
                                'resourcedetection': {
                                    detectors: [
                                        'env',
                                        'eks',
                                    ],
                                },
                                'batch': {
                                    timeout: '60s',
                                },
                            },
                            exporters: {
                                awsemf: {
                                    log_group_name: '/aws/containerinsights/{ClusterName}/performance',
                                    log_stream_name: '{PodName}',
                                    namespace: 'ContainerInsights',
                                    region: this.stack.region,
                                    resource_to_telemetry_conversion: {
                                        enabled: true,
                                    },
                                    eks_fargate_container_insights_enabled: true,
                                    parse_json_encoded_attr_values: [
                                        'kubernetes',
                                    ],
                                    dimension_rollup_option: 'NoDimensionRollup',
                                    metric_declarations: [
                                        {
                                            dimensions: [
                                                [
                                                    'ClusterName',
                                                    'LaunchType',
                                                ],
                                                [
                                                    'ClusterName',
                                                    'Namespace',
                                                    'LaunchType',
                                                ],
                                                [
                                                    'ClusterName',
                                                    'Namespace',
                                                    'PodName',
                                                    'LaunchType',
                                                ],
                                            ],
                                            metric_name_selectors: [
                                                'pod_cpu_utilization_over_pod_limit',
                                                'pod_cpu_usage_total',
                                                'pod_cpu_limit',
                                                'pod_memory_utilization_over_pod_limit',
                                                'pod_memory_working_set',
                                                'pod_memory_limit',
                                                'pod_network_rx_bytes',
                                                'pod_network_tx_bytes',
                                            ],
                                        },
                                    ],
                                },
                            },
                            extensions: {
                                health_check: null,
                            },
                            service: {
                                pipelines: {
                                    metrics: {
                                        receivers: [
                                            'prometheus',
                                        ],
                                        processors: [
                                            'metricstransform/label_1',
                                            'resourcedetection',
                                            'metricstransform/rename',
                                            'filter',
                                            'cumulativetodelta',
                                            'deltatorate',
                                            'experimental_metricsgeneration/1',
                                            'experimental_metricsgeneration/2',
                                            'metricstransform/label_2',
                                            'batch',
                                        ],
                                        exporters: [
                                            'awsemf',
                                        ],
                                    },
                                },
                                extensions: [
                                    'health_check',
                                ],
                            },
                        }),
                    },
                },
                {
                    apiVersion: 'v1',
                    kind: 'Service',
                    metadata: {
                        name: 'adot-collector-service',
                        namespace: this.namespace,
                        labels: {
                            app: 'aws-adot',
                            component: 'adot-collector',
                        },
                    },
                    spec: {
                        ports: [
                            {
                                name: 'metrics',
                                port: 8888,
                            },
                        ],
                        selector: {
                            component: 'adot-collector',
                        },
                        type: 'ClusterIP',
                    },
                },
                {
                    apiVersion: 'apps/v1',
                    kind: 'StatefulSet',
                    metadata: {
                        name: 'adot-collector',
                        namespace: this.namespace,
                        labels: {
                            app: 'aws-adot',
                            component: 'adot-collector',
                        },
                    },
                    spec: {
                        selector: {
                            matchLabels: {
                                app: 'aws-adot',
                                component: 'adot-collector',
                            },
                        },
                        serviceName: 'adot-collector-service',
                        template: {
                            metadata: {
                                labels: {
                                    app: 'aws-adot',
                                    component: 'adot-collector',
                                },
                            },
                            spec: {
                                serviceAccountName: this.serviceAccount.serviceAccountName,
                                securityContext: {
                                    fsGroup: 65534,
                                },
                                containers: [{
                                        image: 'amazon/aws-otel-collector:v0.15.1',
                                        name: 'adot-collector',
                                        imagePullPolicy: 'Always',
                                        command: [
                                            '/awscollector',
                                            '--config=/conf/adot-collector-config.yaml',
                                            '--set=service.telemetry.logs.level=DEBUG',
                                        ],
                                        env: [{
                                                name: 'OTEL_RESOURCE_ATTRIBUTES',
                                                value: `ClusterName=${this.cluster.clusterName}`,
                                            }],
                                        resources: {
                                            limits: {
                                                cpu: 2,
                                                memory: '2Gi',
                                            },
                                            requests: {
                                                cpu: '200m',
                                                memory: '400Mi',
                                            },
                                        },
                                        volumeMounts: [{
                                                name: 'adot-collector-config-volume',
                                                mountPath: '/conf',
                                            }],
                                    }],
                                volumes: [{
                                        configMap: {
                                            name: 'adot-collector-config',
                                            items: [{
                                                    key: 'adot-collector-config',
                                                    path: 'adot-collector-config.yaml',
                                                }],
                                        },
                                        name: 'adot-collector-config-volume',
                                    }],
                            },
                        },
                    },
                },
            ],
        });
    }
}
exports.AdotCollector = AdotCollector;
_a = JSII_RTTI_SYMBOL_1;
AdotCollector[_a] = { fqn: "cdk-extensions.k8s_aws.AdotCollector", version: "0.0.38" };
/**
 * The default Kubernetes namespace where resources related to the ADOT
 * collector will be created if no overriding input is provided.
 */
AdotCollector.DEFAULT_NAMESPACE = 'fargate-container-insights';
//# sourceMappingURL=data:application/json;base64,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