import { Resource, ResourceProps } from 'aws-cdk-lib';
import { Connections, IConnectable, ISecurityGroup, SubnetSelection } from 'aws-cdk-lib/aws-ec2';
import { ICluster, KubernetesManifest } from 'aws-cdk-lib/aws-eks';
import { Construct } from 'constructs';
import { Domain, DomainDiscovery, IDnsResolvable } from '../route53/domain-aspect';
/**
 * Configuration for the Echoserver resource.
 */
export interface EchoserverProps extends ResourceProps {
    /**
       * The EKS Cluster where the service should be deployed.
       */
    readonly cluster: ICluster;
    /**
     * Determines the behavior of automatic DNS discovery and configuration.
     *
     * @default DomainDiscovery.PUBLIC
     */
    readonly domainDiscovery?: DomainDiscovery;
    /**
     * The subnets where the load balancer should be created.
     */
    readonly loadBalancerSubnets?: SubnetSelection;
    /**
     * The name of the Kubernetes service to be created.
     *
     * @default 'echoserver'
     */
    readonly name?: string;
    /**
     * The Kubernetes namespace where the service should be created.
     *
     * @default 'default'
     */
    readonly namespace?: string;
    /**
     * The port which netcat should listen on.
     *
     * @default 80
     */
    readonly port?: number;
    /**
     * The number of replicas that should exist.
     *
     * @default 1
     */
    readonly replicas?: number;
    /**
     * The Security groups which should be applied to the service.
     */
    readonly securityGroups?: ISecurityGroup[];
    /**
     * A subdomain that should be prefixed to the beginning of all registered
     * domains.
     */
    readonly subdomain?: string;
    /**
     * The Docker tag specifying the version of echoserver to use.
     *
     * @see [Google echoserver image repository](https://console.cloud.google.com/gcr/images/google-containers/GLOBAL/echoserver)
     */
    readonly tag?: string;
}
/**
 * Creates a simple Kubernetes test service using the Google echoserver test
 * image.
 *
 * The server listens for incoming web requests and echos the details of the
 * request back to the user. Each request results in output being written to
 * the Docker log providing a convenient way to test logging setup.
 *
 * @see [Google echoserver image repository](https://console.cloud.google.com/gcr/images/google-containers/GLOBAL/echoserver)
 */
export declare class Echoserver extends Resource implements IConnectable, IDnsResolvable {
    /**
     * The default setting controlling how automatic DNS configuration should
     * behave if none is provided as input.
     */
    static readonly DEFAULT_DOMAIN_DISCOVERY: DomainDiscovery;
    /**
     * Default subnet selection that will be used if none is provided as input.
     */
    static readonly DEFAULT_LOAD_BALANCER_SUBNETS: SubnetSelection;
    /**
     * Default name of the Kubernetes service that will be created if none is
     * provided as input.
     */
    static readonly DEFAULT_NAME: string;
    /**
     * Default Kubernetes namespace where the service will be created if none is
     * provided as input.
     */
    static readonly DEFAULT_NAMESPACE: string;
    /**
     * Default port where the service will be accessible if none is provided as
     * input.
     */
    static readonly DEFAULT_PORT: number;
    /**
     * Default number of replicas that should be running is none is provided as
     * input.
     */
    static readonly DEFAULT_REPLICAS: number;
    /**
     * The Docker repository where the echoserver image will be pulled from.
     */
    static readonly DEFAULT_REPOSITORY: string;
    /**
     * The default Docker tag of the image to use if none is provided as input.
     */
    static readonly DEFAULT_TAG: string;
    /**
     * Internal collection of domain objects that should be used for configuring
     * DNS resolution.
     *
     * @group Internal
     */
    private readonly _domains;
    /**
     * The EKS Cluster where the service should be deployed.
     *
     * @group Inputs
     */
    readonly cluster: ICluster;
    /**
     * The subnets where the load balancer should be created..
     *
     * @group Inputs
     */
    readonly loadBalancerSubnets: SubnetSelection;
    /**
     * The name of the Kubernetes service to be created.
     *
     * @group Inputs
     */
    readonly name: string;
    /**
     * The Kubernetes namespace where the service should be created.
     *
     * @group Inputs
     */
    readonly namespace: string;
    /**
     * The port which netcat should listen on.
     *
     * @group Inputs
     */
    readonly port: number;
    /**
     * The number of replicas that should exist.
     *
     * @group Inputs
     */
    readonly replicas: number;
    /**
     * A subdomain that should be prefixed to the beginning of all registered
     * domains.
     */
    readonly subdomain?: string;
    /**
     * The Docker tag specifying the version of echoserver to use.
     *
     * @see [Google echoserver image repository](https://console.cloud.google.com/gcr/images/google-containers/GLOBAL/echoserver)
     *
     * @group Inputs
     */
    readonly tag: string;
    /**
     * The Kubernetes manifest that creates the ConfigMap that Fargate uses to
     * configure logging.
     *
     * @group Resources
     */
    readonly manifest: KubernetesManifest;
    /**
     * Access for network connections.
     *
     * @group IConnectable
     */
    readonly connections: Connections;
    /**
     * Determines the behavior of automatic DNS discovery and configuration.
     *
     * @group IDnsResolvable
     */
    readonly domainDiscovery: DomainDiscovery;
    /**
     * Creates a new instance of the Echoserver class.
     *
     * @param scope A CDK Construct that will serve as this resource's parent
     * in the construct tree.
     * @param id A name to be associated with the resource and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param props Arguments related to the configuration of the resource.
     */
    constructor(scope: Construct, id: string, props: EchoserverProps);
    registerDomain(domain: Domain): void;
}
