"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Echoserver = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const domain_aspect_1 = require("../route53/domain-aspect");
/**
 * Creates a simple Kubernetes test service using the Google echoserver test
 * image.
 *
 * The server listens for incoming web requests and echos the details of the
 * request back to the user. Each request results in output being written to
 * the Docker log providing a convenient way to test logging setup.
 *
 * @see [Google echoserver image repository](https://console.cloud.google.com/gcr/images/google-containers/GLOBAL/echoserver)
 */
class Echoserver extends aws_cdk_lib_1.Resource {
    /**
     * Creates a new instance of the Echoserver class.
     *
     * @param scope A CDK Construct that will serve as this resource's parent
     * in the construct tree.
     * @param id A name to be associated with the resource and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param props Arguments related to the configuration of the resource.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
        });
        this._domains = [];
        this.cluster = props.cluster;
        this.domainDiscovery = props.domainDiscovery ?? Echoserver.DEFAULT_DOMAIN_DISCOVERY;
        this.loadBalancerSubnets = props.loadBalancerSubnets ?? Echoserver.DEFAULT_LOAD_BALANCER_SUBNETS;
        this.name = props.name ?? Echoserver.DEFAULT_NAME;
        this.namespace = props.namespace ?? Echoserver.DEFAULT_NAMESPACE;
        this.port = props.port ?? Echoserver.DEFAULT_PORT;
        this.replicas = props.replicas ?? Echoserver.DEFAULT_REPLICAS;
        this.subdomain = props.subdomain !== domain_aspect_1.Domains.ROOT ? (props.subdomain ?? this.name) : undefined;
        this.tag = props.tag ?? Echoserver.DEFAULT_TAG;
        this.connections = new aws_ec2_1.Connections({
            defaultPort: aws_ec2_1.Port.tcp(this.port),
            securityGroups: props.securityGroups ?? [
                new aws_ec2_1.SecurityGroup(this, 'load-balancer-security-group', {
                    vpc: this.cluster.vpc,
                }),
            ],
        });
        const serviceSecurityGroup = new aws_ec2_1.SecurityGroup(this, 'service-security-group', {
            description: `Security group used by pods belonging to the ${this.name} service.`,
            vpc: this.cluster.vpc,
        });
        serviceSecurityGroup.connections.allowFrom(this.cluster.clusterSecurityGroup, aws_ec2_1.Port.allTraffic(), 'Internal Kubernetes communication.');
        serviceSecurityGroup.connections.allowFrom(this.connections, aws_ec2_1.Port.tcp(8080), 'ALB inbound traffic.');
        this.manifest = new aws_eks_1.KubernetesManifest(this, 'Resource', {
            cluster: this.cluster,
            manifest: [
                {
                    apiVersion: 'vpcresources.k8s.aws/v1beta1',
                    kind: 'SecurityGroupPolicy',
                    metadata: {
                        name: this.name,
                        namespace: this.namespace,
                    },
                    spec: {
                        podSelector: {
                            matchLabels: {
                                app: this.name,
                            },
                        },
                        securityGroups: {
                            groupIds: [
                                serviceSecurityGroup.securityGroupId,
                            ],
                        },
                    },
                },
                {
                    apiVersion: 'apps/v1',
                    kind: 'Deployment',
                    metadata: {
                        labels: {
                            app: this.name,
                        },
                        name: this.name,
                        namespace: this.namespace,
                    },
                    spec: {
                        replicas: this.replicas,
                        selector: {
                            matchLabels: {
                                app: this.name,
                            },
                        },
                        template: {
                            metadata: {
                                labels: {
                                    app: this.name,
                                },
                            },
                            spec: {
                                containers: [
                                    {
                                        image: `${Echoserver.DEFAULT_REPOSITORY}:${this.tag}`,
                                        imagePullPolicy: 'Always',
                                        name: 'echoserver',
                                        ports: [
                                            {
                                                containerPort: 8080,
                                            },
                                        ],
                                    },
                                ],
                            },
                        },
                    },
                },
                {
                    apiVersion: 'v1',
                    kind: 'Service',
                    metadata: {
                        annotations: {
                            'alb.ingress.kubernetes.io/target-type': 'ip',
                        },
                        name: this.name,
                        namespace: this.namespace,
                    },
                    spec: {
                        ports: [
                            {
                                port: this.port,
                                protocol: 'TCP',
                                targetPort: 8080,
                            },
                        ],
                        selector: {
                            app: this.name,
                        },
                        type: 'NodePort',
                    },
                },
                {
                    apiVersion: 'networking.k8s.io/v1',
                    kind: 'Ingress',
                    metadata: {
                        annotations: {
                            'kubernetes.io/ingress.class': 'alb',
                            'alb.ingress.kubernetes.io/listen-ports': this.stack.toJsonString([{
                                    HTTP: this.port,
                                }]),
                            'alb.ingress.kubernetes.io/scheme': 'internet-facing',
                            'alb.ingress.kubernetes.io/security-groups': aws_cdk_lib_1.Lazy.string({
                                produce: () => {
                                    return this.connections.securityGroups.map((x) => {
                                        return x.securityGroupId;
                                    }).join(', ');
                                },
                            }),
                            'alb.ingress.kubernetes.io/subnets': this.cluster.vpc.selectSubnets(this.loadBalancerSubnets).subnetIds.join(', '),
                            'external-dns.alpha.kubernetes.io/hostname': aws_cdk_lib_1.Lazy.string({
                                produce: () => {
                                    if (this._domains.length === 0) {
                                        return undefined;
                                    }
                                    return this._domains.map((x) => {
                                        return this.subdomain ? `${this.subdomain}.${x.fqdn}` : x.fqdn;
                                    }).join(',');
                                },
                            }),
                        },
                        name: this.name,
                        namespace: this.namespace,
                    },
                    spec: {
                        rules: [
                            {
                                http: {
                                    paths: [
                                        {
                                            backend: {
                                                service: {
                                                    name: this.name,
                                                    port: {
                                                        number: this.port,
                                                    },
                                                },
                                            },
                                            path: '/',
                                            pathType: 'Prefix',
                                        },
                                    ],
                                },
                            },
                        ],
                    },
                },
            ],
        });
    }
    registerDomain(domain) {
        this._domains.push(domain);
    }
}
exports.Echoserver = Echoserver;
_a = JSII_RTTI_SYMBOL_1;
Echoserver[_a] = { fqn: "cdk-extensions.k8s_aws.Echoserver", version: "0.0.38" };
/**
 * The default setting controlling how automatic DNS configuration should
 * behave if none is provided as input.
 */
Echoserver.DEFAULT_DOMAIN_DISCOVERY = domain_aspect_1.DomainDiscovery.PUBLIC;
/**
 * Default subnet selection that will be used if none is provided as input.
 */
Echoserver.DEFAULT_LOAD_BALANCER_SUBNETS = {
    onePerAz: true,
    subnetType: aws_ec2_1.SubnetType.PUBLIC,
};
/**
 * Default name of the Kubernetes service that will be created if none is
 * provided as input.
 */
Echoserver.DEFAULT_NAME = 'echoserver';
/**
 * Default Kubernetes namespace where the service will be created if none is
 * provided as input.
 */
Echoserver.DEFAULT_NAMESPACE = 'default';
/**
 * Default port where the service will be accessible if none is provided as
 * input.
 */
Echoserver.DEFAULT_PORT = 80;
/**
 * Default number of replicas that should be running is none is provided as
 * input.
 */
Echoserver.DEFAULT_REPLICAS = 1;
/**
 * The Docker repository where the echoserver image will be pulled from.
 */
Echoserver.DEFAULT_REPOSITORY = 'gcr.io/google_containers/echoserver';
/**
 * The default Docker tag of the image to use if none is provided as input.
 */
Echoserver.DEFAULT_TAG = '1.10';
//# sourceMappingURL=data:application/json;base64,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