import { Resource, ResourceProps } from 'aws-cdk-lib';
import { HelmChart, ICluster, ServiceAccount } from 'aws-cdk-lib/aws-eks';
import { Construct } from 'constructs';
import { IExternalDnsRegistry } from '.';
/**
 * The format external dns should use to output logs.
 */
export declare enum ExternalDnsLogFormat {
    /**
     * Output logs will be written as JSON objects.
     */
    JSON = "json",
    /**
     * Output logs will be written in plain text.
     */
    TEXT = "text"
}
/**
 * Verbosity of the logs generated by the external-dns service.
 */
export declare enum ExternalDnsLogLevel {
    /**
     * Set log level to 'panic'
     */
    PANIC = "panic",
    /**
     * Set log level to 'debug'
     */
    DEBUG = "debug",
    /**
     * Set log level to 'info'
     */
    INFO = "info",
    /**
     * Set log level to 'warning'
     */
    WARNING = "warning",
    /**
     * Set log level to 'error'
     */
    ERROR = "error",
    /**
     * Set log level to 'fatal'
     */
    FATAL = "fatal",
    /**
     * Set log level to 'trace'
     */
    TRACE = "trace"
}
/**
 * Controls the operations ExternalDNS will perform on the records it manages.
 */
export declare enum ExternalDnsSyncPolicy {
    /**
     * Full sync mode. Records will be created, updated, and deleted based on the
     * statis of their backing resources on the Kubernetes cluster.
     */
    SYNC = "sync",
    /**
     * Only allow create and update operations. Records will have their values
     * set based on the status of their backing Kubernetes resources, however if
     * those resources are removed the DNS records will be retained, set to their
     * last configured value.
     */
    UPSERT_ONLY = "upsert-only"
}
/**
 * Specifies a tag that can be used to restrict which Hosted Zone external-dns
 * will have access to.
 */
export interface ExternalDnsZoneTag {
    /**
     * The name of the tag to filter on.
     */
    readonly key: string;
    /**
     * The value of the tag to filter on.
     */
    readonly value: string;
}
/**
 * Controls the types of Hosted Zones external DNS will create records for.
 */
export declare enum ExternalDnsZoneType {
    /**
     * Create DNS records for both public and private hosted zones.
     */
    ALL = "ALL",
    /**
     * Only create DNS records for private hosted zones.
     */
    PRIVATE = "private",
    /**
     * Only create DNS records for public hosted zones.
     */
    PUBLIC = "public"
}
/**
 * Optional configuration for the Route53Dns resource.
 */
export interface Route53DnsOptions {
    /**
     * Maximum number of retries for AWS API calls before giving up.
     *
     * @default 3
     */
    readonly apiRetries?: number;
    /**
     * Set the maximum number of changes that will be applied in each batch.
     *
     * @default 1000
     */
    readonly batchChangeSize?: number;
    /**
     * Limits possible target zones by domain suffixes
     */
    readonly domainFilter?: string[];
    /**
     * Sets a flag determining whether the health of the backend service should
     * be evaluated when determining DNS routing.
     */
    readonly evaluateTargetHealth?: boolean;
    /**
     * Sets the output format external dns will use when generating logs.
     *
     * @default {@link ExternalDnsLogLevel.JSON}
     */
    readonly logFormat?: ExternalDnsLogFormat;
    /**
     * Controls the verbosity of logs generated using the external-dns service.
     *
     * @default {@link ExternalDnsLogLevel.INFO}
     */
    readonly logLevel?: ExternalDnsLogLevel;
    /**
     * The Kubernetes namespace where the service should be deployed.
     *
     * @default 'kube-system'
     */
    readonly namespace?: string;
    /**
     * When true, alias records will be avoided and CNAME records will be used
     * instead.
     *
     * @default false
     */
    readonly preferCname?: boolean;
    /**
     * Registry specifying how ExternalDNS should track record ownership.
     *
     * Without a registry to track record ownership, External has no way to know
     * which records it owns and manages and which are owned and managed by a
     * different service.
     *
     * This can cause conflicts if there are multiple instances of External DNS
     * running or if there are other services managing DNS records in similar
     * zones as the different services could try to make conflicting changes due
     * to lacking a shared state.
     *
     * @default A TXT registry configured with defaults.
     */
    readonly recordOwnershipRegistry?: IExternalDnsRegistry;
    /**
     * Override the default region external-dns uses when calling AWS API's.
     */
    readonly region?: string;
    /**
     * Desired number of ExternalDNS replicas.
     *
     * @default 1
     */
    readonly replicaCount?: number;
    /**
     * Controls the operations ExternalDNS will perform on the records it manages.
     *
     * @default {@link ExternalDnsSyncPolicy.SYNC}
     */
    readonly syncPolicy?: ExternalDnsSyncPolicy;
    /**
     * A set of tags that can be used to restrict which hosted zones external
     * DNS will make changes to.
     */
    readonly zoneTags?: ExternalDnsZoneTag[];
    /**
     * Controls the types of hosted zones external-dns will create records for.
     *
     * @default ExternalDnsZoneType.ALL
     */
    readonly zoneType?: ExternalDnsZoneType;
}
/**
 * Full configuration for the Route53Dns resource.
 */
export interface Route53DnsProps extends ResourceProps, Route53DnsOptions {
    /**
     * The EKS cluster where external-dns should be deployed.
     */
    readonly cluster: ICluster;
}
/**
 * External DNS is a Kubernetes service that make Kubernetes resources
 * dicoverable via public DNS servers. It retrieves a list of resources
 * (Services, Ingresses, etc.) from the Kubernetes API to determine a desired
 * list of DNS records and configures DNS providers accordingly.
 *
 * The version provided here specifically targets Amazon's Route 53 service and
 * all options provded are for configuring Route 53. After being installed
 * external-dns will create and manage Route 53 DNS records automatically to
 * allow easy network access to your pods and services.
 *
 * @see [Kubernetes SIGs](https://github.com/kubernetes-sigs/external-dns)
 */
export declare class Route53Dns extends Resource {
    /**
     * The name of the external-dns Helm chart.
     */
    static readonly CHART_NAME: string;
    /**
     * The Helm repository providing the chart to be used for installing the
     * external-dns service.
     */
    static readonly CHART_REPOSITORY: string;
    /**
     * The default Kubernetes namespace where external-dns will be installed if
     * an alternative isn't given as input.
     */
    static readonly DEFAULT_NAMESPACE: string;
    /**
     * Internal collection of domain suffixes that control which hosted zones
     * external-dns is allowed to make changes for.
     *
     * @group Internal
     */
    private readonly _domainFilter;
    /**
     * Internal collection of AWS tags that control which hosted zones
     * external-dns is allowed to make changes for.
     *
     * @group Internal
     */
    private readonly _zoneTags;
    /**
     * Maximum number of retries for AWS API calls before giving up.
     *
     * @group Inputs
     */
    readonly apiRetries?: number;
    /**
     * Set the maximum number of changes that will be applied in each batch.
     *
     * @group Inputs
     */
    readonly batchChangeSize?: number;
    /**
     * The EKS cluster where external-dns should be deployed.
     *
     * @group Inputs
     */
    readonly cluster: ICluster;
    /**
     * Sets a flag determining whether the health of the backend service should
     * be evaluated when determining DNS routing.
     *
     * @group Inputs
     */
    readonly evaluateTargetHealth?: boolean;
    /**
     * Sets the output format external dns will use when generating logs.
     *
     * @group Inputs
     */
    readonly logFormat?: ExternalDnsLogFormat;
    /**
     * Controls the verbosity of logs generated using the external-dns service.
     *
     * @group Inputs
     */
    readonly logLevel?: ExternalDnsLogLevel;
    /**
     * The Kubernetes namespace where the service should be deployed.
     *
     * @group Inputs
     */
    readonly namespace?: string;
    /**
     * When true, alias records will be avoided and CNAME records will be used
     * instead.
     *
     * @group Inputs
     */
    readonly preferCname?: boolean;
    /**
     * Registry specifying how ExternalDNS should track record ownership.
     *
     * Without a registry to track record ownership, External has no way to know
     * which records it owns and manages and which are owned and managed by a
     * different service.
     *
     * This can cause conflicts if there are multiple instances of External DNS
     * running or if there are other services managing DNS records in similar
     * zones as the different services could try to make conflicting changes due
     * to lacking a shared state.
     *
     * @group Inputs
     */
    readonly recordOwnershipRegistry: IExternalDnsRegistry;
    /**
     * Override the default region external-dns uses when calling AWS API's.
     *
     * @group Inputs
     */
    readonly region: string;
    /**
     * Desired number of ExternalDNS replicas.
     *
     * @group Inputs
     */
    readonly replicaCount?: number;
    /**
     * Controls the operations ExternalDNS will perform on the records it manages.
     *
     * @group Inputs
     */
    readonly syncPolicy?: ExternalDnsSyncPolicy;
    /**
     * Controls the types of hosted zones external-dns will create records for.
     *
     * @group Inputs
     */
    readonly zoneType?: ExternalDnsZoneType;
    /**
     * The domain suffixes that control which hosted zones external-dns is
     * allowed to make changes for.
     *
     * @group Inputs
     */
    get domainFilter(): string[];
    /**
     * The AWS tags that control which hosted zones external-dns is allowed to
     * make changes for.
     *
     * @group Inputs
     */
    get zoneTags(): ExternalDnsZoneTag[];
    /**
     * The Helm chart that provides the installation of external-dns.
     *
     * @group Resources
     */
    readonly chart: HelmChart;
    /**
     * The Kubernetes service account that is linked with the IAM Role that
     * allows external-dns to make changes on your behalf.
     *
     * @group Resources
     */
    readonly serviceAccount: ServiceAccount;
    /**
     * Creates a new instance of the Route53Dns class.
     *
     * @param scope A CDK Construct that will serve as this resource's parent
     * in the construct tree.
     * @param id A name to be associated with the resource and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param props Arguments related to the configuration of the resource.
     */
    constructor(scope: Construct, id: string, props: Route53DnsProps);
    /**
     * Adds a domain to the domain filter list.
     *
     * The domain filter list acts as a whitelist for the domains/hosted zones
     * which external-dns will manage.
     *
     * When domains are added to the domain filter list, external-dns will only
     * create and manage records when their domain ends in with a domain that has
     * been approved.
     *
     * @param domain The domain to be added to the whitelist.
     * @returns The external-dns service object that the domain filter was added
     * for.
     */
    addDomainFilter(domain: string): Route53Dns;
    /**
     * Adds a zone tag filter to the external DNS service.
     *
     * When zone tags are provided only Routew 53 Hosted Zones that have matching
     * tags will be managed by external DNS.
     *
     * @param tag The tag that external-dns is allowed to manage.
     * @returns The external-dns service object that the zone tag was added for.
     */
    addZoneTag(tag: ExternalDnsZoneTag): Route53Dns;
}
