import { Duration, Resource, ResourceProps } from 'aws-cdk-lib';
import { ICluster, KubernetesManifest } from 'aws-cdk-lib/aws-eks';
import { Construct, IConstruct } from 'constructs';
import { ISecretStore } from '.';
/**
 * Options for fetching tags/labels from provider secrets.
 */
export declare enum MetadataPolicy {
    /**
     * Fetch tags/labels from provider secrets.
     */
    FETCH = "Fetch",
    /**
     * Do not fetch tags/labels from provider secrets.
     */
    NONE = "None"
}
/**
 * Options for how to synchronize a specific field in a secret being imported.
 */
export interface SecretFieldReference {
    /**
     * The name of the data key to be used for the field in the imported
     * Kubernetes secret.
     */
    readonly kubernetesKey: string;
    /**
     * Policy for fetching tags/labels from provider secrets.
     */
    readonly metadataPolicy?: MetadataPolicy;
    /**
     * The JSON key for the field in the secret being imported.
     */
    readonly remoteKey?: string;
}
/**
 * Configuration detailing how secrets are to be synchronized.
 */
export interface SecretReferenceConfiguration {
    /**
     * A mapping of fields and per field options to use when synchronizing a
     * secret from a provider.
     */
    readonly fields?: SecretFieldReference[];
    /**
     * The ID of the secret to be imported from the provider.
     */
    readonly remoteRef: string;
}
/**
 * Represents a resource the can be synchronized into a Kubernetes secret.
 */
export interface ISecretReference {
    /**
     * Gets the configuration details for the resource being sychronized in a
     * form that can be universally used to create the synchronization
     * configuration.
     *
     * @param scope The scope of the construct that will be configuring the
     * synchronization configuration.
     */
    bind(scope: IConstruct): SecretReferenceConfiguration;
}
/**
 * Configuration for the ExternalSecret resource.
 */
export interface ExternalSecretProps extends ResourceProps {
    /**
     * The EKS cluster where the secret should be created.
     */
    readonly cluster: ICluster;
    /**
     * The name to use for the Kubernetes secret resource when it is synchronized
     * into the cluster.
     */
    readonly name?: string;
    /**
     * The name where the synchronized secret should be created.
     */
    readonly namespace?: string;
    /**
     * The frequency at which synchronization should occur.
     */
    readonly refreshInterval?: Duration;
    /**
     * The secrets to synchronize into this Kubernetes secret.
     *
     * If multiple secrets are provided their fields will be merged.
     */
    readonly secrets?: ISecretReference[];
    /**
     * The Kubernetes secret store resource that provides details and permissions
     * to use for importing secrets from the provider.
     */
    readonly secretStore: ISecretStore;
}
/**
 * Represents a Kubernetes secret that is being synchronized from an external
 * provider.
 *
 * On a technical level, provides the configuration for how the external
 * secrets operator service should manage the synchronization of the Kubernetes
 * secret.
 */
export declare class ExternalSecret extends Resource {
    /**
     * The internal collection of referenced provider secrets that are referenced
     * in the Kubernetes secret.
     */
    private readonly _secrets;
    /**
     * The EKS cluster where the secret should be created.
     *
     * @group Inputs
     */
    readonly cluster: ICluster;
    /**
     * The name to use for the Kubernetes secret resource when it is synchronized
     * into the cluster.
     *
     * @group Inputs
     */
    readonly name: string;
    /**
     * The name where the synchronized secret should be created.
     *
     * @group Inputs
     */
    readonly namespace?: string;
    /**
     * The frequency at which synchronization should occur.
     *
     * @group Inputs
     */
    readonly refreshInterval?: Duration;
    /**
     * The Kubernetes secret store resource that provides details and permissions
     * to use for importing secrets from the provider.
     *
     * @group Inputs
     */
    readonly secretStore: ISecretStore;
    /**
     * The collection of referenced provider secrets that are referenced in the
     * Kubernetes secret.
     *
     * @group Inputs
     */
    get secrets(): ISecretReference[];
    /**
     * The Kubernetes manifest defining the configuration of how to synchronize
     * the Kubernetes secret from the provider secrets.
     *
     * @group Resources
     */
    readonly manifest: KubernetesManifest;
    /**
     * The name of the Kubernetes secret.
     */
    readonly secretName: string;
    /**
     * Creates a new instance of the ExternalSecret class.
     *
     * @param scope A CDK Construct that will serve as this resource's parent
     * in the construct tree.
     * @param id A name to be associated with the resource and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param props Arguments related to the configuration of the resource.
     */
    constructor(scope: Construct, id: string, props: ExternalSecretProps);
    /**
     * Adds a provider secret reference to the synchronized Kubernetes secret.
     *
     * For external secrets that reference multiple provider secrets the keys of
     * all provider secrets will be merged into the single Kubernetes secret.
     *
     * @param secret The provider secret to reference.
     * @returns The external secret resoiurce where the reference was added.
     */
    addSecret(secret: ISecretReference): ExternalSecret;
}
