"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExternalSecret = exports.MetadataPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
/**
 * Options for fetching tags/labels from provider secrets.
 */
var MetadataPolicy;
(function (MetadataPolicy) {
    /**
     * Fetch tags/labels from provider secrets.
     */
    MetadataPolicy["FETCH"] = "Fetch";
    /**
     * Do not fetch tags/labels from provider secrets.
     */
    MetadataPolicy["NONE"] = "None";
})(MetadataPolicy = exports.MetadataPolicy || (exports.MetadataPolicy = {}));
/**
 * Represents a Kubernetes secret that is being synchronized from an external
 * provider.
 *
 * On a technical level, provides the configuration for how the external
 * secrets operator service should manage the synchronization of the Kubernetes
 * secret.
 */
class ExternalSecret extends aws_cdk_lib_1.Resource {
    /**
     * Creates a new instance of the ExternalSecret class.
     *
     * @param scope A CDK Construct that will serve as this resource's parent
     * in the construct tree.
     * @param id A name to be associated with the resource and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param props Arguments related to the configuration of the resource.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
        });
        /**
         * The internal collection of referenced provider secrets that are referenced
         * in the Kubernetes secret.
         */
        this._secrets = [];
        this.cluster = props.cluster;
        this.name = props.name ?? `es${aws_cdk_lib_1.Names.uniqueId(this).slice(-61).toLowerCase()}`;
        this.namespace = props.namespace;
        this.refreshInterval = props.refreshInterval ?? aws_cdk_lib_1.Duration.minutes(1);
        this.secretStore = props.secretStore;
        this.manifest = new aws_eks_1.KubernetesManifest(this, 'Resource', {
            cluster: this.cluster,
            manifest: [
                {
                    apiVersion: 'external-secrets.io/v1beta1',
                    kind: 'ExternalSecret',
                    metadata: {
                        name: this.name,
                        namespace: this.namespace,
                    },
                    spec: {
                        refreshInterval: `${this.refreshInterval.toMinutes()}m`,
                        secretStoreRef: {
                            name: this.secretStore.secretStoreName,
                            kind: 'SecretStore',
                        },
                        target: {
                            name: this.name,
                            creationPolicy: 'Owner',
                        },
                        data: aws_cdk_lib_1.Lazy.uncachedAny({
                            produce: () => {
                                return this._secrets.reduce((prev, cur) => {
                                    const config = cur.bind(this);
                                    config.fields?.forEach((x) => {
                                        prev.push({
                                            secretKey: x.kubernetesKey,
                                            remoteRef: {
                                                key: config.remoteRef,
                                                metadataPolicy: x.metadataPolicy,
                                                property: x.remoteKey,
                                            },
                                        });
                                    });
                                    return prev;
                                }, []);
                            },
                        }, {
                            omitEmptyArray: true,
                        }),
                        dataFrom: aws_cdk_lib_1.Lazy.uncachedAny({
                            produce: () => {
                                return this._secrets.reduce((prev, cur) => {
                                    const config = cur.bind(this);
                                    if ((config.fields?.length ?? 0) === 0) {
                                        prev.push({
                                            extract: {
                                                key: config.remoteRef,
                                            },
                                        });
                                    }
                                    return prev;
                                }, []);
                            },
                        }, {
                            omitEmptyArray: true,
                        }),
                    },
                },
            ],
        });
        this.manifest.node.addDependency(this.secretStore);
        this.secretName = this.name;
        props.secrets?.map((x) => {
            this.addSecret(x);
        });
    }
    /**
     * The collection of referenced provider secrets that are referenced in the
     * Kubernetes secret.
     *
     * @group Inputs
     */
    get secrets() {
        return [...this._secrets];
    }
    /**
     * Adds a provider secret reference to the synchronized Kubernetes secret.
     *
     * For external secrets that reference multiple provider secrets the keys of
     * all provider secrets will be merged into the single Kubernetes secret.
     *
     * @param secret The provider secret to reference.
     * @returns The external secret resoiurce where the reference was added.
     */
    addSecret(secret) {
        this._secrets.push(secret);
        return this;
    }
}
exports.ExternalSecret = ExternalSecret;
_a = JSII_RTTI_SYMBOL_1;
ExternalSecret[_a] = { fqn: "cdk-extensions.k8s_aws.ExternalSecret", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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