"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExternalSecretsOperator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const secrets_manager_secret_store_1 = require("./secrets-manager-secret-store");
const ssm_parameter_secret_store_1 = require("./ssm-parameter-secret-store");
/**
 * External Secrets Operator is a Kubernetes operator that integrates external
 * secret management systems like AWS Secrets Manager, HashiCorp Vault, Google
 * Secrets Manager, Azure Key Vault and many more. The operator reads
 * information from external APIs and automatically injects the values into a
 * Kubernetes Secret.
 *
 * @see [External Secrets Website](https://external-secrets.io/)
 */
class ExternalSecretsOperator extends aws_cdk_lib_1.Resource {
    /**
     * Creates a new instance of the ExternalSecretsOperator class.
     *
     * @param scope A CDK Construct that will serve as this resource's parent
     * in the construct tree.
     * @param id A name to be associated with the resource and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param props Arguments related to the configuration of the resource.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
        });
        this.cluster = props.cluster;
        this.createNamespace = props.createNamespace ?? true;
        this.namespace = props.namespace ?? ExternalSecretsOperator.DEFAULT_NAMESPACE;
        this.helmChart = new aws_eks_1.HelmChart(this, 'helm-chart', {
            cluster: this.cluster,
            chart: ExternalSecretsOperator.CHART_NAME,
            createNamespace: this.createNamespace,
            namespace: this.namespace,
            release: 'external-secrets-operator',
            repository: ExternalSecretsOperator.CHART_REPOSITORY,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            values: {
                installCRDs: true,
                webhook: {
                    port: '9443',
                },
            },
            wait: true,
        });
    }
    /**
     * Registers a Secrets Manager secret with the external secrets operator,
     * enabling syncing from the Secrets Manager secret into Kubernetes.
     *
     * @param id A name to be associated with the resource and used in resource
     * naming. Must be unique within for each secrets manager secret within a
     * Kubernetes namespace.
     * @param secret The Secrets Manager secret to enable syncing for.
     * @param options Options for configuring syncing of the Secrets Manager
     * secret.
     * @returns The external secret object that was created.
     */
    registerSecretsManagerSecret(id, secret, options = {}) {
        const namespace = options?.namespace ?? 'default';
        const storeId = `store::${namespace}::secrets-manager`;
        const store = this.node.tryFindChild(storeId) ?? new secrets_manager_secret_store_1.SecretsManagerSecretStore(this, storeId, {
            cluster: this.cluster,
            namespace: options?.namespace,
        });
        store.node.addDependency(this.helmChart);
        return store.addSecret(id, secret, options);
    }
    /**
     * Registers a Systems Manager parameter with the external secrets operator,
     * enabling syncing from the Systems Manager parameter into Kubernetes.
     *
     * @param id A name to be associated with the resource and used in resource
     * naming. Must be unique within for each Systems Manager parameter within a
     * Kubernetes namespace.
     * @param parameter The Systems Manager parameter to enable syncing for.
     * @param options Options for configuring syncing of the Systems Manager
     * parameter.
     * @returns The external secret object that was created.
     */
    registerSsmParameterSecret(id, parameter, options = {}) {
        const namespace = options.namespace ?? 'default';
        const storeId = `store::${namespace}::ssm`;
        const store = this.node.tryFindChild(storeId) ?? new ssm_parameter_secret_store_1.SsmParameterSecretStore(this, storeId, {
            cluster: this.cluster,
            namespace: options.namespace,
        });
        store.node.addDependency(this.helmChart);
        return store.addSecret(id, parameter, options);
    }
}
exports.ExternalSecretsOperator = ExternalSecretsOperator;
_a = JSII_RTTI_SYMBOL_1;
ExternalSecretsOperator[_a] = { fqn: "cdk-extensions.k8s_aws.ExternalSecretsOperator", version: "0.0.38" };
/**
 * The name of the Helm chart to install from the Helm repository.
 */
ExternalSecretsOperator.CHART_NAME = 'external-secrets';
/**
 * The URL of the Helm repository that hostys the Helm charts used to install
 * the externalk secrets operator service.
 */
ExternalSecretsOperator.CHART_REPOSITORY = 'https://charts.external-secrets.io';
/**
 * The default Kubernetes namespace where the external secrets operator
 * service should be installed and configured if no overriding input is
 * provided.
 */
ExternalSecretsOperator.DEFAULT_NAMESPACE = 'external-secrets';
//# sourceMappingURL=data:application/json;base64,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