import { Resource, ResourceProps } from 'aws-cdk-lib';
import { FargateProfile, ICluster, KubernetesManifest } from 'aws-cdk-lib/aws-eks';
import { ILogGroup } from 'aws-cdk-lib/aws-logs';
import { Construct } from 'constructs';
import { IFluentBitFilterPlugin, IFluentBitOutputPlugin, IFluentBitParserPlugin } from '.';
/**
 * Optional configuration for the FargateLogger resource.
 */
export interface FargateLoggerOptions {
    /**
       * A default list of Fargate profiles that should have permissions
       * configured. Alternatively profiles can be added at any time by calling
       * `addProfile`.
       */
    readonly fargateProfiles?: FargateProfile[];
    /**
     * The filters that should be applied to logs being processed.
     */
    readonly filters?: IFluentBitFilterPlugin[];
    /**
       * The CloudWatch log group where Farget container logs will be sent.
       */
    readonly logGroup?: ILogGroup;
    /**
     * The output destinations where logs should be written.
     */
    readonly outputs?: IFluentBitOutputPlugin[];
    /**
     * The parsers to be used when reading log files.
     */
    readonly parsers?: IFluentBitParserPlugin[];
}
/**
 * Required configuration for the Fargate logger resource.
 */
export interface FargateLoggerProps extends FargateLoggerOptions, ResourceProps {
    /**
     * The EKS Cluster to configure Fargate logging for.
     */
    readonly cluster: ICluster;
}
/**
 * Creates a ConfigMap that configures logging for containers running in EKS
 * on Fargate.
 */
export declare class FargateLogger extends Resource {
    /**
     * Internal collection of Fargate Profiles that will be using this
     * configuration for setting up container logging.
     *
     * @group Internal
     */
    private readonly _fargateProfiles;
    /**
     * Internal collection of Fluent Bit filter plugins being configured for
     * logging.
     *
     * @group Internal
     */
    private readonly _filters;
    /**
     * Internal collection for Fluent Bit output plugins being configured for
     * logging.
     *
     * @group Internal
     */
    private readonly _outputs;
    /**
     * Internal collection for Fluent Bit parser plugins being configured for
     * logging.
     *
     * @group Internal
     */
    private readonly _parsers;
    /**
     * The EKS cluster where Fargate logging is being configured.
     *
     * @group Inputs
     */
    readonly cluster: ICluster;
    /**
     * Collection of Fluent Bit filter plugins being configured for logging.
     *
     * @group Inputs
     */
    get filters(): IFluentBitFilterPlugin[];
    /**
     * Collection of Fluent Bit output plugins being configured for logging.
     *
     * @group Inputs
     */
    get outputs(): IFluentBitOutputPlugin[];
    /**
     * Collection of Fluent Bit parser plugins being configured for logging.
     *
     * @group Inputs
     */
    get parsers(): IFluentBitParserPlugin[];
    /**
     * The Kubernetes manifest that creates the ConfigMap that Fargate uses to
     * configure logging.
     *
     * @group Resources
     */
    readonly manifest: KubernetesManifest;
    /**
     * Creates a new instance of the FargateLogger class.
     *
     * @param scope A CDK Construct that will serve as this resource's parent
     * in the construct tree.
     * @param id A name to be associated with the resource and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param props Arguments related to the configuration of the resource.
     */
    constructor(scope: Construct, id: string, props: FargateLoggerProps);
    addFargateProfile(profile: FargateProfile): FargateLogger;
    addFilter(filter: IFluentBitFilterPlugin): FargateLogger;
    addOutput(output: IFluentBitOutputPlugin): FargateLogger;
    addParser(parser: IFluentBitParserPlugin): FargateLogger;
}
