"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateLogger = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const _1 = require(".");
/**
 * Creates a ConfigMap that configures logging for containers running in EKS
 * on Fargate.
 */
class FargateLogger extends aws_cdk_lib_1.Resource {
    /**
     * Creates a new instance of the FargateLogger class.
     *
     * @param scope A CDK Construct that will serve as this resource's parent
     * in the construct tree.
     * @param id A name to be associated with the resource and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param props Arguments related to the configuration of the resource.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
        });
        this._fargateProfiles = [];
        this._filters = [];
        this._outputs = [];
        this._parsers = [];
        this.cluster = props.cluster;
        this.manifest = new aws_eks_1.KubernetesManifest(this, 'Resource', {
            cluster: this.cluster,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: {
                        name: 'aws-observability',
                        labels: {
                            'aws-observability': 'enabled',
                        },
                    },
                },
                {
                    apiVersion: 'v1',
                    kind: 'ConfigMap',
                    metadata: {
                        name: 'aws-logging',
                        namespace: 'aws-observability',
                    },
                    data: {
                        'filters.conf': aws_cdk_lib_1.Lazy.string({
                            produce: () => {
                                if (!this._filters.length) {
                                    this.addFilter(new _1.FluentBitKubernetesFilter());
                                }
                                const result = this._filters.map((filter) => {
                                    const boundConfig = filter.bind(this);
                                    boundConfig.parsers?.forEach((parser) => {
                                        this.addParser(parser);
                                    });
                                    boundConfig.permissions?.forEach((statement) => {
                                        this._fargateProfiles.forEach((profile) => {
                                            profile.podExecutionRole.addToPrincipalPolicy(statement);
                                        });
                                    });
                                    return boundConfig.configFile;
                                });
                                return result.length > 0 ? result.join('\n') : undefined;
                            },
                        }),
                        'output.conf': aws_cdk_lib_1.Lazy.string({
                            produce: () => {
                                if (!this._outputs.length) {
                                    this.addOutput(new _1.FluentBitCloudWatchLogsOutput());
                                }
                                return this._outputs.map((x) => {
                                    const boundConfig = x.bind(this);
                                    boundConfig.parsers?.forEach((parser) => {
                                        this.addParser(parser);
                                    });
                                    boundConfig.permissions?.forEach((statement) => {
                                        this._fargateProfiles.forEach((profile) => {
                                            profile.podExecutionRole.addToPrincipalPolicy(statement);
                                        });
                                    });
                                    return boundConfig.configFile;
                                }).join('\n');
                            },
                        }),
                        'parsers.conf': aws_cdk_lib_1.Lazy.string({
                            produce: () => {
                                const result = [];
                                let batch = this._parsers;
                                let queue = [];
                                while (batch.length > 0) {
                                    batch.forEach((x) => {
                                        const boundConfig = x.bind(this);
                                        queue.concat(boundConfig.parsers ?? []);
                                        boundConfig.permissions?.forEach((statement) => {
                                            this._fargateProfiles.forEach((profile) => {
                                                profile.podExecutionRole.addToPrincipalPolicy(statement);
                                            });
                                        });
                                        result.push(boundConfig.configFile);
                                    });
                                    batch = queue;
                                    queue = [];
                                }
                                return result.length > 0 ? result.join('\n') : undefined;
                            },
                        }),
                    },
                },
            ],
        });
        this.node.addValidation({
            validate: () => {
                let hasKubernetesFilter = false;
                const result = [];
                this.filters.forEach((x) => {
                    if (x.name === _1.FluentBitKubernetesFilter.PLUGIN_NAME) {
                        hasKubernetesFilter = true;
                    }
                    else if (hasKubernetesFilter && x.name === _1.FluentBitRewriteTagFilter.PLUGIN_NAME) {
                        result.push([
                            'Cannot have rewrite_tag filters applied after kubernetes',
                            'filters in Fluent Bit filter configuration. See:',
                            'https://github.com/aws/containers-roadmap/issues/1697',
                        ].join(' '));
                    }
                });
                return result;
            },
        });
        props.fargateProfiles?.forEach((x) => {
            this.addFargateProfile(x);
        });
        props.filters?.forEach((x) => {
            this.addFilter(x);
        });
        props.outputs?.forEach((x) => {
            this.addOutput(x);
        });
        props.parsers?.forEach((x) => {
            this.addParser(x);
        });
    }
    /**
     * Collection of Fluent Bit filter plugins being configured for logging.
     *
     * @group Inputs
     */
    get filters() {
        return [...this._filters];
    }
    /**
     * Collection of Fluent Bit output plugins being configured for logging.
     *
     * @group Inputs
     */
    get outputs() {
        return [...this._outputs];
    }
    /**
     * Collection of Fluent Bit parser plugins being configured for logging.
     *
     * @group Inputs
     */
    get parsers() {
        return [...this._parsers];
    }
    addFargateProfile(profile) {
        this._fargateProfiles.push(profile);
        return this;
    }
    addFilter(filter) {
        this._filters.push(filter);
        return this;
    }
    addOutput(output) {
        this._outputs.push(output);
        return this;
    }
    addParser(parser) {
        this._parsers.push(parser);
        return this;
    }
}
exports.FargateLogger = FargateLogger;
_a = JSII_RTTI_SYMBOL_1;
FargateLogger[_a] = { fqn: "cdk-extensions.k8s_aws.FargateLogger", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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