"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FluentBitFilter = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const _1 = require(".");
/**
 * Standard filter options which can be applied to Fluent Bit to control the
 * output and formatting of logs.
 *
 * Filters change the structure of log records by doing things like adding
 * metadata to a record, restructuring a record, or adding and removing fields.
 */
class FluentBitFilter {
    /**
       * Creates a filter that adds fields to a record that matches the given
       * pattern.
       *
       * @param match A pattern filtering to which records the filter should be
       * applied.
       * @param records The fields to be added to matched records.
       * @returns A filter object that can be applied to the Fluent Bit
       * configuration.
       */
    static appendFields(match, ...records) {
        return new _1.FluentBitRecordModifierFilter({
            match: match,
            records: records,
        });
    }
    /**
       * Creates a filter that removes a set of fields from any records that
       * match a given pattern.
       *
       * @param match A pattern filtering to which records the filter should be
       * applied.
       * @param fields The fields which should be removed from the record if they
       * are present.
       * @returns A filter object that can be applied to the Fluent Bit
       * configuration.
       */
    static blacklistFields(match, ...fields) {
        return new _1.FluentBitRecordModifierFilter({
            match: match,
            remove: fields,
        });
    }
    /**
       * Filters log entries based on a pattern. Log entries can be removed and
       * not forwarded based on whether they match or do not match the given
       * pattern.
       *
       * @param match A pattern filtering to which records the filter should be
       * applied.
       * @param pattern The pattern to match against incoming records.
       * @returns A filter object that can be applied to the Fluent Bit
       * configuration.
       */
    static grep(match, pattern) {
        return new _1.FluentBitGrepFilter({
            match: match,
            pattern: pattern,
        });
    }
    /**
       * Adds Kubernetes metadata to output records including pod information,
       * labels, etc..
       *
       * @param match A pattern filtering to which records the filter should be
       * applied.
       * @returns A filter object that can be applied to the Fluent Bit
       * configuration.
       */
    static kubernetes(match) {
        return new _1.FluentBitKubernetesFilter({
            match: match,
        });
    }
    /**
       * Applies various transformations to matched records including adding,
       * removing, copying, and renaming fields.
       *
       * @param match A pattern filtering to which records the filter should be
       * applied.
       * @param operations The operations to apply to the matched records.
       * @returns A filter object that can be applied to the Fluent Bit
       * configuration.
       */
    static modify(match, ...operations) {
        return new _1.FluentBitModifyFilter({
            match: match,
            operations: operations,
        });
    }
    /**
       * Lifts nested fields in a record up to their parent object.
       *
       * @param match A pattern filtering to which records the filter should be
       * applied.
       * @param nestedUnder The record object under which you want to lift
       * fields.
       * @returns A filter object that can be applied to the Fluent Bit
       * configuration.
       */
    static lift(match, nestedUnder) {
        return new _1.FluentBitNestFilter({
            match: match,
            operation: _1.NestFilterOperation.lift({
                nestedUnder: nestedUnder,
            }),
        });
    }
    /**
       * Nests a set of fields in a record under into a specified object.
       *
       * @param match A pattern filtering to which records the filter should be
       * applied.
       * @param nestUnder The record object under which you want to nest matched
       * fields.
       * @param fields The fields to nest under the specified object.
       * @returns A filter object that can be applied to the Fluent Bit
       * configuration.
       */
    static nest(match, nestUnder, ...fields) {
        return new _1.FluentBitNestFilter({
            match: match,
            operation: _1.NestFilterOperation.nest({
                nestUnder: nestUnder,
                wildcards: fields,
            }),
        });
    }
    /**
       * Applies a set of parsers to matched records.
       *
       * The parser is used to read the input record and set structured fields in
       * the output.
       *
       * @param match A pattern filtering to which records the filter should be
       * applied.
       * @param key The key of the field to be parsed.
       * @param parsers The parser plugins to use to read matched records.
       * @returns A filter object that can be applied to the Fluent Bit
       * configuration.
       */
    static parser(match, key, ...parsers) {
        return new _1.FluentBitParserFilter({
            keyName: key,
            match: match,
            parsers: parsers,
        });
    }
    /**
       * Allows modification of tags set by the input configuration to affect the
       * routing of when records are output.
       *
       * @param match A pattern filtering to which records the filter should be
       * applied.
       * @param rules The rules that define the matching criteria of format of
       * the tag for the matching record.
       * @returns A filter object that can be applied to the Fluent Bit
       * configuration.
       */
    static rewriteTag(match, ...rules) {
        return new _1.FluentBitRewriteTagFilter({
            match: match,
            rules: rules,
        });
    }
    /**
       * Sets an average rate of messages that are allowed to be output over a
       * configured period of time.
       *
       * When the rate of messages surpasses the configured limits messages will
       * be dropped.
       *
       * @param match A pattern filtering to which records the filter should be
       * applied.
       * @param rate The average amount of messages over a given period.
       * @param interval The interval of time over rate should be sampled to
       * calculate an average.
       * @param window Amount of intervals to calculate average over.
       * @returns A filter object that can be applied to the Fluent Bit
       * configuration.
       */
    static throttle(match, interval, rate, window) {
        return new _1.FluentBitThrottleFilter({
            interval: interval,
            match: match,
            rate: rate,
            window: window,
        });
    }
    /**
       * Creates a filter that removes all fields in a record that are not
       * approved.
       *
       * @param match A pattern filtering to which records the filter should be
       * applied.
       * @param fields The fields which are allowed to appear in the output
       * record.
       * @returns A filter object that can be applied to the Fluent Bit
       * configuration.
       */
    static whitelistFields(match, ...fields) {
        return new _1.FluentBitRecordModifierFilter({
            allow: fields,
            match: match,
        });
    }
}
exports.FluentBitFilter = FluentBitFilter;
_a = JSII_RTTI_SYMBOL_1;
FluentBitFilter[_a] = { fqn: "cdk-extensions.k8s_aws.FluentBitFilter", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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