import { IConstruct } from 'constructs';
import { ResolvedFluentBitConfiguration } from '../resolved-fluent-bit-configuration';
import { FluentBitFilterPluginBase, FluentBitFilterPluginCommonOptions } from './filter-plugin-base';
export declare class ModifyCondition {
    /**
       * Condition that returns true if any key matches a specified regular
       * expression.
       *
       * @param regex The regular expression to evaluate against field keys.
       * @returns A ModifyCondition object representing the condition.
       */
    static aKeyMatches(regex: string): ModifyCondition;
    /**
       * Condition that returns true if a specified key does not exist.
       *
       * @param key The key to check for existence.
       * @returns A ModifyCondition object representing the condition.
       */
    static keyDoesNotExists(key: string): ModifyCondition;
    /**
       * Condition that returns true if a specified key exists.
       *
       * @param key The key to check for existence.
       * @returns A ModifyCondition object representing the condition.
       */
    static keyExists(key: string): ModifyCondition;
    /**
       * Condition that returns true if a specified key exists and its value
       * does not match the specified value.
       *
       * @param key The key to check for existence.
       * @param value The value to check for the given key.
       * @returns A ModifyCondition object representing the condition.
       */
    static keyValueDoesNotEqual(key: string, value: string): ModifyCondition;
    /**
       * Condition that returns true if a specified key exists and its value
       * does not match the specified regular expression.
       *
       * @param key The key to check for existence.
       * @param value The regular expression to check for the given key.
       * @returns A ModifyCondition object representing the condition.
       */
    static keyValueDoesNotMatch(key: string, value: string): ModifyCondition;
    /**
       * Condition that returns true if a specified key exists and its value
       * matches the specified regular expression.
       *
       * @param key The key to check for existence.
       * @param value The regular expression to match for the given key.
       * @returns A ModifyCondition object representing the condition.
       */
    static keyValueMatches(key: string, value: string): ModifyCondition;
    /**
       * Condition that returns true if a specified key exists and its value
       * matches the specified value.
       *
       * @param key The key to check for existence.
       * @param value The value to match for the given key.
       * @returns A ModifyCondition object representing the condition.
       */
    static keyValueEquals(key: string, value: string): ModifyCondition;
    /**
       * Condition that returns true if all keys matching a specified regular
       * expression have values that do not match another regular expression.
       *
       * @param key The regular expression to use to filter keys.
       * @param value The regular expression to check the value of fields.
       * @returns A ModifyCondition object representing the condition.
       */
    static matchingKeysDoNotHaveMatchingValues(key: string, value: string): ModifyCondition;
    /**
       * Condition that returns true if all keys matching a specified regular
       * expression have values that match another regular expression.
       *
       * @param key The regular expression to use to filter keys.
       * @param value The regular expression to check the value of fields.
       * @returns A ModifyCondition object representing the condition.
       */
    static matchingKeysHaveMatchingValues(key: string, value: string): ModifyCondition;
    /**
       * Condition that returns true if no key matches a specified regular
       * expression.
       *
       * @param regex The regular expression to evaluate against field keys.
       * @returns A ModifyCondition object representing the condition.
       */
    static noKeyMatches(regex: string): ModifyCondition;
    /**
       * An escape hatch method that allows fo defining custom conditions to be
       * evaluated by the modify Fluent Bit filter plugin.
       *
       * @param condition The name of the condition to be evaluated.
       * @param args The arguments to the operation.
       * @returns A ModifyCondition object representing the options provided.
       */
    static of(condition: string, args: string[]): ModifyCondition;
    /**
       * Internal collection of arguments that apply to the operation.
       */
    private _args;
    /**
       * The name of the condition being evaluated.
       */
    readonly condition: string;
    /**
       * Collection of arguments that apply to the condition.
       */
    get args(): string[];
    /**
       * Creates a new instance of the ModifyCondition class.
       *
       * @param condition The name of the condition being performed.
       * @param args The arguments that apply to the condition.
       */
    private constructor();
    /**
       * Gets a string representation of the arguments of this condition for use
       * in a Fluent Bit plugin field.
       *
       * @returns A fluent bit value string.
       */
    toString(): string;
}
export declare class ModifyOperation {
    /**
       * Sets a field in the output to a specific value.
       *
       * If a field with the same name already exists it will be kept as is.
       *
       * @param key The key name of the field to set.
       * @param value The value to set for the specified field.
       * @returns A ModifyOperation object representing the add operation.
       */
    static add(key: string, value: string): ModifyOperation;
    /**
       * Copies a field from the input to a field with a new name if the field
       * exists and a field with the new name does not exist.
       *
       * If a field with the new name already exists it is overwritten.
       *
       * @param originalKey The key in the input to be copied.
       * @param newKey The new name of the field to be copied to.
       * @returns A ModifyOperation object representing the copy operation.
       */
    static copy(originalKey: string, newKey: string): ModifyOperation;
    /**
       * Copies a field from the input to a field with a new name if the field
       * exists and a field with the new name does not exist.
       *
       * @param originalKey The key in the input to be copied.
       * @param newKey The new name of the field to be copied to.
       * @returns A ModifyOperation object representing the copy operation.
       */
    static hardCopy(originalKey: string, newKey: string): ModifyOperation;
    /**
       * Renames a field from the input if the field exists.
       *
       * If a field with the desired name already exists it is overwritten.
       *
       * @param originalKey The key in the input to be renamed.
       * @param renamedKey The new name of the key in the output.
       * @returns A ModifyOperation object representing the rename operation.
       */
    static hardRename(originalKey: string, renamedKey: string): ModifyOperation;
    /**
       * Moves fiels matching the given wildcard key to the end of the message.
       *
       * @param key The wildcard to to match.
       * @returns A ModifyOperation object representing the move operation.
       */
    static moveToEnd(key: string): ModifyOperation;
    /**
       * Moves fiels matching the given wildcard key to the start of the message.
       *
       * @param key The wildcard to to match.
       * @returns A ModifyOperation object representing the move operation.
       */
    static moveToStart(key: string): ModifyOperation;
    /**
       * Removes a field in the output with a specific key.
       *
       * @param key The key name of the field to remove.
       * @returns A ModifyOperation object representing the remove operation.
       */
    static remove(key: string): ModifyOperation;
    /**
       * Removes all fields in the output matching the regular expression.
       *
       * @param regex The regular expression specifying which fields to remove.
       * @returns A ModifyOperation object representing the remove operation.
       */
    static removeRegex(regex: string): ModifyOperation;
    /**
       * Removes all fields in the output matching the wildcard key.
       *
       * @param key The wildcard expression specifying which fields to remove.
       * @returns A ModifyOperation object representing the remove operation.
       */
    static removeWildcard(key: string): ModifyOperation;
    /**
       * Renames a field from the input if the field exists and a field with the
       * new name does not exist.
       *
       * @param originalKey The key in the input to be renamed.
       * @param renamedKey The new name of the key in the output.
       * @returns A ModifyOperation object representing the rename operation.
       */
    static rename(originalKey: string, renamedKey: string): ModifyOperation;
    /**
       * Sets a field in the output to a specific value.
       *
       * If a field with the same name already exists it will be overridden with
       * the specified value.
       *
       * @param key The key name of the field to set.
       * @param value The value to set for the specified field.
       * @returns A ModifyOperation object representing the set operation.
       */
    static set(key: string, value: string): ModifyOperation;
    /**
       * An escape hatch method that allows fo defining custom operations to be
       * performed by the modify Fluent Bit filter plugin.
       *
       * @param operation The name of the operation to be performed.
       * @param args The arguments to the operation.
       * @returns A ModifyOperation object representing the options provided.
       */
    static of(operation: string, args: string[]): ModifyOperation;
    /**
       * Internal collection of arguments that apply to the operation.
       */
    private _args;
    /**
       * The name of the operation being performed.
       */
    readonly operation: string;
    /**
       * Collection of arguments that apply to the operation.
       */
    get args(): string[];
    /**
       * Creates a new instance of the Modify operations class.
       *
       * @param operation The name of the operation being performed.
       * @param args The arguments that apply to the operation.
       */
    private constructor();
    /**
       * Gets a string representation of the arguments of this operation for use
       * in a Fluent Bit plugin field.
       *
       * @returns A fluent bit value string.
       */
    toString(): string;
}
/**
 * Options for configuring the Modify Fluent Bit filter plugin.
 *
 * @see [Modify Plugin Documention](https://docs.fluentbit.io/manual/pipeline/filters/modify)
 */
export interface FluentBitModifyFilterOptions extends FluentBitFilterPluginCommonOptions {
    readonly conditions?: ModifyCondition[];
    readonly operations?: ModifyOperation[];
}
/**
 * A Fluent Bit filter that allows changing records using rules and conditions.
 */
export declare class FluentBitModifyFilter extends FluentBitFilterPluginBase {
    /**
       * Internal collection of conditions to apply for the filter.
       */
    private readonly _conditions;
    /**
       * Internal collection of operations to apply for the filter.
       */
    private readonly _operations;
    /**
       * Collection of conditions to apply for the filter.
       */
    get conditions(): ModifyCondition[];
    /**
       * Collection of operations to apply for the filter.
       */
    get operations(): ModifyOperation[];
    /**
       * Creates a new instance of the FluentBitModifyFilter class.
       *
       * @param options The configuration options to use for filter.
       */
    constructor(options?: FluentBitModifyFilterOptions);
    /**
       * Adds a new condition to the modify filter.
       *
       * All conditions must evaluate to `true` in order for operations are
       * performed.
       *
       * If one or more conditions do not evaluate to true, no conditions are
       * performed.
       *
       * @param condition The condition to add to the filter.
       * @returns The modify filter to which the condition was added.
       */
    addCondition(condition: ModifyCondition): FluentBitModifyFilter;
    /**
       * Adds a new operation to the modify filter.
       *
       * @param operation The operation to add to the filter.
       * @returns The modify filter to which the operation was added.
       */
    addOperation(operation: ModifyOperation): FluentBitModifyFilter;
    /**
       * Builds a configuration for this plugin and returns the details for
       * consumtion by a resource that is configuring logging.
       *
       * @param _scope The construct configuring logging using Fluent Bit.
       * @returns A configuration for the plugin that con be used by the resource
       * configuring logging.
       */
    bind(_scope: IConstruct): ResolvedFluentBitConfiguration;
}
