import { IConstruct } from 'constructs';
import { ResolvedFluentBitConfiguration } from '../resolved-fluent-bit-configuration';
import { FluentBitFilterPluginBase, FluentBitFilterPluginCommonOptions } from './filter-plugin-base';
export interface LiftOptions {
    /**
       * Lift records nested under the this key.
       */
    readonly nestedUnder: string;
}
export interface NestOptions {
    /**
       * Nest records matching `wildcard` under this key.
       */
    readonly nestUnder: string;
    /**
       * Nest records which field matches this wildcard,
       */
    readonly wildcards: string[];
}
/**
 * The modes that the Fluent Bit Nest filter plugin can work in.
 */
export declare enum NestFilterOperationType {
    /**
       * Lift data from a nested object.
       */
    LIFT = "lift",
    /**
       * Nest data into a specified object.
       */
    NEST = "nest"
}
/**
 * Represents an operation with excludive options that can be performed by the
 * Fluent Bit Nest filter plugin.
 */
export interface INestFilterOperation {
    readonly fields: {
        [key: string]: string[];
    };
    readonly operation: NestFilterOperationType;
}
/**
 * Operations with exclusive options that can be performed by the Fluent Bit
 * Nest filter plugin.
 */
export declare class NestFilterOperation implements INestFilterOperation {
    static lift(options: LiftOptions): INestFilterOperation;
    static nest(options: NestOptions): INestFilterOperation;
    /**
       * The fields representing configuration options for the operation.
       */
    readonly fields: {
        [key: string]: string[];
    };
    /**
       * The type of operation to be performed.
       */
    readonly operation: NestFilterOperationType;
    /**
       * Creates a new instance of the NestFilterOperation class.
       *
       * @param operation The type of operation to be performed.
       * @param fields The fields representing configuration options for the
       * operation.
       */
    private constructor();
}
/**
 * Options for configuring the Nest Fluent Bit filter plugin.
 *
 * @see [Nest Plugin Documention](https://docs.fluentbit.io/manual/pipeline/filters/nest)
 */
export interface FluentBitNestFilterOptions extends FluentBitFilterPluginCommonOptions {
    /**
       * Prefix affected keys with this string.
       */
    readonly addPrefix?: string;
    /**
       * The operation the filter will perform.
       */
    readonly operation: NestFilterOperation;
    /**
       * Remove prefix from affected keys if it matches this string.
       */
    readonly removePrefix?: string;
}
/**
 * A Fluent Bit filter that allows operating on or with nested data.
 */
export declare class FluentBitNestFilter extends FluentBitFilterPluginBase {
    /**
       * Prefix affected keys with this string.
       */
    readonly addPrefix?: string;
    /**
       * Operation specific details for the plugin.
       */
    readonly operation: NestFilterOperation;
    /**
       * Remove prefix from affected keys if it matches this string.
       */
    readonly removePrefix?: string;
    /**
       * Creates a new instance of the FluentBitNestFilter class.
       *
       * @param options The configuration options for the plugin.
       */
    constructor(options: FluentBitNestFilterOptions);
    /**
       * Builds a configuration for this plugin and returns the details for
       * consumtion by a resource that is configuring logging.
       *
       * @param _scope The construct configuring logging using Fluent Bit.
       * @returns A configuration for the plugin that con be used by the resource
       * configuring logging.
       */
    bind(_scope: IConstruct): ResolvedFluentBitConfiguration;
}
