import { IConstruct } from 'constructs';
import { ResolvedFluentBitConfiguration } from '../resolved-fluent-bit-configuration';
import { FluentBitFilterPluginBase, FluentBitFilterPluginCommonOptions } from './filter-plugin-base';
/**
 * Represents a record field to be added by the record modifier Fluent Bit
 * filter plugin.
 */
export interface AppendedRecord {
    /**
       * The name of the field to be added.
       */
    readonly fieldName: string;
    /**
       * The value that the added field should be set to.
       */
    readonly value: string;
}
/**
 * Options for configuring the Record Modifier Fluent Bit filter plugin.
 *
 * @see [Record Modifier Plugin Documention](https://docs.fluentbit.io/manual/pipeline/filters/record-modifier)
 */
export interface FluentBitRecordModifierFilterOptions extends FluentBitFilterPluginCommonOptions {
    /**
       * If a tag is not match, that field is removed.
       */
    readonly allow?: string[];
    /**
       * Add fields to the output.
       */
    readonly records?: AppendedRecord[];
    /**
       * If a tag is match, that field is removed.
       */
    readonly remove?: string[];
}
/**
 * A Fluent Bit filter that allows appending fields or excluding specific
 * fields.
 */
export declare class FluentBitRecordModifierFilter extends FluentBitFilterPluginBase {
    /**
       * Internal collection of tags that are allowed on a matched input record.
       *
       * If a tag is not match it is removed.
       */
    private readonly _allow;
    /**
       * Internal collection of the records to be appending to matched input.
       */
    private readonly _records;
    /**
       * Internal collection of tags to exclude from a matched input record.
       *
       * If a tag is matched it is removed.
       */
    private readonly _remove;
    /**
       * Collection of tags that are allowed on a matched input record.
       *
       * If a tag is not matched it is removed.
       */
    get allow(): string[];
    /**
       * Collection of the records to be appending to matched input.
       */
    get records(): AppendedRecord[];
    /**
       * Collection of tags to exclude from a matched input record.
       *
       * If a tag is matched it is removed.
       */
    get remove(): string[];
    /**
       * Creates a new instance of the FluentBitRecordModifierFilter class.
       *
       * @param options The configuration options to use for filter.
       */
    constructor(options?: FluentBitRecordModifierFilterOptions);
    /**
       * Adds a tag to be allowed on a matched input record.
       *
       * If a tag is not matched it is removed.
       *
       * @param tag The tag to add to the allow list
       * @returns The record modifier filter that the tag plugin was registered
       * with.
       */
    addAllow(tag: string): FluentBitRecordModifierFilter;
    /**
       * Add a record to be appended to matched events.
       *
       * @param record The record to be appended to matched input.
       * @returns The record modifier filter that the tag plugin was registered
       * with.
       */
    addRecord(record: AppendedRecord): FluentBitRecordModifierFilter;
    /**
       * Adds a tag to be removed on a matched input record.
       *
       * If a tag is matched it is removed.
       *
       * @param tag The tag to add to the allow list
       * @returns The record modifier filter that the tag plugin was registered
       * with.
       */
    addRemove(tag: string): FluentBitRecordModifierFilter;
    /**
       * Builds a configuration for this plugin and returns the details for
       * consumtion by a resource that is configuring logging.
       *
       * @param _scope The construct configuring logging using Fluent Bit.
       * @returns A configuration for the plugin that con be used by the resource
       * configuring logging.
       */
    bind(_scope: IConstruct): ResolvedFluentBitConfiguration;
}
