import { IRole } from 'aws-cdk-lib/aws-iam';
import { ILogGroup, ILogStream, RetentionDays } from 'aws-cdk-lib/aws-logs';
import { IConstruct } from 'constructs';
import { ResolvedFluentBitConfiguration } from '../resolved-fluent-bit-configuration';
import { FluentBitOutputPluginBase, FluentBitOutputPluginCommonOptions } from './output-plugin-base';
/**
 * Represents valid log stream output configuration options to be used by
 * Fluent Bit when writing to CloudWatch Logs.
 */
export declare class FluentBitLogStreamOutput {
    /**
     * Sets output to be a log stream resource object.
     *
     * @param logStream The log stream where records should be written.
     * @returns A FluentBitLogStreamOutput object representing the configured
     * log stream destination.
     */
    static fromLogStream(logStream: ILogStream): FluentBitLogStreamOutput;
    /**
     * Sets output to a named log stream.
     *
     * If a log stream with the given name doesn't exist in the configured log
     * group a log stream with the given name will be created.
     *
     * @param name The name of the log stream where records should be written.
     * @returns A FluentBitLogStreamOutput object representing the configured
     * log stream destination.
     */
    static fromName(name: string): FluentBitLogStreamOutput;
    /**
     * Sets output to a prefixed log stream.
     *
     * Log streams will be created on a per-pod basis with the name oof the log
     * streams starting with the provided prefix.
     *
     * @param prefix The prefix for log streams which will be created.
     * @returns A FluentBitLogStreamOutput object representing the configured
     * log stream destination.
     */
    static fromPrefix(prefix: string): FluentBitLogStreamOutput;
    /**
     * The name of the log stream where records should be created.
     */
    readonly logStreamName?: string;
    /**
     * The prefix for log streams that will be created on a per-pod basis.
     */
    readonly logStreamPrefix?: string;
    /**
     * Creates a new instance of the FluentBitLogStreamOutput class.
     *
     * @param options  Options for configuring log stream output.
     */
    private constructor();
}
/**
 * Represents valid log group output configuration options to be used by
 * Fluent Bit when writing to CloudWatch Logs.
 */
export declare class FluentBitLogGroupOutput {
    /**
     * Sets a flag saying that a log group should be created automatically.
     *
     * Depending on the configuration of the plugin, this flag will either cause
     * permissions to be granted for Fluent Bit to create the log group itself or
     * the plugin CDK resource will create a Log Group and use that as the
     * destination.
     *
     * @returns A FluentBitLogGroupOutput object representing the configured log
     * group destination.
     */
    static create(): FluentBitLogGroupOutput;
    /**
     * Sets the destination log group to a LogGroup CDK resource.
     *
     * @param logGroup The log group where output records should be written.
     * @returns A FluentBitLogGroupOutput object representing the configured log
     * group destination.
     */
    static fromLogGroup(logGroup: ILogGroup): FluentBitLogGroupOutput;
    /**
     * Sets the destination for logs to the named log group.
     *
     * @param name The name of the log group where output records should be written.
     * @returns A FluentBitLogGroupOutput object representing the configured log
     * group destination.
     */
    static fromName(name: string, create?: boolean): FluentBitLogGroupOutput;
    /**
     * Flag that determines whether or not a log group should be automatically
     * created.
     */
    readonly autoCreate?: boolean;
    /**
     * A log group resource object to use as the destination.
     */
    readonly logGroup?: ILogGroup;
    /**
     * The name for the log group that should be used for output records.
     */
    readonly logGroupName?: string;
    /**
     * Creates a new instance of the FluentBitLogStreamOutput class.
     *
     * @param options  Options for configuring log stream output.
     */
    private constructor();
}
/**
 * Options for configuring the CloudWatch Logs Fluent Bit output plugin.
 *
 * @see [CloudWatch Logs Plugin Documention](https://docs.fluentbit.io/manual/pipeline/outputs/cloudwatch)
 */
export interface FluentBitCloudWatchLogsOutputOptions extends FluentBitOutputPluginCommonOptions {
    /**
       * Automatically create the log group.
       *
       * @default false
       */
    readonly autoCreateGroup?: boolean;
    /**
       * Immediately retry failed requests to AWS services once. This option does
       * not affect the normal Fluent Bit retry mechanism with backoff. Instead,
       * it enables an immediate retry with no delay for networking errors, which
       * may help improve throughput when there are transient/random networking
       * issues.
       *
       * @default true
       */
    readonly autoRetryRequests?: boolean;
    /**
       * Specify a custom endpoint for the CloudWatch Logs API.
       */
    readonly endpoint?: string;
    /**
       * By default, the whole log record will be sent to CloudWatch. If you
       * specify a key name with this option, then only the value of that key
       * will be sent to CloudWatch.
       */
    readonly logKey?: string;
    /**
       * An optional parameter that can be used to tell CloudWatch the format of
       * the data. A value of json/emf enables CloudWatch to extract custom
       * metrics embedded in a JSON payload.
       *
       * @see [Embedded Metric Format](https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Embedded_Metric_Format_Specification.html)
       */
    readonly logFormat?: string;
    /**
       * The CloudWatch Log Group configuration for output records.
       */
    readonly logGroup?: FluentBitLogGroupOutput;
    /**
       * Template for Log Group name using Fluent Bit record_accessor syntax.
       *
       * This field is optional and if configured it overrides the configured Log
       * Group.
       *
       * If the template translation fails, an error is logged and the provided
       * Log Group (which is still required) is used instead.
       *
       * @see [Fluent Bit record accessor snytax](https://docs.fluentbit.io/manual/administration/configuring-fluent-bit/classic-mode/record-accessor)
       */
    readonly logGroupTemplate?: string;
    /**
       * If set to a number greater than zero, and newly create log group's
       * retention policy is set to this many days.
       */
    readonly logRetention?: RetentionDays;
    /**
       * The CloudWatch LogStream configuration for outbound records.
       */
    readonly logStream?: FluentBitLogStreamOutput;
    /**
       * Template for Log Stream name using Fluent Bit record accessor syntax.
       * This field is optional and if configured it overrides the other log
       * stream options. If the template translation fails, an error is logged
       * and the logStream or logStreamPrefix are used instead (and thus one of
       * those fields is still required to be configured).
       *
       * @see [Fluent Bit record accessor snytax](https://docs.fluentbit.io/manual/administration/configuring-fluent-bit/classic-mode/record-accessor)
       */
    readonly logStreamTemplate?: string;
    /**
       * A list of lists containing the dimension keys that will be applied to
       * all metrics. The values within a dimension set MUST also be members on
       * the root-node.
       *
       * @see [Dimensions](https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Dimension)
       */
    readonly metricDimensions?: string[];
    /**
       * An optional string representing the CloudWatch namespace for the
       * metrics.
       *
       * @see [Metric Tutorial](https://docs.fluentbit.io/manual/pipeline/outputs/cloudwatch#metrics-tutorial)
       */
    readonly metricNamespace?: string;
    /**
       * The AWS region.
       */
    readonly region?: string;
    /**
       * ARN of an IAM role to assume (for cross account access).
       */
    readonly role?: IRole;
    /**
       * Specify a custom STS endpoint for the AWS STS API.
       */
    readonly stsEndpoint?: string;
}
/**
 * Represents configuration for outputing logs from Fluent Bit to CloudWatch
 * Logs.
 */
export declare class FluentBitCloudWatchLogsOutput extends FluentBitOutputPluginBase {
    /**
       * Automatically create the log group.
        *
        * @group Inputs
       */
    readonly autoCreateGroup?: boolean;
    /**
        * Immediately retry failed requests to AWS services once. This option does
        * not affect the normal Fluent Bit retry mechanism with backoff. Instead,
        * it enables an immediate retry with no delay for networking errors, which
        * may help improve throughput when there are transient/random networking
        * issues.
        *
        * @group Inputs
        */
    readonly autoRetryRequests?: boolean;
    /**
        * Specify a custom endpoint for the CloudWatch Logs API.
        *
        * @group Inputs
        */
    readonly endpoint?: string;
    /**
        * By default, the whole log record will be sent to CloudWatch. If you
        * specify a key name with this option, then only the value of that key
        * will be sent to CloudWatch.
        *
        * @group Inputs
        */
    readonly logKey?: string;
    /**
        * An optional parameter that can be used to tell CloudWatch the format of
        * the data. A value of json/emf enables CloudWatch to extract custom
        * metrics embedded in a JSON payload.
        *
        * @see [Embedded Metric Format](https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Embedded_Metric_Format_Specification.html)
        *
        * @group Inputs
        */
    readonly logFormat?: string;
    /**
       * The CloudWatch Log Group configuration for output records.
       */
    readonly logGroup: FluentBitLogGroupOutput;
    /**
        * Template for Log Group name using Fluent Bit record_accessor syntax.
        *
        * This field is optional and if configured it overrides the configured Log
        * Group.
        *
        * If the template translation fails, an error is logged and the provided
        * Log Group (which is still required) is used instead.
        *
        * @see [Fluent Bit record accessor snytax](https://docs.fluentbit.io/manual/administration/configuring-fluent-bit/classic-mode/record-accessor)
        *
        * @group Inputs
        */
    readonly logGroupTemplate?: string;
    /**
        * If set to a number greater than zero, and newly create log group's
        * retention policy is set to this many days.
        *
        * @group Inputs
        */
    readonly logRetention?: RetentionDays;
    /**
       * The CloudWatch LogStream configuration for outbound records.
       */
    readonly logStream: FluentBitLogStreamOutput;
    /**
        * Template for Log Stream name using Fluent Bit record accessor syntax.
        * This field is optional and if configured it overrides the other log
        * stream options. If the template translation fails, an error is logged
        * and the logStream or logStreamPrefix are used instead (and thus one of
        * those fields is still required to be configured).
        *
        * @see [Fluent Bit record accessor snytax](https://docs.fluentbit.io/manual/administration/configuring-fluent-bit/classic-mode/record-accessor)
        *
        * @group Inputs
        */
    readonly logStreamTemplate?: string;
    /**
        * A list of lists containing the dimension keys that will be applied to
        * all metrics. The values within a dimension set MUST also be members on
        * the root-node.
        *
        * @see [Dimensions](https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Dimension)
        *
        * @group Inputs
        */
    readonly metricDimensions?: string[];
    /**
        * An optional string representing the CloudWatch namespace for the
        * metrics.
        *
        * @see [Metric Tutorial](https://docs.fluentbit.io/manual/pipeline/outputs/cloudwatch#metrics-tutorial)
        *
        * @group Inputs
        */
    readonly metricNamespace?: string;
    /**
        * The AWS region.
        *
        * @group Inputs
        */
    readonly region?: string;
    /**
        * ARN of an IAM role to assume (for cross account access).
        *
        * @group Inputs
        */
    readonly role?: IRole;
    /**
        * Specify a custom STS endpoint for the AWS STS API.
        *
        * @group Inputs
        */
    readonly stsEndpoint?: string;
    /**
      * Creates a new instance of the FluentBitCloudWatchLogsOutput class.
      *
      * @param options Options for configuring the output.
      */
    constructor(options?: FluentBitCloudWatchLogsOutputOptions);
    /**
       * Builds a configuration for this plugin and returns the details for
       * consumtion by a resource that is configuring logging.
       *
       * @param scope The construct configuring logging using Fluent Bit.
       * @returns A configuration for the plugin that con be used by the resource
       * configuring logging.
       */
    bind(scope: IConstruct): ResolvedFluentBitConfiguration;
    /**
       * Gets a log group object that can be used to set the required
       * log_group_name property if one hasn't been provided.
       *
       * @param scope The construct configuring logging using Fluent Bit.
       * @returns The log group where output logs should be sent.
       */
    private getLogGroup;
}
