"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FluentBitCloudWatchLogsOutput = exports.FluentBitLogGroupOutput = exports.FluentBitLogStreamOutput = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const output_plugin_base_1 = require("./output-plugin-base");
/**
 * Represents valid log stream output configuration options to be used by
 * Fluent Bit when writing to CloudWatch Logs.
 */
class FluentBitLogStreamOutput {
    /**
     * Creates a new instance of the FluentBitLogStreamOutput class.
     *
     * @param options  Options for configuring log stream output.
     */
    constructor(options) {
        this.logStreamName = options.logStreamName;
        this.logStreamPrefix = options.logStreamPrefix;
    }
    /**
     * Sets output to be a log stream resource object.
     *
     * @param logStream The log stream where records should be written.
     * @returns A FluentBitLogStreamOutput object representing the configured
     * log stream destination.
     */
    static fromLogStream(logStream) {
        return new FluentBitLogStreamOutput({
            logStreamName: logStream.logStreamName,
        });
    }
    /**
     * Sets output to a named log stream.
     *
     * If a log stream with the given name doesn't exist in the configured log
     * group a log stream with the given name will be created.
     *
     * @param name The name of the log stream where records should be written.
     * @returns A FluentBitLogStreamOutput object representing the configured
     * log stream destination.
     */
    static fromName(name) {
        return new FluentBitLogStreamOutput({
            logStreamName: name,
        });
    }
    /**
     * Sets output to a prefixed log stream.
     *
     * Log streams will be created on a per-pod basis with the name oof the log
     * streams starting with the provided prefix.
     *
     * @param prefix The prefix for log streams which will be created.
     * @returns A FluentBitLogStreamOutput object representing the configured
     * log stream destination.
     */
    static fromPrefix(prefix) {
        return new FluentBitLogStreamOutput({
            logStreamPrefix: prefix,
        });
    }
}
exports.FluentBitLogStreamOutput = FluentBitLogStreamOutput;
_a = JSII_RTTI_SYMBOL_1;
FluentBitLogStreamOutput[_a] = { fqn: "cdk-extensions.k8s_aws.FluentBitLogStreamOutput", version: "0.0.38" };
/**
 * Represents valid log group output configuration options to be used by
 * Fluent Bit when writing to CloudWatch Logs.
 */
class FluentBitLogGroupOutput {
    /**
     * Creates a new instance of the FluentBitLogStreamOutput class.
     *
     * @param options  Options for configuring log stream output.
     */
    constructor(options) {
        this.autoCreate = options.create;
        this.logGroup = options.logGroup;
        this.logGroupName = options.logGroupName;
    }
    /**
     * Sets a flag saying that a log group should be created automatically.
     *
     * Depending on the configuration of the plugin, this flag will either cause
     * permissions to be granted for Fluent Bit to create the log group itself or
     * the plugin CDK resource will create a Log Group and use that as the
     * destination.
     *
     * @returns A FluentBitLogGroupOutput object representing the configured log
     * group destination.
     */
    static create() {
        return new FluentBitLogGroupOutput({
            create: true,
        });
    }
    /**
     * Sets the destination log group to a LogGroup CDK resource.
     *
     * @param logGroup The log group where output records should be written.
     * @returns A FluentBitLogGroupOutput object representing the configured log
     * group destination.
     */
    static fromLogGroup(logGroup) {
        return new FluentBitLogGroupOutput({
            logGroup: logGroup,
        });
    }
    /**
     * Sets the destination for logs to the named log group.
     *
     * @param name The name of the log group where output records should be written.
     * @returns A FluentBitLogGroupOutput object representing the configured log
     * group destination.
     */
    static fromName(name, create) {
        return new FluentBitLogGroupOutput({
            create: create,
            logGroupName: name,
        });
    }
}
exports.FluentBitLogGroupOutput = FluentBitLogGroupOutput;
_b = JSII_RTTI_SYMBOL_1;
FluentBitLogGroupOutput[_b] = { fqn: "cdk-extensions.k8s_aws.FluentBitLogGroupOutput", version: "0.0.38" };
/**
 * Represents configuration for outputing logs from Fluent Bit to CloudWatch
 * Logs.
 */
class FluentBitCloudWatchLogsOutput extends output_plugin_base_1.FluentBitOutputPluginBase {
    /**
      * Creates a new instance of the FluentBitCloudWatchLogsOutput class.
      *
      * @param options Options for configuring the output.
      */
    constructor(options = {}) {
        super('cloudwatch_logs', options);
        this.autoCreateGroup = options.autoCreateGroup;
        this.autoRetryRequests = options.autoRetryRequests;
        this.endpoint = options.endpoint;
        this.logFormat = options.logFormat;
        this.logGroup = options.logGroup ?? FluentBitLogGroupOutput.create();
        this.logGroupTemplate = options.logGroupTemplate;
        this.logKey = options.logKey;
        this.logRetention = options.logRetention;
        this.logStream = options.logStream ?? FluentBitLogStreamOutput.fromPrefix('eks');
        this.logStreamTemplate = options.logStreamTemplate;
        this.metricDimensions = options.metricDimensions;
        this.metricNamespace = options.metricNamespace;
        this.region = options.region;
        this.role = options.role;
        this.stsEndpoint = options.stsEndpoint;
    }
    /**
       * Builds a configuration for this plugin and returns the details for
       * consumtion by a resource that is configuring logging.
       *
       * @param scope The construct configuring logging using Fluent Bit.
       * @returns A configuration for the plugin that con be used by the resource
       * configuring logging.
       */
    bind(scope) {
        const logGroup = this.getLogGroup(scope);
        return {
            configFile: super.renderConfigFile({
                auto_create_group: this.autoCreateGroup,
                auto_retry_requests: this.autoRetryRequests,
                endpoint: this.endpoint,
                log_format: this.logFormat,
                log_group_name: logGroup.logGroupName,
                log_group_template: this.logGroupTemplate,
                log_key: this.logKey,
                log_retention_days: this.logRetention?.toString(),
                log_stream_name: this.logStream.logStreamName,
                log_stream_prefix: this.logStream.logStreamPrefix,
                log_stream_template: this.logStreamTemplate,
                metric_dimensions: this.metricDimensions?.join(','),
                metric_namespace: this.metricNamespace,
                region: this.region ?? aws_cdk_lib_1.Stack.of(scope).region,
                role_arn: this.role?.roleArn,
                sts_endpoint: this.stsEndpoint,
            }),
            permissions: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        ...(this.autoCreateGroup ? ['logs:CreateLogGroup'] : []),
                        'logs:DescribeLogStreams',
                    ],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: [
                        aws_cdk_lib_1.Stack.of(scope).formatArn({
                            arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                            resource: 'log-group',
                            resourceName: logGroup.logGroupName,
                            service: 'log-group',
                        }),
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    actions: [
                        'logs:CreateLogStream',
                        'logs:PutLogEvents',
                    ],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: [
                        logGroup.logGroupArn,
                    ],
                }),
            ],
        };
    }
    /**
       * Gets a log group object that can be used to set the required
       * log_group_name property if one hasn't been provided.
       *
       * @param scope The construct configuring logging using Fluent Bit.
       * @returns The log group where output logs should be sent.
       */
    getLogGroup(scope) {
        const logGroupSuffix = this.logGroup.logGroupName ? `-${this.logGroup.logGroupName}` : '::default';
        const stubSuffix = this.logGroup.autoCreate ? '' : '::stub';
        const logGroupId = `fluent-bit-output-log-group${logGroupSuffix}${stubSuffix}`;
        const inheritedLogGroup = scope.node.tryFindChild(logGroupId);
        if (this.logGroup.logGroup) {
            return this.logGroup.logGroup;
        }
        else if (inheritedLogGroup) {
            return inheritedLogGroup;
        }
        else if (this.logGroup.autoCreate) {
            return new aws_logs_1.LogGroup(scope, logGroupId, {
                logGroupName: this.logGroup.logGroupName,
                retention: this.logRetention ?? aws_logs_1.RetentionDays.TWO_WEEKS,
            });
        }
        else {
            return aws_logs_1.LogGroup.fromLogGroupName(scope, logGroupId, this.logGroup.logGroupName ?? aws_cdk_lib_1.Names.uniqueId(scope));
        }
    }
}
exports.FluentBitCloudWatchLogsOutput = FluentBitCloudWatchLogsOutput;
_c = JSII_RTTI_SYMBOL_1;
FluentBitCloudWatchLogsOutput[_c] = { fqn: "cdk-extensions.k8s_aws.FluentBitCloudWatchLogsOutput", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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