"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretsManagerSecretStore = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_secret_store_1 = require("./aws-secret-store");
const external_secret_1 = require("./external-secret");
const secrets_manager_reference_1 = require("./lib/secrets-manager-reference");
/**
 * A secret store that allows secrets from AWS Secrets Managers to be
 * synchronized into Kubernetes as Kubernetes secrets.
 */
class SecretsManagerSecretStore extends aws_secret_store_1.AwsSecretStore {
    /**
     * Creates a new instance of the SecretsManagerSecretStore class.
     *
     * @param scope A CDK Construct that will serve as this resource's parent
     * in the construct tree.
     * @param id A name to be associated with the resource and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param props Arguments related to the configuration of the resource.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            service: 'SecretsManager',
        });
        this.serviceAccount.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'secretsmanager:DescribeSecret',
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:ListSecretVersionIds',
            ],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [
                this.stack.formatArn({
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                    resource: 'secret',
                    resourceName: '*',
                    service: 'secretsmanager',
                }),
            ],
        }));
    }
    /**
     * Registers a new Secrets Manager secret to be synchronized into Kubernetes.
     *
     * @param id The ID of the secret import configuration in the CDK construct
     * tree.
     *
     * The configuration is placed under the Secrets Manager secret it
     * synchronizes and so must be unique per secret.
     * @param secret The Secrets Manager secret to synchronize into Kubernetes.
     * @param options Configuration options for how the secret should be
     * synchronized.
     * @returns The external secret configuration that was added.
     */
    addSecret(id, secret, options = {}) {
        const output = new external_secret_1.ExternalSecret(secret, `external-secret-${id}`, {
            cluster: this.cluster,
            name: options.name,
            namespace: this.namespace,
            secrets: [
                new secrets_manager_reference_1.SecretsManagerReference(secret, {
                    fields: options.fields,
                }),
            ],
            secretStore: this,
        });
        if (this.stack === secret.stack) {
            output.node.addDependency(this.manifest);
        }
        return output;
    }
}
exports.SecretsManagerSecretStore = SecretsManagerSecretStore;
_a = JSII_RTTI_SYMBOL_1;
SecretsManagerSecretStore[_a] = { fqn: "cdk-extensions.k8s_aws.SecretsManagerSecretStore", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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