"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SsmParameterSecretStore = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_secret_store_1 = require("./aws-secret-store");
const external_secret_1 = require("./external-secret");
const ssm_parameter_reference_1 = require("./lib/ssm-parameter-reference");
/**
 * A secret store that allows parameters from Systems Manager to be
 * synchronized into Kubernetes as Kubernetes secrets.
 */
class SsmParameterSecretStore extends aws_secret_store_1.AwsSecretStore {
    /**
     * Creates a new instance of the SsmParameterSecretStore class.
     *
     * @param scope A CDK Construct that will serve as this resource's parent
     * in the construct tree.
     * @param id A name to be associated with the resource and used in resource
     * naming. Must be unique within the context of 'scope'.
     * @param props Arguments related to the configuration of the resource.
     */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            service: 'ParameterStore',
        });
        this.serviceAccount.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ssm:GetParameter*',
            ],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [
                this.stack.formatArn({
                    resource: 'parameter',
                    resourceName: '*',
                    service: 'ssm',
                }),
            ],
        }));
    }
    /**
     * Registers a new SSSM parameter to be synchronized into Kubernetes.
     *
     * @param id The ID of the secret import configuration in the CDK construct
     * tree.
     *
     * The configuration is placed under the SSM parameter it synchronizes and so
     * must be unique per secret.
     * @param parameter The SSM parameter to synchronize into Kubernetes.
     * @param options Configuration options for how the secret should be
     * synchronized.
     * @returns The external secret configuration that was added.
     */
    addSecret(id, parameter, options = {}) {
        const output = new external_secret_1.ExternalSecret(parameter, `external-secret-${id}`, {
            cluster: this.cluster,
            name: options.name,
            namespace: this.namespace,
            secrets: [
                new ssm_parameter_reference_1.SsmParameterReference(parameter, {
                    fields: options.fields,
                }),
            ],
            secretStore: this,
        });
        if (this.stack === parameter.stack) {
            output.node.addDependency(this.manifest);
        }
        return output;
    }
}
exports.SsmParameterSecretStore = SsmParameterSecretStore;
_a = JSII_RTTI_SYMBOL_1;
SsmParameterSecretStore[_a] = { fqn: "cdk-extensions.k8s_aws.SsmParameterSecretStore", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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