"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Destination = exports.S3CompressionFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const formatting_1 = require("../../../utils/formatting");
const cloudwatch_logging_configuration_1 = require("../cloudwatch-logging-configuration");
const delivery_stream_destination_1 = require("./delivery-stream-destination");
var S3CompressionFormat;
(function (S3CompressionFormat) {
    S3CompressionFormat["GZIP"] = "GZIP";
    S3CompressionFormat["HADOOP_SNAPPY"] = "HADOOP_SNAPPY";
    S3CompressionFormat["SNAPPY"] = "Snappy";
    S3CompressionFormat["UNCOMPRESSED"] = "UNCOMPRESSED";
    S3CompressionFormat["ZIP"] = "ZIP";
})(S3CompressionFormat = exports.S3CompressionFormat || (exports.S3CompressionFormat = {}));
class S3Destination extends delivery_stream_destination_1.DeliveryStreamDestination {
    constructor(bucket, options = {}) {
        super();
        this.bucket = bucket;
        this.buffering = options.buffering;
        this.cloudwatchLoggingConfiguration = options.cloudwatchLoggingConfiguration ?? new cloudwatch_logging_configuration_1.CloudWatchLoggingConfiguration({
            enabled: true,
        });
        this.compressionFormat = options.compressionFormat;
        this.encryptionEnabled = options.encryptionEnabled ?? !!options.encryptionKey;
        this.encryptionKey = options.encryptionKey;
        this.errorOutputPrefix = options.errorOutputPrefix;
        this.keyPrefix = options.keyPrefix;
        this._role = options.role;
    }
    // Private accessors
    get role() {
        return this._role;
    }
    bind(scope) {
        return {
            s3DestinationConfiguration: this.buildConfiguration(scope),
        };
    }
    buildConfiguration(scope) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const isCrossAccount = this.bucket.stack.account !== stack.account;
        this._role = this._role ?? scope.node.tryFindChild('s3-delivery-role') ?? new aws_iam_1.Role(scope, 's3-delivery-role', {
            assumedBy: new aws_iam_1.PrincipalWithConditions(new aws_iam_1.ServicePrincipal('firehose.amazonaws.com'), {
                StringEquals: {
                    'sts:ExternalId': aws_cdk_lib_1.Stack.of(scope).account,
                },
            }),
        });
        this._role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                's3:GetBucketLocation',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
            ],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [
                this.bucket.bucketArn,
            ],
        }));
        this._role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                's3:AbortMultipartUpload',
                's3:GetObject',
                's3:PutObject',
                ...(isCrossAccount ? ['s3:PutObjectAcl'] : []),
            ],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [
                this.bucket.arnForObjects('*'),
            ],
        }));
        if (this.encryptionKey && this.encryptionEnabled) {
            this._role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                actions: [
                    'kms:Decrypt',
                    'kms:GenerateDataKey',
                ],
                conditions: {
                    StringEquals: {
                        'kms:ViaService': `s3.${aws_cdk_lib_1.Stack.of(scope).region}.amazonaws.com`,
                    },
                    StringLike: {
                        'kms:EncryptionContext:aws:s3:arn': [
                            this.bucket.arnForObjects(`${this.keyPrefix ?? ''}*`),
                            ...(this.errorOutputPrefix ? this.bucket.arnForObjects(`${this.errorOutputPrefix ?? ''}*`) : []),
                        ],
                    },
                },
                effect: aws_iam_1.Effect.ALLOW,
                resources: [
                    this.encryptionKey.keyArn,
                ],
            }));
        }
        // Configure cross account bucket access
        if (!aws_cdk_lib_1.Token.isUnresolved(stack.account) && isCrossAccount) {
            this.bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: [
                    's3:GetBucketLocation',
                    's3:ListBucket',
                    's3:ListBucketMultipartUploads',
                ],
                effect: aws_iam_1.Effect.ALLOW,
                principals: [
                    new aws_iam_1.AccountPrincipal(stack.account),
                ],
                resources: [
                    stack.resolve(this.bucket.bucketArn),
                ],
            }));
            this.bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: [
                    's3:AbortMultipartUpload',
                    's3:GetObject',
                    's3:PutObject',
                    's3:PutObjectAcl',
                ],
                effect: aws_iam_1.Effect.ALLOW,
                principals: [
                    new aws_iam_1.AccountPrincipal(stack.account),
                ],
                resources: [
                    this.bucket.arnForObjects('*'),
                ],
            }));
        }
        if (this.cloudwatchLoggingConfiguration?.enabled) {
            const logGroupScope = this.cloudwatchLoggingConfiguration?.logGroup?.logGroupName ?? '*';
            const logStreamScope = this.cloudwatchLoggingConfiguration?.logStream?.logStreamName ?? '*';
            this._role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                actions: [
                    'logs:PutLogEvents',
                ],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [
                    aws_cdk_lib_1.Stack.of(this.cloudwatchLoggingConfiguration?.logGroup ?? scope).formatArn({
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resource: 'log-group',
                        resourceName: `${logGroupScope}:log-stream:${logStreamScope}`,
                        service: 'logs',
                    }),
                ],
            }));
        }
        return {
            bucketArn: this.bucket.bucketArn,
            bufferingHints: this.buffering?.bind(scope),
            cloudWatchLoggingOptions: this.cloudwatchLoggingConfiguration?.bind(scope),
            compressionFormat: this.compressionFormat,
            encryptionConfiguration: formatting_1.definedFieldsOrUndefined({
                kmsEncryptionConfig: this.encryptionKey === undefined ? undefined : formatting_1.definedFieldsOrUndefined({
                    awskmsKeyArn: this.encryptionKey.keyArn,
                }),
                noEncryptionConfig: (!this.encryptionEnabled && this.encryptionKey) ? 'NoEncryption' : undefined,
            }),
            errorOutputPrefix: this.errorOutputPrefix,
            prefix: this.keyPrefix,
            roleArn: this._role.roleArn,
        };
    }
    renderBackupConfiguration(scope, enabled) {
        const mode = enabled ? 'Enabled' : 'Disabled';
        return {
            s3BackupConfiguration: this.buildConfiguration(scope),
            s3BackupMode: enabled === undefined ? undefined : mode,
        };
    }
}
exports.S3Destination = S3Destination;
_a = JSII_RTTI_SYMBOL_1;
S3Destination[_a] = { fqn: "cdk-extensions.kinesis_firehose.S3Destination", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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