"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceShare = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ram_1 = require("aws-cdk-lib/aws-ram");
const shared_principal_1 = require("./lib/shared-principal");
/**
 * Creates a resource share that can used to share AWS resources with other AWS
 * accounts, organizations, or organizational units (OU's).
 *
 * @see [AWS::RAM::ResourceShare](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ram-resourceshare.html)
 */
class ResourceShare extends aws_cdk_lib_1.Resource {
    /**
       * Creates a new instance of the ResourceShare class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        // Internal properties
        this._autoDiscovery = false;
        this._principals = [];
        this._resources = [];
        this.allowExternalPrincipals = props.allowExternalPrincipals;
        this.name = props.name ?? aws_cdk_lib_1.Names.uniqueId(this);
        this.resource = new aws_ram_1.CfnResourceShare(this, 'Resource', {
            allowExternalPrincipals: this.allowExternalPrincipals,
            name: this.name,
            principals: aws_cdk_lib_1.Lazy.uncachedList({
                produce: () => {
                    return this._principals.map((x) => {
                        return x.bind(this);
                    });
                },
            }),
            resourceArns: aws_cdk_lib_1.Lazy.uncachedList({
                produce: () => {
                    return this._resources.map((x) => {
                        return x.bind(this);
                    });
                },
            }),
        });
    }
    /**
     *
     */
    get autoDiscovery() {
        return this._autoDiscovery;
    }
    addPrincipal(principal) {
        this._principals.push(principal);
    }
    addResource(resource) {
        this._resources.push(resource);
    }
    autoDiscover() {
        const accounts = new Set(this.node.root.node.findAll().reduce((prev, cur) => {
            if (cur instanceof aws_cdk_lib_1.Stage && cur.account && !aws_cdk_lib_1.Token.isUnresolved(cur.account)) {
                prev.push(cur.account);
            }
            else if (cur instanceof aws_cdk_lib_1.Stack && !aws_cdk_lib_1.Token.isUnresolved(cur.account)) {
                prev.push(cur.account);
            }
            return prev;
        }, []).filter((x) => {
            return x !== this.stack.account;
        }));
        accounts.forEach((x) => {
            this.addPrincipal(shared_principal_1.SharedPrincipal.fromAccountId(x));
        });
    }
    enableAutoDiscovery() {
        if (!this.autoDiscovery) {
            aws_cdk_lib_1.Aspects.of(this).add({
                visit: (node) => {
                    if (node === this) {
                        this.autoDiscover();
                    }
                },
            });
            this._autoDiscovery = true;
        }
    }
}
exports.ResourceShare = ResourceShare;
_a = JSII_RTTI_SYMBOL_1;
ResourceShare[_a] = { fqn: "cdk-extensions.ram.ResourceShare", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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