"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InboundResolver = void 0;
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const resolver_endpoint_base_1 = require("./resolver-endpoint-base");
class InboundResolver extends resolver_endpoint_base_1.ResolverEndpointBase {
    /**
       * Creates a new instance of the InboundResolver class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            direction: InboundResolver.DIRECTION,
            subnetOffset: InboundResolver.SUBNET_OFFSET,
        });
        this.inboundCidrs = props.inboundCidrs ?? InboundResolver.DEFAULT_INBOUND_CIDRS;
        for (let cidr of this.inboundCidrs) {
            this.securityGroup.addIngressRule(aws_ec2_1.Peer.ipv4(cidr), aws_ec2_1.Port.tcp(53), `TCP DNS from ${cidr}`);
            this.securityGroup.addIngressRule(aws_ec2_1.Peer.ipv4(cidr), aws_ec2_1.Port.udp(53), `UDP DNS from ${cidr}`);
        }
    }
}
exports.InboundResolver = InboundResolver;
// Static properties
/**
 * By default allows outbound to all RFC1918 ranges.
 */
InboundResolver.DEFAULT_INBOUND_CIDRS = [
    '10.0.0.0/8',
    '172.16.0.0/12',
    '192.168.0.0/16',
];
InboundResolver.DIRECTION = 'INBOUND';
InboundResolver.SUBNET_OFFSET = 4;
//# sourceMappingURL=data:application/json;base64,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