"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OutboundResolver = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ram_1 = require("aws-cdk-lib/aws-ram");
const forward_resolver_rule_1 = require("./forward-resolver-rule");
const resolver_endpoint_base_1 = require("./resolver-endpoint-base");
class OutboundResolver extends resolver_endpoint_base_1.ResolverEndpointBase {
    /**
       * Creates a new instance of the OutboundResolver class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            direction: OutboundResolver.DIRECTION,
            subnetOffset: OutboundResolver.SUBNET_OFFSET,
        });
        // Resource properties
        this.resolverRules = [];
        this.organizationArn = props.organizationArn;
        this.outboundCidrs = props.outboundCidrs ?? OutboundResolver.DEFAULT_OUTBOUND_CIDRS;
        for (let cidr of this.outboundCidrs) {
            this.securityGroup.addEgressRule(aws_ec2_1.Peer.ipv4(cidr), aws_ec2_1.Port.tcp(53), `TCP DNS to ${cidr}`);
            this.securityGroup.addEgressRule(aws_ec2_1.Peer.ipv4(cidr), aws_ec2_1.Port.udp(53), `UDP DNS to ${cidr}`);
        }
    }
    /**
       * Adds a new Resolver Rule for a given domain to the Outbound Resolver. Also adds the created
       * rule to a RAM Resource Share if an organization ARN was specified when the Outbound Resolver
       * was created.
       *
       * @param domain The domain you would like to add the rule for.
       * @param targets The IP addresses of the external resolver that should be used to resolve the
       * domain.
       */
    addRule(domain, targets) {
        // Create the resolver rule, forwarding requests for the domain (and all subdomains) to the
        // specified IP addressed (on port 53).
        const rule = new forward_resolver_rule_1.ForwardResolverRule(this, `resolver-rule-${domain}`, {
            domainName: domain,
            resolverEndpoint: this,
            targetIps: targets,
        });
        this.resolverRules.push(rule);
        // If this is the first rule that was added and an organization ARN was given, create a Resource
        // Share to share the created rules with the organization. The list of rules in the resource share is
        // Lazy evaluated so any subsequent rules that get added will be automatically included without
        // and further action needing to be taken.
        if (this.organizationArn && !this.resourceShare) {
            this.resourceShare = new aws_ram_1.CfnResourceShare(this, 'resource-share', {
                allowExternalPrincipals: false,
                name: aws_cdk_lib_1.Lazy.uncachedString({
                    produce: (_) => {
                        return aws_cdk_lib_1.Names.uniqueId(this.resourceShare);
                    },
                }),
                principals: [
                    this.stack.formatArn(this.organizationArn),
                ],
                resourceArns: aws_cdk_lib_1.Lazy.uncachedList({
                    produce: () => {
                        return this.resolverRules.map((x) => {
                            return x.resolverRuleArn;
                        });
                    },
                }),
            });
        }
        return rule;
    }
}
exports.OutboundResolver = OutboundResolver;
// Static properties
/**
 * By default allows outbound to all RFC1918 ranges.
 */
OutboundResolver.DEFAULT_OUTBOUND_CIDRS = [
    '10.0.0.0/8',
    '172.16.0.0/12',
    '192.168.0.0/16',
];
OutboundResolver.DIRECTION = 'OUTBOUND';
OutboundResolver.SUBNET_OFFSET = 5;
//# sourceMappingURL=data:application/json;base64,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