"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResolverEndpointBase = exports.DemoAccessibility = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_route53resolver_1 = require("aws-cdk-lib/aws-route53resolver");
const networking_1 = require("../utils/networking");
/**
 * The configuration that determines the function of the Route 53 being created
 */
var DemoAccessibility;
(function (DemoAccessibility) {
    /**
       * Create an inbound resolver. The inbound resolver allows connections from outside of the VPC to
       * resolve DNS queries using the VPC's internal resolver.
       */
    DemoAccessibility["INBOUND"] = "INBOUND";
    /**
       * Create an outbound resolver. The outbound resolver allows the VPC's internal resolver to make
       * DNS queries to external DNS endpoints based on domain.
       */
    DemoAccessibility["OUTBOUND"] = "OUTBOUND";
})(DemoAccessibility = exports.DemoAccessibility || (exports.DemoAccessibility = {}));
class ResolverEndpointBase extends aws_cdk_lib_1.Resource {
    /**
       * Creates a new instance of the ResolverEndpointBase class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.direction = props.direction;
        this.subnetOffset = props.subnetOffset;
        this.vpc = props.vpc;
        const fmtDirection = this.direction.charAt(0).toUpperCase() + this.direction.slice(1).toLowerCase();
        this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'security-group', {
            allowAllOutbound: false,
            description: `${fmtDirection} resolver security group for ${this.vpc.vpcId}`,
            vpc: this.vpc,
        });
        this.resolverEndpoint = new aws_route53resolver_1.CfnResolverEndpoint(this, 'Resource', {
            direction: props.direction,
            ipAddresses: this.vpc.selectSubnets(props.subnets).subnets.map((x) => {
                return {
                    ip: this.offsetSubnet(x.ipv4CidrBlock),
                    subnetId: x.subnetId,
                };
            }),
            name: `${this.direction.toLowerCase()}-resolver`,
            securityGroupIds: [
                this.securityGroup.securityGroupId,
            ],
        });
        // If IP's were explicitly specified we will make them available
        const resolverIps = this.resolverEndpoint.ipAddresses.map((x) => {
            return x.ip;
        });
        this.resolverEndpointIps = resolverIps.every((x) => x) ? resolverIps : undefined;
        // Expose CloudFormation tokens so they'll be available to other resources
        this.resolverEndpointArn = this.resolverEndpoint.attrArn;
        this.resolverEndpointDirection = this.resolverEndpoint.attrDirection;
        this.resolverEndpointHostVpcId = this.resolverEndpoint.attrHostVpcId;
        this.resolverEndpointId = this.resolverEndpoint.attrResolverEndpointId;
        this.resolverEndpointIpAddressCount = this.resolverEndpoint.attrIpAddressCount;
        this.resolverEndpointName = this.resolverEndpoint.attrName;
    }
    /**
       * Gets an IP address offset by a specific amount from the start of a CIDR range. The amount it is
       * offset is determined by the {@link ResolverEndpointBaseProps.subnetOffset | subnet offsetSubnet}
       * property.
       *
       * @param cidr The CIDR range you want to calculate the offset for.
       * @returns An IP address offset by the configured amount.
       */
    offsetSubnet(cidr) {
        if (this.subnetOffset) {
            let ip = cidr.split('/')[0];
            return networking_1.num2dot(networking_1.dot2num(ip) + this.subnetOffset);
        }
        else {
            return undefined;
        }
    }
}
exports.ResolverEndpointBase = ResolverEndpointBase;
//# sourceMappingURL=data:application/json;base64,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