"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResolverRule = exports.ResolverRuleType = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_route53resolver_1 = require("aws-cdk-lib/aws-route53resolver");
const resolver_rule_association_1 = require("./resolver-rule-association");
/**
 * @see [AWS::Route53Resolver::ResolverRule](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-route53resolver-resolverrule.html#cfn-route53resolver-resolverrule-ruletype)
 */
var ResolverRuleType;
(function (ResolverRuleType) {
    /**
     * When you want to forward DNS queries for specified domain name to resolvers on your network, specify FORWARD.
     */
    ResolverRuleType["FORWARD"] = "FORWARD";
    /**
     * Currently, only the Resolver can create rules that have a value of RECURSIVE for RuleType.
     */
    ResolverRuleType["RECURSIVE"] = "RECURSIVE";
    /**
     * When you have a forwarding rule to forward DNS queries for a domain to your network and you want Resolver to process queries for a subdomain of that domain, specify SYSTEM.
     */
    ResolverRuleType["SYSTEM"] = "SYSTEM";
})(ResolverRuleType = exports.ResolverRuleType || (exports.ResolverRuleType = {}));
class ResolverRuleBase extends aws_cdk_lib_1.Resource {
    /**
       * Creates an association between a VPC and a resolver rule.
       *
       * A resolver rule that isn't associated with any VPC's will have no
       * effect, even in the VPC containing the rule's outbound resolver.
       *
       * VPC associations don't need to be in the same account as
       * the resolver rule for rules shared with [AWS RAM](https://docs.aws.amazon.com/ram/latest/userguide/what-is.html).
       *
       * @param vpc The VPC you want to create the association for.
       * @param options Additional configuration for the association.
       * @returns The association that was created.
       */
    associateVpc(vpc, options) {
        return new resolver_rule_association_1.ResolverRuleAssociation(this, `association-${vpc.node.addr}`, {
            name: options?.name,
            resolverRule: this,
            vpc: vpc,
        });
    }
}
class ResolverRule extends ResolverRuleBase {
    /**
       * Creates a new instance of the ResolverRule class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in the construct tree.
       * @param id A name to be associated with the stack and used in resource naming. Must be unique
       * within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope, id, props) {
        super(scope, id, props);
        // Internal properties
        this._targetIps = [];
        this.domainName = props.domainName;
        this.resolverEndpoint = props.resolverEndpoint;
        this.ruleName = props.ruleName;
        this.ruleType = props.ruleType;
        if (this.ruleType === ResolverRuleType.RECURSIVE) {
            aws_cdk_lib_1.Annotations.of(this).addError("Currently, only Resolver can create rules that have a value of 'RECURSIVE' for 'RuleType'.");
        }
        this.resource = new aws_route53resolver_1.CfnResolverRule(this, 'Resource', {
            domainName: this.domainName,
            ruleType: this.ruleType,
            name: this.ruleName,
            resolverEndpointId: this.resolverEndpoint?.resolverEndpointId,
            targetIps: aws_cdk_lib_1.Lazy.any({
                produce: () => {
                    return this._targetIps.map((x) => {
                        return {
                            ip: x.address,
                            port: x.port?.toString(),
                        };
                    });
                },
            }, {
                omitEmptyArray: true,
            }),
        });
        this.resolverRuleArn = this.resource.attrArn;
        this.resolverRuleDomainName = this.resource.attrDomainName;
        this.resolverRuleEndpointId = this.resource.attrResolverEndpointId;
        this.resolverRuleId = this.resource.attrResolverRuleId;
        this.resolverRuleName = this.resource.attrName;
        this.resolverRuleTargetIps = this.resource.attrTargetIps;
        props.targetIps?.map((x) => {
            this.addTargetIp(x);
        });
    }
    static fromResolverRuleId(scope, id, resolverRuleId) {
        class Import extends ResolverRuleBase {
            constructor() {
                super(...arguments);
                this.resolverRuleArn = aws_cdk_lib_1.Stack.of(scope).formatArn({
                    resource: 'resolver-rule',
                    resourceName: resolverRuleId,
                    service: 'route53resolver',
                });
                this.resolverRuleDomainName = aws_cdk_lib_1.Lazy.string({
                    produce: () => {
                        throw new Error('Cannot get domain name for resolver rules imported using only an ID.');
                    },
                });
                this.resolverRuleEndpointId = aws_cdk_lib_1.Lazy.string({
                    produce: () => {
                        throw new Error('Cannot get endpoint ID for resolver rules imported using only an ID.');
                    },
                });
                this.resolverRuleId = resolverRuleId;
                this.resolverRuleName = aws_cdk_lib_1.Lazy.string({
                    produce: () => {
                        throw new Error('Cannot get name for resolver rules imported using only an ID.');
                    },
                });
                this.resolverRuleTargetIps = aws_cdk_lib_1.Lazy.any({
                    produce: () => {
                        throw new Error("Cannot get target IP's for resolver rules imported using only an ID.");
                    },
                });
            }
        }
        return new Import(scope, id);
    }
    addTargetIp(target) {
        const port = target.port ?? 53;
        if ((port < 0) || (port > 65535)) {
            aws_cdk_lib_1.Annotations.of(this).addError(`Specified resolver target port (${port}) is  invalid (must be between 0 and 65535).`);
        }
        if (this.ruleType === ResolverRuleType.SYSTEM) {
            aws_cdk_lib_1.Annotations.of(this).addError("Cannot add target IP's to rules of type 'SYSTEM'.");
        }
        this._targetIps.push(target);
        return this;
    }
}
exports.ResolverRule = ResolverRule;
//# sourceMappingURL=data:application/json;base64,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