"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessControlAttribute = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * Represents and ABAC attribute in IAM Identity Center.
 *
 * These are IAM Identity Center identity store attributes that you can
 * configure for use in attributes-based access control (ABAC). You can create
 * permissions policies that determine who can access your AWS resources based
 * upon the configured attribute values. When you enable ABAC and specify
 * `AccessControlAttributes`, IAM Identity Center passes the attribute values
 * of the authenticated user into IAM for use in policy evaluation.
 */
class AccessControlAttribute {
    /**
       * Creates a new instance of the AccessControlAttribute class.
       *
       * @param options The configuration settings to use when configuring the
       * attribute.
       */
    constructor(options) {
        // Internal properties
        this._sources = [];
        this.name = options.name;
    }
    // Standard accessors
    /**
       * A list of identity sources to use when mapping a specified attribute to
       * IAM Identity Center. Note that the array is readonly and changes made
       * to it will not be reflected when generating ABAC attribute
       * configuration. To add a source to the attribute use the {@link addSource}
       * method.
       */
    get sources() {
        return [...this._sources];
    }
    /**
       * Adds an identity source to use when mapping the attribute to IAM
       * Identity Center.
       *
       * @param source The source to add.
       * @returns The ABAC attribute the source was associated with.
       */
    addSource(source) {
        this._sources.push(source);
        return this;
    }
    /**
       * Generates the raw CloudFormation configuration that this attribute
       * represents within the context of a given scope.
       *
       * @param scope The construct managing the access control attribute
       * configuration that will consume details of this attribute.
       * @returns The raw CloudFormation configuration that this attribute
       * represents.
       */
    bind(scope) {
        return {
            key: this.name,
            value: {
                source: aws_cdk_lib_1.Lazy.list({
                    produce: () => {
                        if (this._sources.length === 0) {
                            aws_cdk_lib_1.Annotations.of(scope).addError([
                                `AccessControlAttribute for ${this.name} has no sources.`,
                                'At least one source is required.',
                            ].join(' '));
                        }
                        return this._sources;
                    },
                }),
            },
        };
    }
}
exports.AccessControlAttribute = AccessControlAttribute;
_a = JSII_RTTI_SYMBOL_1;
AccessControlAttribute[_a] = { fqn: "cdk-extensions.sso.AccessControlAttribute", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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