import { Duration, Resource, ResourceProps } from 'aws-cdk-lib';
import { AddToPrincipalPolicyResult, IManagedPolicy, PolicyDocument, PolicyStatement } from 'aws-cdk-lib/aws-iam';
import { CfnPermissionSet } from 'aws-cdk-lib/aws-sso';
import { Construct } from 'constructs';
import { IInstance } from './instance';
import { PermissionsBoundary } from './lib/permissions-boundary';
import { ReferencedManagedPolicy, ReferenceOptions } from './lib/referenced-managed-policy';
/**
 * Configuration for PermissionSet resource.
 */
export interface PermissionSetProps extends ResourceProps {
    /**
       * A user friendly description providing details about the permission set.
       *
       * @see [AWS::SSO::PermissionSet](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-sso-permissionset.html#cfn-sso-permissionset-description)
       */
    readonly description?: string;
    /**
       * Adds inline policy documents that will be embedded in the permission
       * set.
       *
       * @see [AWS::SSO::PermissionSet](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-sso-permissionset.html#cfn-sso-permissionset-inlinepolicy)
       */
    readonly inlinePolicies?: {
        [key: string]: PolicyDocument;
    };
    /**
       * The ARN of the IAM Identity Center instance under which the operation
       * will be executed.
       */
    readonly instance: IInstance;
    /**
       * A list of the IAM managed policies that you want to attach to the
       * permission set. Managed policies specified here must be AWS managed.
       * To reference custom managed policies use the {@link PermissionSet.addCustomerManagedPolicy}
       * method.
       *
       * @see [AWS::SSO::PermissionSet](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-sso-permissionset.html#cfn-sso-permissionset-managedpolicies)
       */
    readonly managedPolicies?: IManagedPolicy[];
    /**
       * The name of the permission set.
       *
       * @see [AWS::SSO::PermissionSet](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-sso-permissionset.html#cfn-sso-permissionset-name)
       */
    readonly name?: string;
    /**
       * Specifies the configuration of the AWS managed or customer managed
       * policy that you want to set as a permissions boundary. Specify either
       * CustomerManagedPolicyReference to use the name and path of a customer
       * managed policy, or ManagedPolicyArn to use the ARN of an AWS managed
       * policy. A permissions boundary represents the maximum permissions that
       * any policy can grant your role. For more information, see [Permissions
       * boundaries](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_boundaries.html) for IAM entities in the AWS Identity and Access Management
       * User Guide.
       *
       * @see [AWS::SSO::PermissionSet](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-sso-permissionset.html#cfn-sso-permissionset-permissionsboundary)
       */
    readonly permissionsBoundary?: PermissionsBoundary;
    /**
       * Used to redirect users within the application during the federation
       * authentication process. For example, you can redirect users to a
       * specific page that is most applicable to their job after singing in to
       * an AWS account.
       *
       * @see [AWS::SSO::PermissionSet](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-sso-permissionset.html#cfn-sso-permissionset-relaystatetype)
       */
    readonly relayState?: string;
    /**
       * The length of time that the application user sessions are valid for.
       *
       * @see [AWS::SSO::PermissionSet](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-sso-permissionset.html#cfn-sso-permissionset-sessionduration)
       */
    readonly sessionDuration?: Duration;
}
/**
 * Represents an IAM Identity Center permission set resource.
 */
export interface IPermissionSet {
    readonly permissionSetArn: string;
}
declare abstract class PermissionSetBase extends Resource implements IPermissionSet {
    abstract readonly permissionSetArn: string;
    constructor(scope: Construct, id: string, props?: ResourceProps);
}
/**
 * Specifies a permission set within a specified IAM Identity Center instance.
 */
export declare class PermissionSet extends PermissionSetBase {
    static fromArn(scope: Construct, id: string, arn: string): IPermissionSet;
    private readonly _customerManagedPolicies;
    private readonly _inlinePolicies;
    private readonly _managedPolicies;
    /**
       * A user friendly description providing details about the permission set.
       */
    readonly description?: string;
    /**
       * The ARN of the IAM Identity Center instance under which the operation
       * will be executed.
       */
    readonly instance: IInstance;
    /**
       * The name of the permission set.
       */
    readonly name?: string;
    /**
       * Specifies the configuration of the AWS managed or customer managed
       * policy that you want to set as a permissions boundary. Specify either
       * CustomerManagedPolicyReference to use the name and path of a customer
       * managed policy, or ManagedPolicyArn to use the ARN of an AWS managed
       * policy. A permissions boundary represents the maximum permissions that
       * any policy can grant your role. For more information, see [Permissions
       * boundaries](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_boundaries.html) for IAM entities in the AWS Identity and Access Management
       * User Guide.
       */
    readonly permissionsBoundary?: PermissionsBoundary;
    /**
       * Used to redirect users within the application during the federation
       * authentication process. For example, you can redirect users to a
       * specific page that is most applicable to their job after singing in to
       * an AWS account.
       */
    readonly relayState?: string;
    /**
       * The length of time that the application user sessions are valid for.
       */
    readonly sessionDuration?: Duration;
    /**
       * The underlying PermissionSet CloudFormation resource.
       */
    readonly resource: CfnPermissionSet;
    /**
       * The permission set ARN of the permission set, such as
       * `arn:aws:sso:::permissionSet/ins-instanceid/ps-permissionsetid`.
       */
    readonly permissionSetArn: string;
    /**
       * Creates a new instance of the PermissionSet class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in
       * the construct tree.
       * @param id A name to be associated with the stack and used in resource
       * naming. Must be unique within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope: Construct, id: string, props: PermissionSetProps);
    /**
       * Adds a custom managed policy to the permission set. When using customer
       * managed policies it is required that a managed policy with a matching
       * name and path exist in any AWS account for which the permission set
       * will be assigned.
       *
       * @param options The configuration for the customer managed policy.
       * @returns A dynamically generated ManagedPolicy class that can be used
       * to create compatible managed policies in other accounts.
       */
    addCustomerManagedPolicy(options: ReferenceOptions): typeof ReferencedManagedPolicy;
    /**
       * Adds a new Managed Policy to the permission set. Only Managed Policies
       * created and maintained by AWS are supported. To add a custom Managed
       * Policy that you control use the {@link addCustomerManagedPolicy} method.
       *
       * @param policy The AWS Managed Policy to associate with the Permission
       * Set.
       * @returns The Permission Set resource the Managed Policy was added to.
       */
    addManagedPolicy(policy: IManagedPolicy): PermissionSet;
    /**
       * Adds a permission to the permission set's default policy document.
       *
       * If there is no default policy attached to this permission set, it will be created.
       *
       * @param statement The permission statement to add to the policy document.
       * @returns An [AddToPrincipalPolicyResult](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_iam.AddToPrincipalPolicyResult.html) object that provides details of
       * the result of the operation.
       */
    addToPrincipalPolicy(statement: PolicyStatement): AddToPrincipalPolicyResult;
}
export {};
