"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PermissionSet = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_sso_1 = require("aws-cdk-lib/aws-sso");
const referenced_managed_policy_1 = require("./lib/referenced-managed-policy");
class PermissionSetBase extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
    }
}
/**
 * Specifies a permission set within a specified IAM Identity Center instance.
 */
class PermissionSet extends PermissionSetBase {
    /**
       * Creates a new instance of the PermissionSet class.
       *
       * @param scope A CDK Construct that will serve as this stack's parent in
       * the construct tree.
       * @param id A name to be associated with the stack and used in resource
       * naming. Must be unique within the context of 'scope'.
       * @param props Arguments related to the configuration of the resource.
       */
    constructor(scope, id, props) {
        super(scope, id, props);
        // Internal properties
        this._customerManagedPolicies = {};
        this._inlinePolicies = {};
        this._managedPolicies = [];
        this.description = props.description;
        this.instance = props.instance;
        this.name = props.name ?? aws_cdk_lib_1.Names.uniqueId(this);
        this.permissionsBoundary = props.permissionsBoundary;
        this.relayState = props.relayState;
        this.sessionDuration = props.sessionDuration;
        this._inlinePolicies = props.inlinePolicies ?? {};
        this.resource = new aws_sso_1.CfnPermissionSet(this, 'Resource', {
            customerManagedPolicyReferences: aws_cdk_lib_1.Lazy.any({
                produce: () => {
                    return this.stack.toJsonString(Object.keys(this._customerManagedPolicies).map((x) => {
                        const item = this._customerManagedPolicies[x];
                        return {
                            name: item.policyName,
                            path: item.policyPath,
                        };
                    }));
                },
            }, {
                omitEmptyArray: true,
            }),
            description: this.description,
            instanceArn: this.instance.instanceArn,
            managedPolicies: aws_cdk_lib_1.Lazy.list({
                produce: () => {
                    return this._managedPolicies.map((x) => {
                        return x.managedPolicyArn;
                    });
                },
            }, {
                omitEmpty: true,
            }),
            inlinePolicy: aws_cdk_lib_1.Lazy.string({
                produce: () => {
                    if (Object.keys(this._inlinePolicies).length === 0) {
                        return undefined;
                    }
                    return this.stack.toJsonString(Object.keys(this._inlinePolicies).map((x) => {
                        return {
                            PolicyDocument: this._inlinePolicies[x].toJSON(),
                            PolicyName: x,
                        };
                    }));
                },
            }),
            name: this.name,
            permissionsBoundary: this.permissionsBoundary?.bind(this),
            relayStateType: this.relayState,
            sessionDuration: this.sessionDuration ? this.sessionDuration.toIsoString() : undefined,
        });
        this.permissionSetArn = this.resource.attrPermissionSetArn;
        props.managedPolicies?.forEach((x) => {
            this.addManagedPolicy(x);
        });
    }
    static fromArn(scope, id, arn) {
        class Import extends PermissionSetBase {
            constructor() {
                super(scope, id);
                this.permissionSetArn = arn;
            }
        }
        return new Import();
    }
    /**
       * Adds a custom managed policy to the permission set. When using customer
       * managed policies it is required that a managed policy with a matching
       * name and path exist in any AWS account for which the permission set
       * will be assigned.
       *
       * @param options The configuration for the customer managed policy.
       * @returns A dynamically generated ManagedPolicy class that can be used
       * to create compatible managed policies in other accounts.
       */
    addCustomerManagedPolicy(options) {
        const fullPath = `${options.path ?? ''}/${options.name}`;
        const result = this._customerManagedPolicies[fullPath] ?? referenced_managed_policy_1.ReferencedManagedPolicy.of(options);
        this._customerManagedPolicies[fullPath] = result;
        return result;
    }
    /**
       * Adds a new Managed Policy to the permission set. Only Managed Policies
       * created and maintained by AWS are supported. To add a custom Managed
       * Policy that you control use the {@link addCustomerManagedPolicy} method.
       *
       * @param policy The AWS Managed Policy to associate with the Permission
       * Set.
       * @returns The Permission Set resource the Managed Policy was added to.
       */
    addManagedPolicy(policy) {
        this._managedPolicies.push(policy);
        return this;
    }
    /**
       * Adds a permission to the permission set's default policy document.
       *
       * If there is no default policy attached to this permission set, it will be created.
       *
       * @param statement The permission statement to add to the policy document.
       * @returns An [AddToPrincipalPolicyResult](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_iam.AddToPrincipalPolicyResult.html) object that provides details of
       * the result of the operation.
       */
    addToPrincipalPolicy(statement) {
        const defaultPolicyName = `default-policy-${this.node.addr}`;
        const document = this._inlinePolicies[defaultPolicyName] ?? new aws_iam_1.PolicyDocument();
        this._inlinePolicies[defaultPolicyName] = document;
        document.addStatements(statement);
        return {
            statementAdded: true,
            policyDependable: document,
        };
    }
}
exports.PermissionSet = PermissionSet;
_a = JSII_RTTI_SYMBOL_1;
PermissionSet[_a] = { fqn: "cdk-extensions.sso.PermissionSet", version: "0.0.38" };
//# sourceMappingURL=data:application/json;base64,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