# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['botocove']

package_data = \
{'': ['*']}

install_requires = \
['boto3>=1.16.12,<2.0.0']

setup_kwargs = {
    'name': 'botocove',
    'version': '1.2.0',
    'description': 'A decorator to allow running a function against all AWS accounts in an organization',
    'long_description': '# Botocove\n\nRun a function against a selection of AWS accounts, or all AWS accounts in an\norganization, concurrently. By default, opinionated to work with the standard\nAWS Organization master/member configuration from an organization master\naccount.\n\n- Fast\n- Easy\n- Dolphin Themed 🐬\n\nA simple decorator for functions to remove time and complexity burden. Uses\n`async.io` and `ThreadPoolExecutor` to run boto3 sessions against one to all\nof your AWS accounts at (nearly!) the same speed as running against one.\n\nWrap a function in `@cove` and inject an assumed role session into every account\nrequired, gathering all results into a dictionary.\n\n**Warning**: this tool gives you the potential to make dangerous changes\nat scale. **Test carefully and make idempotent changes**! Please read available\narguments to understand safe experimentation with this package.\n\n## Requirements\n\nAn IAM user with `sts:assumerole` privilege, and accounts that have a trust\nrelationship to the IAM user\'s account.\n\nBy default, the IAM user is expected to be in an AWS Organization Master account\n\nDefault (customisable if unsuitable) expectations are:\n\nIn the organization master account:\n* IAM permissions `sts:assumerole`, `sts:get-caller-identity` and\n`organizations:list-accounts`\n\nIn the organization accounts:\n* An `AccountOrganizationAccessRole` role\n\nSee the Arguments section for how to change these defaults to work with any\naccounts.\n\n## Quickstart\nA function written to interact with a `boto3 session` can now be called with\na `session` from every account required in your AWS organization, assuming\na role in each account.\n\nFor example:\n\nA standard approach: this function takes a boto3 `session` and gets all IAM\nusers from a single AWS account\n\n```\nimport boto3\n\n\ndef get_iam_users(session):\n    iam = session.client("iam", region_name="eu-west-1")\n    all_users = iam.get_paginator("list_users").paginate().build_full_result()\n    return all_users\n\ndef main():\n    session = boto3.session.Session(profile_name="my_dev_account")\n    users = get_iam_users(session)\n    print(users) # A single account\'s IAM users\n```\n\nNow with `@cove`: a session for every account in the organization is injected\nby the decorator. A safe example to run as a test!\n\n```\nimport boto3\nfrom botocove import cove\n\n@cove()\ndef get_iam_users(session):\n    iam = session.client("iam", region_name="eu-west-1")\n    all_users = iam.get_paginator("list_users").paginate().build_full_result()\n    return all_users\n\ndef main():\n    # No session passed as the decorator injects it\n    all_results = get_iam_users() \n    # Now returns a Dict with keys Results, Exceptions and FailedAssumeRole\n    \n    # A list of dictionaries for each account, with account details included.\n    # Each account\'s get_iam_users return is in a "Result" key.\n    print(all_results["Results"]) \n    \n    # A list of dictionaries for each account that raised an exception\n    print(all_results["Exceptions"])\n\n    # A list of dictionaries for each account that could not be assumed into\n    print(all_results["FailedAssumeRole"])\n```\n\n## Arguments\n\n### Cove\n`@cove()`: Uses boto3 credential chain to get every AWS account within the\norganization, assume the `OrganizationAccountAccessRole` in it and run the\nwrapped function with that role.\n\nEquivalent to:\n`@cove(target_ids=None, ignore_ids=None, rolename=None, assuming_session=None, \n    raise_exception=False, org_master=True)`\n\n`target_ids`: Optional[List[str]]\nA list of AWS accounts as strings to attempt to assume role in to. When unset,\ndefault attempts to use every available account ID in an AWS organization.\n\n`ignore_ids`: Optional[List[str]]\nA list of AWS account ID\'s that will not attempt assumption in to. Allows IDs to\nbe ignored.\n\n`rolename`: Optional[str]\nAn IAM role name that will be attempted to assume in all target accounts. \nDefaults to the AWS Organization default, `OrganizationAccountAccessRole`.\n\n`assuming_session`: Optional[Session]\nA Boto3 `Session` object that will be used to call `sts:assumerole`. If not\nprovided, defaults to standard boto3 credential chain.\n\n`raise_exception`: bool\nDefaults to False. Default behaviour is that exceptions are not raised from\ndecorated function. This is due to `cove` running asynchronously and preferring\nto resolve all tasks and report their results instead of exiting early.\n\n`raise_exception=True` will allow a full stack trace to escape on the first\nexception seen; but will not gracefully or consistently interrupt running tasks.\n It is vital to run interruptible, idempotent code with this argument as `True`.\n\n`org_master`: bool\nDefaults to True. When True, will leverage the Boto3 Organizations API to list\nall accounts in the organization, and enrich each `CoveSession` with information\navailable (`Id`, `Arn`, `Name`). \n\n`org_master=False` means `target_ids` must be provided (as no list of accounts\ncan be created for you), as well as likely `rolename`. Only `Id` will be\navailable to `CoveSession`.\n\n### CoveSession\n\nCove supplies an enriched Boto3 session to each function called. Account details\nare available with the `session_information` attribute.\n\nOtherwise, it functions exactly as calling `boto3` would.\n\n```\n@cove()\ndef do_nothing(session: CoveSession):\n    print(session.session_information) # Outputs a dict of known information\n    # This function runs no boto3-specific API, but does test that a role\n    # can be assumed\n```\n\n## Return values\n\nWrapped functions return a dictionary. Each value contains List[Dict[str, Any]]:\n```\n{\n    "Results": results: \n    "Exceptions": exceptions,\n    "FailedAssumeRole": invalid_sessions,\n}\n```\nAn example of cove_output["Results"]:\n```\n[ # A list of dictionaries per account called\n    {\n    \'Id\': \'123456789010\',\n    \'Email\': \'email@address.com\',\n    \'Name\': \'account-name\',\n    \'Status\': \'ACTIVE\',\n    \'AssumeRoleSuccess\': True,\n    \'Result\': wrapped_function_return_value # Result of wrapped func\n    } \n] \n```\n\n### botocove?\n\nIt turns out that the Amazon\'s Boto dolphins are solitary or small-group animals,\nunlike the large pods of dolphins in the oceans. This killed my "large group of \nboto" idea, so the next best idea was where might they all shelter together... a\ncove!',
    'author': 'Dave Connell',
    'author_email': 'daveconn41@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/connelldave/botocove',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
