"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SonarCodeScanner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const constructs_1 = require("constructs");
const sonarqube_commands_1 = require("./sonarqube-commands");
const unpackSourceAndArtifacts = () => [
    'export SYNTH_ARTIFACT_URI=`echo $SYNTH_ARTIFACT_LOCATION | awk \'{sub("arn:aws:s3:::","s3://")}1\' $1`',
    'export SYNTH_SOURCE_URI=`echo $SYNTH_SOURCE_VERSION | awk \'{sub("arn:aws:s3:::","s3://")}1\' $1`',
    "aws s3 cp $SYNTH_SOURCE_URI source.zip",
    "aws s3 cp $SYNTH_ARTIFACT_URI dist.zip",
    "unzip source.zip",
    "unzip dist.zip -d cdk.out",
    "rm source.zip dist.zip",
];
const owaspScan = () => "npx owasp-dependency-check --format HTML --out reports --exclude '**/node_modules/**/*' --exclude '**/reports/**/*' --exclude '**/cdk.out/**/*' --exclude '**/.env/**/*' --exclude '**/dist/**/*' --exclude '**/.git/**/*' --scan . --enableExperimental --bin /tmp/dep-check --disableRetireJS";
const cfnNagScan = (cfnNagIgnorePath) => `cfn_nag ${cfnNagIgnorePath !== null && cfnNagIgnorePath !== void 0 ? cfnNagIgnorePath : ""} cdk.out/**/*.template.json --output-format=json > reports/cfn-nag-report.json`;
class SonarCodeScanner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const sonarQubeToken = new aws_secretsmanager_1.Secret(this, "SonarQubeToken");
        const synthBuildProject = aws_codebuild_1.Project.fromProjectArn(this, "SynthBuildProject", props.synthBuildArn);
        const validationProject = new aws_codebuild_1.Project(this, "ValidationProject", {
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
            },
            environmentVariables: {
                SONARQUBE_TOKEN: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    value: sonarQubeToken.secretArn,
                },
                SONARQUBE_ENDPOINT: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeEndpoint,
                },
                PROJECT_NAME: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeProjectName,
                },
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: "0.2",
                env: {
                    shell: "bash",
                },
                phases: {
                    install: {
                        commands: ["gem install cfn-nag"],
                    },
                    build: {
                        commands: [
                            ...unpackSourceAndArtifacts(),
                            ...sonarqube_commands_1.createSonarqubeProject(props),
                            owaspScan(),
                            cfnNagScan(props.cfnNagIgnorePath),
                            sonarqube_commands_1.sonarqubeScanner(),
                            ...sonarqube_commands_1.generateSonarqubeReports(),
                            ...(props.preArchiveCommands || []),
                        ],
                    },
                },
            }),
        });
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["codebuild:BatchGetBuilds"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [synthBuildProject.projectArn],
        }));
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["s3:GetObject*"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [props.artifactBucketArn, `${props.artifactBucketArn}/**`],
        }));
        props.artifactBucketKeyArn &&
            validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ["kms:Decrypt", "kms:DescribeKey"],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [props.artifactBucketKeyArn],
            }));
        synthBuildProject.onBuildSucceeded("OnSynthSuccess", {
            target: new aws_events_targets_1.CodeBuildProject(validationProject, {
                event: aws_events_1.RuleTargetInput.fromObject({
                    environmentVariablesOverride: [
                        {
                            name: "SYNTH_BUILD_ID",
                            type: "PLAINTEXT",
                            value: aws_events_1.EventField.fromPath("$.detail.build-id"),
                        },
                        {
                            name: "SYNTH_ARTIFACT_LOCATION",
                            type: "PLAINTEXT",
                            value: aws_events_1.EventField.fromPath("$.detail.additional-information.artifact.location"),
                        },
                        {
                            name: "SYNTH_SOURCE_VERSION",
                            type: "PLAINTEXT",
                            value: aws_events_1.EventField.fromPath("$.detail.additional-information.source-version"),
                        },
                    ],
                }),
            }),
        });
    }
}
exports.SonarCodeScanner = SonarCodeScanner;
_a = JSII_RTTI_SYMBOL_1;
SonarCodeScanner[_a] = { fqn: "aws-prototyping-sdk.pdk_pipeline.SonarCodeScanner", version: "0.0.41" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic29uYXItY29kZS1zY2FubmVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL3Bka19waXBlbGluZS9jb2RlX3NjYW5uZXIvc29uYXItY29kZS1zY2FubmVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEscUVBQXFFO0FBQ3JFLHNDQUFzQztBQUV0Qyw2REFLbUM7QUFDbkMsdURBQXFFO0FBQ3JFLHVFQUFrRTtBQUNsRSxpREFBOEQ7QUFDOUQsdUVBQXdEO0FBQ3hELDJDQUF1QztBQUN2Qyw2REFJOEI7QUFtQjlCLE1BQU0sd0JBQXdCLEdBQUcsR0FBRyxFQUFFLENBQUM7SUFDckMsd0dBQXdHO0lBQ3hHLG1HQUFtRztJQUNuRyx3Q0FBd0M7SUFDeEMsd0NBQXdDO0lBQ3hDLGtCQUFrQjtJQUNsQiwyQkFBMkI7SUFDM0Isd0JBQXdCO0NBQ3pCLENBQUM7QUFFRixNQUFNLFNBQVMsR0FBRyxHQUFHLEVBQUUsQ0FDckIsaVNBQWlTLENBQUM7QUFFcFMsTUFBTSxVQUFVLEdBQUcsQ0FBQyxnQkFBeUIsRUFBRSxFQUFFLENBQy9DLFdBQ0UsZ0JBQWdCLGFBQWhCLGdCQUFnQixjQUFoQixnQkFBZ0IsR0FBSSxFQUN0QixnRkFBZ0YsQ0FBQztBQUVuRixNQUFhLGdCQUFpQixTQUFRLHNCQUFTO0lBQzdDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBNEI7UUFDcEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixNQUFNLGNBQWMsR0FBRyxJQUFJLDJCQUFNLENBQUMsSUFBSSxFQUFFLGdCQUFnQixDQUFDLENBQUM7UUFFMUQsTUFBTSxpQkFBaUIsR0FBRyx1QkFBTyxDQUFDLGNBQWMsQ0FDOUMsSUFBSSxFQUNKLG1CQUFtQixFQUNuQixLQUFLLENBQUMsYUFBYSxDQUNwQixDQUFDO1FBRUYsTUFBTSxpQkFBaUIsR0FBRyxJQUFJLHVCQUFPLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFO1lBQy9ELFdBQVcsRUFBRTtnQkFDWCxVQUFVLEVBQUUsK0JBQWUsQ0FBQyxZQUFZO2FBQ3pDO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLGVBQWUsRUFBRTtvQkFDZixJQUFJLEVBQUUsNENBQTRCLENBQUMsZUFBZTtvQkFDbEQsS0FBSyxFQUFFLGNBQWMsQ0FBQyxTQUFTO2lCQUNoQztnQkFDRCxrQkFBa0IsRUFBRTtvQkFDbEIsSUFBSSxFQUFFLDRDQUE0QixDQUFDLFNBQVM7b0JBQzVDLEtBQUssRUFBRSxLQUFLLENBQUMsaUJBQWlCO2lCQUMvQjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osSUFBSSxFQUFFLDRDQUE0QixDQUFDLFNBQVM7b0JBQzVDLEtBQUssRUFBRSxLQUFLLENBQUMsb0JBQW9CO2lCQUNsQzthQUNGO1lBQ0QsU0FBUyxFQUFFLHlCQUFTLENBQUMsVUFBVSxDQUFDO2dCQUM5QixPQUFPLEVBQUUsS0FBSztnQkFDZCxHQUFHLEVBQUU7b0JBQ0gsS0FBSyxFQUFFLE1BQU07aUJBQ2Q7Z0JBQ0QsTUFBTSxFQUFFO29CQUNOLE9BQU8sRUFBRTt3QkFDUCxRQUFRLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQztxQkFDbEM7b0JBQ0QsS0FBSyxFQUFFO3dCQUNMLFFBQVEsRUFBRTs0QkFDUixHQUFHLHdCQUF3QixFQUFFOzRCQUM3QixHQUFHLDJDQUFzQixDQUFDLEtBQUssQ0FBQzs0QkFDaEMsU0FBUyxFQUFFOzRCQUNYLFVBQVUsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUM7NEJBQ2xDLHFDQUFnQixFQUFFOzRCQUNsQixHQUFHLDZDQUF3QixFQUFFOzRCQUM3QixHQUFHLENBQUMsS0FBSyxDQUFDLGtCQUFrQixJQUFJLEVBQUUsQ0FBQzt5QkFDcEM7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBRUgsaUJBQWlCLENBQUMsZUFBZSxDQUMvQixJQUFJLHlCQUFlLENBQUM7WUFDbEIsT0FBTyxFQUFFLENBQUMsMEJBQTBCLENBQUM7WUFDckMsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSztZQUNwQixTQUFTLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxVQUFVLENBQUM7U0FDMUMsQ0FBQyxDQUNILENBQUM7UUFFRixpQkFBaUIsQ0FBQyxlQUFlLENBQy9CLElBQUkseUJBQWUsQ0FBQztZQUNsQixPQUFPLEVBQUUsQ0FBQyxlQUFlLENBQUM7WUFDMUIsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSztZQUNwQixTQUFTLEVBQUUsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLEVBQUUsR0FBRyxLQUFLLENBQUMsaUJBQWlCLEtBQUssQ0FBQztTQUN0RSxDQUFDLENBQ0gsQ0FBQztRQUVGLEtBQUssQ0FBQyxvQkFBb0I7WUFDeEIsaUJBQWlCLENBQUMsZUFBZSxDQUMvQixJQUFJLHlCQUFlLENBQUM7Z0JBQ2xCLE9BQU8sRUFBRSxDQUFDLGFBQWEsRUFBRSxpQkFBaUIsQ0FBQztnQkFDM0MsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSztnQkFDcEIsU0FBUyxFQUFFLENBQUMsS0FBSyxDQUFDLG9CQUFvQixDQUFDO2FBQ3hDLENBQUMsQ0FDSCxDQUFDO1FBRUosaUJBQWlCLENBQUMsZ0JBQWdCLENBQUMsZ0JBQWdCLEVBQUU7WUFDbkQsTUFBTSxFQUFFLElBQUkscUNBQWdCLENBQUMsaUJBQWlCLEVBQUU7Z0JBQzlDLEtBQUssRUFBRSw0QkFBZSxDQUFDLFVBQVUsQ0FBQztvQkFDaEMsNEJBQTRCLEVBQUU7d0JBQzVCOzRCQUNFLElBQUksRUFBRSxnQkFBZ0I7NEJBQ3RCLElBQUksRUFBRSxXQUFXOzRCQUNqQixLQUFLLEVBQUUsdUJBQVUsQ0FBQyxRQUFRLENBQUMsbUJBQW1CLENBQUM7eUJBQ2hEO3dCQUNEOzRCQUNFLElBQUksRUFBRSx5QkFBeUI7NEJBQy9CLElBQUksRUFBRSxXQUFXOzRCQUNqQixLQUFLLEVBQUUsdUJBQVUsQ0FBQyxRQUFRLENBQ3hCLG1EQUFtRCxDQUNwRDt5QkFDRjt3QkFDRDs0QkFDRSxJQUFJLEVBQUUsc0JBQXNCOzRCQUM1QixJQUFJLEVBQUUsV0FBVzs0QkFDakIsS0FBSyxFQUFFLHVCQUFVLENBQUMsUUFBUSxDQUN4QixnREFBZ0QsQ0FDakQ7eUJBQ0Y7cUJBQ0Y7aUJBQ0YsQ0FBQzthQUNILENBQUM7U0FDSCxDQUFDLENBQUM7SUFDTCxDQUFDOztBQTFHSCw0Q0EyR0MiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbi8vIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG5cbmltcG9ydCB7XG4gIEJ1aWxkRW52aXJvbm1lbnRWYXJpYWJsZVR5cGUsXG4gIEJ1aWxkU3BlYyxcbiAgTGludXhCdWlsZEltYWdlLFxuICBQcm9qZWN0LFxufSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWNvZGVidWlsZFwiO1xuaW1wb3J0IHsgRXZlbnRGaWVsZCwgUnVsZVRhcmdldElucHV0IH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1ldmVudHNcIjtcbmltcG9ydCB7IENvZGVCdWlsZFByb2plY3QgfSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWV2ZW50cy10YXJnZXRzXCI7XG5pbXBvcnQgeyBFZmZlY3QsIFBvbGljeVN0YXRlbWVudCB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtaWFtXCI7XG5pbXBvcnQgeyBTZWNyZXQgfSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLXNlY3JldHNtYW5hZ2VyXCI7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tIFwiY29uc3RydWN0c1wiO1xuaW1wb3J0IHtcbiAgY3JlYXRlU29uYXJxdWJlUHJvamVjdCxcbiAgZ2VuZXJhdGVTb25hcnF1YmVSZXBvcnRzLFxuICBzb25hcnF1YmVTY2FubmVyLFxufSBmcm9tIFwiLi9zb25hcnF1YmUtY29tbWFuZHNcIjtcblxuZXhwb3J0IGludGVyZmFjZSBTb25hckNvZGVTY2FubmVyQ29uZmlnIHtcbiAgcmVhZG9ubHkgY2ZuTmFnSWdub3JlUGF0aD86IHN0cmluZztcbiAgcmVhZG9ubHkgc29uYXJxdWJlRW5kcG9pbnQ6IHN0cmluZztcbiAgcmVhZG9ubHkgc29uYXJxdWJlRGVmYXVsdFByb2ZpbGVPckdhdGVOYW1lOiBzdHJpbmc7XG4gIHJlYWRvbmx5IHNvbmFycXViZVNwZWNpZmljUHJvZmlsZU9yR2F0ZU5hbWU/OiBzdHJpbmc7XG4gIHJlYWRvbmx5IHNvbmFycXViZUF1dGhvcml6ZWRHcm91cDogc3RyaW5nO1xuICByZWFkb25seSBzb25hcnF1YmVQcm9qZWN0TmFtZTogc3RyaW5nO1xuICByZWFkb25seSBzb25hcnF1YmVUYWdzPzogc3RyaW5nW107XG4gIHJlYWRvbmx5IHByZUFyY2hpdmVDb21tYW5kcz86IHN0cmluZ1tdO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFNvbmFyQ29kZVNjYW5uZXJQcm9wcyBleHRlbmRzIFNvbmFyQ29kZVNjYW5uZXJDb25maWcge1xuICByZWFkb25seSBzeW50aEJ1aWxkQXJuOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGFydGlmYWN0QnVja2V0QXJuOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGFydGlmYWN0QnVja2V0S2V5QXJuPzogc3RyaW5nO1xufVxuXG5jb25zdCB1bnBhY2tTb3VyY2VBbmRBcnRpZmFjdHMgPSAoKSA9PiBbXG4gICdleHBvcnQgU1lOVEhfQVJUSUZBQ1RfVVJJPWBlY2hvICRTWU5USF9BUlRJRkFDVF9MT0NBVElPTiB8IGF3ayBcXCd7c3ViKFwiYXJuOmF3czpzMzo6OlwiLFwiczM6Ly9cIil9MVxcJyAkMWAnLFxuICAnZXhwb3J0IFNZTlRIX1NPVVJDRV9VUkk9YGVjaG8gJFNZTlRIX1NPVVJDRV9WRVJTSU9OIHwgYXdrIFxcJ3tzdWIoXCJhcm46YXdzOnMzOjo6XCIsXCJzMzovL1wiKX0xXFwnICQxYCcsXG4gIFwiYXdzIHMzIGNwICRTWU5USF9TT1VSQ0VfVVJJIHNvdXJjZS56aXBcIixcbiAgXCJhd3MgczMgY3AgJFNZTlRIX0FSVElGQUNUX1VSSSBkaXN0LnppcFwiLFxuICBcInVuemlwIHNvdXJjZS56aXBcIixcbiAgXCJ1bnppcCBkaXN0LnppcCAtZCBjZGsub3V0XCIsXG4gIFwicm0gc291cmNlLnppcCBkaXN0LnppcFwiLFxuXTtcblxuY29uc3Qgb3dhc3BTY2FuID0gKCkgPT5cbiAgXCJucHggb3dhc3AtZGVwZW5kZW5jeS1jaGVjayAtLWZvcm1hdCBIVE1MIC0tb3V0IHJlcG9ydHMgLS1leGNsdWRlICcqKi9ub2RlX21vZHVsZXMvKiovKicgLS1leGNsdWRlICcqKi9yZXBvcnRzLyoqLyonIC0tZXhjbHVkZSAnKiovY2RrLm91dC8qKi8qJyAtLWV4Y2x1ZGUgJyoqLy5lbnYvKiovKicgLS1leGNsdWRlICcqKi9kaXN0LyoqLyonIC0tZXhjbHVkZSAnKiovLmdpdC8qKi8qJyAtLXNjYW4gLiAtLWVuYWJsZUV4cGVyaW1lbnRhbCAtLWJpbiAvdG1wL2RlcC1jaGVjayAtLWRpc2FibGVSZXRpcmVKU1wiO1xuXG5jb25zdCBjZm5OYWdTY2FuID0gKGNmbk5hZ0lnbm9yZVBhdGg/OiBzdHJpbmcpID0+XG4gIGBjZm5fbmFnICR7XG4gICAgY2ZuTmFnSWdub3JlUGF0aCA/PyBcIlwiXG4gIH0gY2RrLm91dC8qKi8qLnRlbXBsYXRlLmpzb24gLS1vdXRwdXQtZm9ybWF0PWpzb24gPiByZXBvcnRzL2Nmbi1uYWctcmVwb3J0Lmpzb25gO1xuXG5leHBvcnQgY2xhc3MgU29uYXJDb2RlU2Nhbm5lciBleHRlbmRzIENvbnN0cnVjdCB7XG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBTb25hckNvZGVTY2FubmVyUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3Qgc29uYXJRdWJlVG9rZW4gPSBuZXcgU2VjcmV0KHRoaXMsIFwiU29uYXJRdWJlVG9rZW5cIik7XG5cbiAgICBjb25zdCBzeW50aEJ1aWxkUHJvamVjdCA9IFByb2plY3QuZnJvbVByb2plY3RBcm4oXG4gICAgICB0aGlzLFxuICAgICAgXCJTeW50aEJ1aWxkUHJvamVjdFwiLFxuICAgICAgcHJvcHMuc3ludGhCdWlsZEFyblxuICAgICk7XG5cbiAgICBjb25zdCB2YWxpZGF0aW9uUHJvamVjdCA9IG5ldyBQcm9qZWN0KHRoaXMsIFwiVmFsaWRhdGlvblByb2plY3RcIiwge1xuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgYnVpbGRJbWFnZTogTGludXhCdWlsZEltYWdlLlNUQU5EQVJEXzVfMCxcbiAgICAgIH0sXG4gICAgICBlbnZpcm9ubWVudFZhcmlhYmxlczoge1xuICAgICAgICBTT05BUlFVQkVfVE9LRU46IHtcbiAgICAgICAgICB0eXBlOiBCdWlsZEVudmlyb25tZW50VmFyaWFibGVUeXBlLlNFQ1JFVFNfTUFOQUdFUixcbiAgICAgICAgICB2YWx1ZTogc29uYXJRdWJlVG9rZW4uc2VjcmV0QXJuLFxuICAgICAgICB9LFxuICAgICAgICBTT05BUlFVQkVfRU5EUE9JTlQ6IHtcbiAgICAgICAgICB0eXBlOiBCdWlsZEVudmlyb25tZW50VmFyaWFibGVUeXBlLlBMQUlOVEVYVCxcbiAgICAgICAgICB2YWx1ZTogcHJvcHMuc29uYXJxdWJlRW5kcG9pbnQsXG4gICAgICAgIH0sXG4gICAgICAgIFBST0pFQ1RfTkFNRToge1xuICAgICAgICAgIHR5cGU6IEJ1aWxkRW52aXJvbm1lbnRWYXJpYWJsZVR5cGUuUExBSU5URVhULFxuICAgICAgICAgIHZhbHVlOiBwcm9wcy5zb25hcnF1YmVQcm9qZWN0TmFtZSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICBidWlsZFNwZWM6IEJ1aWxkU3BlYy5mcm9tT2JqZWN0KHtcbiAgICAgICAgdmVyc2lvbjogXCIwLjJcIixcbiAgICAgICAgZW52OiB7XG4gICAgICAgICAgc2hlbGw6IFwiYmFzaFwiLFxuICAgICAgICB9LFxuICAgICAgICBwaGFzZXM6IHtcbiAgICAgICAgICBpbnN0YWxsOiB7XG4gICAgICAgICAgICBjb21tYW5kczogW1wiZ2VtIGluc3RhbGwgY2ZuLW5hZ1wiXSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGJ1aWxkOiB7XG4gICAgICAgICAgICBjb21tYW5kczogW1xuICAgICAgICAgICAgICAuLi51bnBhY2tTb3VyY2VBbmRBcnRpZmFjdHMoKSxcbiAgICAgICAgICAgICAgLi4uY3JlYXRlU29uYXJxdWJlUHJvamVjdChwcm9wcyksXG4gICAgICAgICAgICAgIG93YXNwU2NhbigpLFxuICAgICAgICAgICAgICBjZm5OYWdTY2FuKHByb3BzLmNmbk5hZ0lnbm9yZVBhdGgpLFxuICAgICAgICAgICAgICBzb25hcnF1YmVTY2FubmVyKCksXG4gICAgICAgICAgICAgIC4uLmdlbmVyYXRlU29uYXJxdWJlUmVwb3J0cygpLFxuICAgICAgICAgICAgICAuLi4ocHJvcHMucHJlQXJjaGl2ZUNvbW1hbmRzIHx8IFtdKSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pLFxuICAgIH0pO1xuXG4gICAgdmFsaWRhdGlvblByb2plY3QuYWRkVG9Sb2xlUG9saWN5KFxuICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGFjdGlvbnM6IFtcImNvZGVidWlsZDpCYXRjaEdldEJ1aWxkc1wiXSxcbiAgICAgICAgZWZmZWN0OiBFZmZlY3QuQUxMT1csXG4gICAgICAgIHJlc291cmNlczogW3N5bnRoQnVpbGRQcm9qZWN0LnByb2plY3RBcm5dLFxuICAgICAgfSlcbiAgICApO1xuXG4gICAgdmFsaWRhdGlvblByb2plY3QuYWRkVG9Sb2xlUG9saWN5KFxuICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGFjdGlvbnM6IFtcInMzOkdldE9iamVjdCpcIl0sXG4gICAgICAgIGVmZmVjdDogRWZmZWN0LkFMTE9XLFxuICAgICAgICByZXNvdXJjZXM6IFtwcm9wcy5hcnRpZmFjdEJ1Y2tldEFybiwgYCR7cHJvcHMuYXJ0aWZhY3RCdWNrZXRBcm59LyoqYF0sXG4gICAgICB9KVxuICAgICk7XG5cbiAgICBwcm9wcy5hcnRpZmFjdEJ1Y2tldEtleUFybiAmJlxuICAgICAgdmFsaWRhdGlvblByb2plY3QuYWRkVG9Sb2xlUG9saWN5KFxuICAgICAgICBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbXCJrbXM6RGVjcnlwdFwiLCBcImttczpEZXNjcmliZUtleVwiXSxcbiAgICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgICByZXNvdXJjZXM6IFtwcm9wcy5hcnRpZmFjdEJ1Y2tldEtleUFybl0sXG4gICAgICAgIH0pXG4gICAgICApO1xuXG4gICAgc3ludGhCdWlsZFByb2plY3Qub25CdWlsZFN1Y2NlZWRlZChcIk9uU3ludGhTdWNjZXNzXCIsIHtcbiAgICAgIHRhcmdldDogbmV3IENvZGVCdWlsZFByb2plY3QodmFsaWRhdGlvblByb2plY3QsIHtcbiAgICAgICAgZXZlbnQ6IFJ1bGVUYXJnZXRJbnB1dC5mcm9tT2JqZWN0KHtcbiAgICAgICAgICBlbnZpcm9ubWVudFZhcmlhYmxlc092ZXJyaWRlOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIG5hbWU6IFwiU1lOVEhfQlVJTERfSURcIixcbiAgICAgICAgICAgICAgdHlwZTogXCJQTEFJTlRFWFRcIixcbiAgICAgICAgICAgICAgdmFsdWU6IEV2ZW50RmllbGQuZnJvbVBhdGgoXCIkLmRldGFpbC5idWlsZC1pZFwiKSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIG5hbWU6IFwiU1lOVEhfQVJUSUZBQ1RfTE9DQVRJT05cIixcbiAgICAgICAgICAgICAgdHlwZTogXCJQTEFJTlRFWFRcIixcbiAgICAgICAgICAgICAgdmFsdWU6IEV2ZW50RmllbGQuZnJvbVBhdGgoXG4gICAgICAgICAgICAgICAgXCIkLmRldGFpbC5hZGRpdGlvbmFsLWluZm9ybWF0aW9uLmFydGlmYWN0LmxvY2F0aW9uXCJcbiAgICAgICAgICAgICAgKSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIG5hbWU6IFwiU1lOVEhfU09VUkNFX1ZFUlNJT05cIixcbiAgICAgICAgICAgICAgdHlwZTogXCJQTEFJTlRFWFRcIixcbiAgICAgICAgICAgICAgdmFsdWU6IEV2ZW50RmllbGQuZnJvbVBhdGgoXG4gICAgICAgICAgICAgICAgXCIkLmRldGFpbC5hZGRpdGlvbmFsLWluZm9ybWF0aW9uLnNvdXJjZS12ZXJzaW9uXCJcbiAgICAgICAgICAgICAgKSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSksXG4gICAgICB9KSxcbiAgICB9KTtcbiAgfVxufVxuIl19