import * as sfn from '@aws-cdk/aws-stepfunctions';
import { RequestContext } from '.';
import { IntegrationOptions } from '../integration';
import { AwsIntegration } from './aws';
/**
 * Options when configuring Step Functions synchronous integration with Rest API.
 *
 * @stability stable
 */
export interface StepFunctionsExecutionIntegrationOptions extends IntegrationOptions {
    /**
     * Which details of the incoming request must be passed onto the underlying state machine, such as, account id, user identity, request id, etc.
     *
     * The execution input will include a new key `requestContext`:
     *
     * {
     *    "body": {},
     *    "requestContext": {
     *        "key": "value"
     *    }
     * }
     *
     * @default - all parameters within request context will be set as false
     * @stability stable
     */
    readonly requestContext?: RequestContext;
    /**
     * Check if querystring is to be included inside the execution input.
     *
     * The execution input will include a new key `queryString`:
     *
     * {
     *    "body": {},
     *    "querystring": {
     *      "key": "value"
     *    }
     * }
     *
     * @default true
     * @stability stable
     */
    readonly querystring?: boolean;
    /**
     * Check if path is to be included inside the execution input.
     *
     * The execution input will include a new key `path`:
     *
     * {
     *    "body": {},
     *    "path": {
     *      "resourceName": "resourceValue"
     *    }
     * }
     *
     * @default true
     * @stability stable
     */
    readonly path?: boolean;
    /**
     * Check if header is to be included inside the execution input.
     *
     * The execution input will include a new key `headers`:
     *
     * {
     *    "body": {},
     *    "headers": {
     *       "header1": "value",
     *       "header2": "value"
     *    }
     * }
     *
     * @default false
     * @stability stable
     */
    readonly headers?: boolean;
}
/**
 * Options to integrate with various StepFunction API.
 *
 * @stability stable
 */
export declare class StepFunctionsIntegration {
    /**
     * Integrates a Synchronous Express State Machine from AWS Step Functions to an API Gateway method.
     *
     * @stability stable
     * @example
     *
     *    const stateMachine = new stepfunctions.StateMachine(this, 'MyStateMachine', {
     *       definition: stepfunctions.Chain.start(new stepfunctions.Pass(this, 'Pass')),
     *    });
     *
     *    const api = new apigateway.RestApi(this, 'Api', {
     *       restApiName: 'MyApi',
     *    });
     *    api.root.addMethod('GET', apigateway.StepFunctionsIntegration.startExecution(stateMachine));
     */
    static startExecution(stateMachine: sfn.IStateMachine, options?: StepFunctionsExecutionIntegrationOptions): AwsIntegration;
}
