"use strict";
const fs = require("fs");
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
const s3_assets = require("@aws-cdk/aws-s3-assets");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
const nested_stack_1 = require("../lib/nested-stack");
module.exports = {
    'fails if defined as a root'(test) {
        // THEN
        test.throws(() => new nested_stack_1.NestedStack(undefined, 'boom'), /Nested stacks cannot be defined as a root construct/);
        test.done();
    },
    'fails if defined without a parent stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const group = new core_1.Construct(app, 'group');
        // THEN
        test.throws(() => new nested_stack_1.NestedStack(app, 'boom'), /must be defined within scope of another non-nested stack/);
        test.throws(() => new nested_stack_1.NestedStack(group, 'bam'), /must be defined within scope of another non-nested stack/);
        test.done();
    },
    'can be defined as a direct child or an indirect child of a Stack'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        // THEN
        new nested_stack_1.NestedStack(parent, 'direct');
        new nested_stack_1.NestedStack(new core_1.Construct(parent, 'group'), 'indirect');
        test.done();
    },
    'nested stack is not synthesized as a stack artifact into the assembly'(test) {
        // GIVEN
        const app = new core_1.App();
        const parentStack = new core_1.Stack(app, 'parent-stack');
        new nested_stack_1.NestedStack(parentStack, 'nested-stack');
        // WHEN
        const assembly = app.synth();
        // THEN
        test.deepEqual(assembly.artifacts.length, 2);
        test.done();
    },
    'the template of the nested stack is synthesized into the cloud assembly'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'parent-stack');
        const nested = new nested_stack_1.NestedStack(parent, 'nested-stack');
        new core_1.CfnResource(nested, 'ResourceInNestedStack', { type: 'AWS::Resource::Nested' });
        // WHEN
        const assembly = app.synth();
        // THEN
        const template = JSON.parse(fs.readFileSync(path.join(assembly.directory, `${nested.node.uniqueId}.nested.template.json`), 'utf-8'));
        test.deepEqual(template, {
            Resources: {
                ResourceInNestedStack: {
                    Type: 'AWS::Resource::Nested',
                },
            },
        });
        test.done();
    },
    'file asset metadata is associated with the parent stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'parent-stack');
        const nested = new nested_stack_1.NestedStack(parent, 'nested-stack');
        new core_1.CfnResource(nested, 'ResourceInNestedStack', { type: 'AWS::Resource::Nested' });
        // WHEN
        const assembly = app.synth();
        // THEN
        test.deepEqual(assembly.getStackByName(parent.stackName).assets, [{
                path: 'parentstacknestedstack844892C0.nested.template.json',
                id: 'c639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096',
                packaging: 'file',
                sourceHash: 'c639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096',
                s3BucketParameter: 'AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3BucketDA8C3345',
                s3KeyParameter: 'AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3VersionKey09D03EE6',
                artifactHashParameter: 'AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096ArtifactHash8DE450C7',
            }]);
        test.done();
    },
    'aws::cloudformation::stack is synthesized in the parent scope'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'parent-stack');
        // WHEN
        const nested = new nested_stack_1.NestedStack(parent, 'nested-stack');
        new core_1.CfnResource(nested, 'ResourceInNestedStack', { type: 'AWS::Resource::Nested' });
        // THEN
        const assembly = app.synth();
        // assembly has one stack (the parent)
        test.deepEqual(assembly.stacks.length, 1);
        // but this stack has an asset that points to the synthesized template
        test.deepEqual(assembly.stacks[0].assets[0].path, 'parentstacknestedstack844892C0.nested.template.json');
        // the template includes our resource
        const filePath = path.join(assembly.directory, assembly.stacks[0].assets[0].path);
        test.deepEqual(JSON.parse(fs.readFileSync(filePath).toString('utf-8')), {
            Resources: { ResourceInNestedStack: { Type: 'AWS::Resource::Nested' } },
        });
        // the parent template includes the parameters and the nested stack resource which points to the s3 url
        assert_1.expect(parent).toMatch({
            Resources: {
                nestedstackNestedStacknestedstackNestedStackResource71CDD241: {
                    Type: 'AWS::CloudFormation::Stack',
                    Properties: {
                        TemplateURL: {
                            'Fn::Join': [
                                '',
                                [
                                    'https://s3.',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    '.',
                                    {
                                        Ref: 'AWS::URLSuffix',
                                    },
                                    '/',
                                    {
                                        Ref: 'AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3BucketDA8C3345',
                                    },
                                    '/',
                                    {
                                        'Fn::Select': [
                                            0,
                                            {
                                                'Fn::Split': [
                                                    '||',
                                                    {
                                                        Ref: 'AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3VersionKey09D03EE6',
                                                    },
                                                ],
                                            },
                                        ],
                                    },
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '||',
                                                    {
                                                        Ref: 'AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3VersionKey09D03EE6',
                                                    },
                                                ],
                                            },
                                        ],
                                    },
                                ],
                            ],
                        },
                    },
                },
            },
            Parameters: {
                AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3BucketDA8C3345: {
                    Type: 'String',
                    Description: 'S3 bucket for asset "c639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096"',
                },
                AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096S3VersionKey09D03EE6: {
                    Type: 'String',
                    Description: 'S3 key for asset version "c639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096"',
                },
                AssetParametersc639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096ArtifactHash8DE450C7: {
                    Type: 'String',
                    Description: 'Artifact hash for asset "c639c0a5e7320758aa22589669ecebc98f185b711300b074f53998c8f9a45096"',
                },
            },
        });
        test.done();
    },
    'Stack.of()'(test) {
        class MyNestedStack extends nested_stack_1.NestedStack {
            constructor(scope, id) {
                super(scope, id);
                const param = new core_1.CfnParameter(this, 'param', { type: 'String' });
                this.stackOfChild = core_1.Stack.of(param);
            }
        }
        const parent = new core_1.Stack();
        const nested = new MyNestedStack(parent, 'nested');
        test.ok(nested.stackOfChild === nested);
        test.ok(core_1.Stack.of(nested) === nested);
        test.done();
    },
    'references within the nested stack are not reported as cross stack references'(test) {
        class MyNestedStack extends nested_stack_1.NestedStack {
            constructor(scope, id) {
                super(scope, id);
                const param = new core_1.CfnParameter(this, 'param', { type: 'String' });
                new core_1.CfnResource(this, 'resource', {
                    type: 'My::Resource',
                    properties: {
                        SomeProp: param.valueAsString,
                    },
                });
            }
        }
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'parent');
        new MyNestedStack(parent, 'nested');
        // references are added during "prepare"
        const assembly = app.synth();
        test.deepEqual(assembly.stacks.length, 1);
        test.deepEqual(assembly.stacks[0].dependencies, []);
        test.done();
    },
    'references to a resource from the parent stack in a nested stack is translated into a cfn parameter'(test) {
        // WHEN
        class MyNestedStack extends nested_stack_1.NestedStack {
            constructor(scope, id, resourceFromParent) {
                super(scope, id);
                new core_1.CfnResource(this, 'resource', {
                    type: 'AWS::Child::Resource',
                    properties: {
                        ReferenceToResourceInParentStack: resourceFromParent.ref,
                    },
                });
                new core_1.CfnResource(this, 'resource2', {
                    type: 'My::Resource::2',
                    properties: {
                        Prop1: resourceFromParent.getAtt('Attr'),
                        Prop2: resourceFromParent.ref,
                    },
                });
            }
        }
        const app = new core_1.App();
        const parentStack = new core_1.Stack(app, 'parent');
        const resource = new core_1.CfnResource(parentStack, 'parent-resource', { type: 'AWS::Parent::Resource' });
        const nested = new MyNestedStack(parentStack, 'nested', resource);
        // THEN
        app.synth();
        // nested template should use a parameter to reference the resource from the parent stack
        assert_1.expect(nested).toMatch({
            Resources: {
                resource: {
                    Type: 'AWS::Child::Resource',
                    Properties: { ReferenceToResourceInParentStack: { Ref: 'referencetoparentparentresourceD56EA8F7Ref' } },
                },
                resource2: {
                    Type: 'My::Resource::2',
                    Properties: {
                        Prop1: { Ref: 'referencetoparentparentresourceD56EA8F7Attr' },
                        Prop2: { Ref: 'referencetoparentparentresourceD56EA8F7Ref' },
                    },
                },
            },
            Parameters: {
                referencetoparentparentresourceD56EA8F7Ref: { Type: 'String' },
                referencetoparentparentresourceD56EA8F7Attr: { Type: 'String' },
            },
        });
        // parent template should pass in the value through the parameter
        assert_1.expect(parentStack).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetoparentparentresourceD56EA8F7Ref: {
                    Ref: 'parentresource',
                },
                referencetoparentparentresourceD56EA8F7Attr: {
                    'Fn::GetAtt': [
                        'parentresource',
                        'Attr',
                    ],
                },
            },
        }));
        test.done();
    },
    'references to a resource in the nested stack in the parent is translated into a cfn output'(test) {
        class MyNestedStack extends nested_stack_1.NestedStack {
            constructor(scope, id) {
                super(scope, id);
                this.resourceFromChild = new core_1.CfnResource(this, 'resource', {
                    type: 'AWS::Child::Resource',
                });
            }
        }
        const app = new core_1.App();
        const parentStack = new core_1.Stack(app, 'parent');
        const nested = new MyNestedStack(parentStack, 'nested');
        new core_1.CfnResource(parentStack, 'another-parent-resource', {
            type: 'AWS::Parent::Resource',
            properties: {
                RefToResourceInNestedStack: nested.resourceFromChild.ref,
            },
        });
        // references are added during "prepare"
        app.synth();
        // nested template should use a parameter to reference the resource from the parent stack
        assert_1.expect(nested).toMatch({
            Resources: {
                resource: { Type: 'AWS::Child::Resource' },
            },
            Outputs: {
                parentnestedresource4D680677Ref: { Value: { Ref: 'resource' } },
            },
        });
        // parent template should pass in the value through the parameter
        assert_1.expect(parentStack).to(assert_1.haveResource('AWS::Parent::Resource', {
            RefToResourceInNestedStack: {
                'Fn::GetAtt': [
                    'nestedNestedStacknestedNestedStackResource3DD143BF',
                    'Outputs.parentnestedresource4D680677Ref',
                ],
            },
        }));
        test.done();
    },
    'nested stack references a resource from another non-nested stack (not the parent)'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1');
        const stack2 = new core_1.Stack(app, 'Stack2');
        const nestedUnderStack1 = new nested_stack_1.NestedStack(stack1, 'NestedUnderStack1');
        const resourceInStack2 = new core_1.CfnResource(stack2, 'ResourceInStack2', { type: 'MyResource' });
        // WHEN
        new core_1.CfnResource(nestedUnderStack1, 'ResourceInNestedStack1', {
            type: 'Nested::Resource',
            properties: {
                RefToSibling: resourceInStack2.getAtt('MyAttribute'),
            },
        });
        // THEN
        const assembly = app.synth();
        // producing stack should have an export
        assert_1.expect(stack2).toMatch({
            Resources: {
                ResourceInStack2: { Type: 'MyResource' },
            },
            Outputs: {
                ExportsOutputFnGetAttResourceInStack2MyAttributeC15F1009: {
                    Value: { 'Fn::GetAtt': ['ResourceInStack2', 'MyAttribute'] },
                    Export: { Name: 'Stack2:ExportsOutputFnGetAttResourceInStack2MyAttributeC15F1009' },
                },
            },
        });
        // nested stack uses Fn::ImportValue like normal
        assert_1.expect(nestedUnderStack1).toMatch({
            Resources: {
                ResourceInNestedStack1: {
                    Type: 'Nested::Resource',
                    Properties: {
                        RefToSibling: {
                            'Fn::ImportValue': 'Stack2:ExportsOutputFnGetAttResourceInStack2MyAttributeC15F1009',
                        },
                    },
                },
            },
        });
        // verify a depedency was established between the parents
        const stack1Artifact = assembly.getStackByName(stack1.stackName);
        const stack2Artifact = assembly.getStackByName(stack2.stackName);
        test.deepEqual(stack1Artifact.dependencies.length, 1);
        test.deepEqual(stack2Artifact.dependencies.length, 0);
        test.same(stack1Artifact.dependencies[0], stack2Artifact);
        test.done();
    },
    'nested stack within a nested stack references a resource in a sibling top-level stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const consumerTopLevel = new core_1.Stack(app, 'ConsumerTopLevel');
        const consumerNested1 = new nested_stack_1.NestedStack(consumerTopLevel, 'ConsumerNested1');
        const consumerNested2 = new nested_stack_1.NestedStack(consumerNested1, 'ConsumerNested2');
        const producerTopLevel = new core_1.Stack(app, 'ProducerTopLevel');
        const producer = new core_1.CfnResource(producerTopLevel, 'Producer', { type: 'Producer' });
        // WHEN
        new core_1.CfnResource(consumerNested2, 'Consumer', {
            type: 'Consumer',
            properties: {
                Ref: producer.ref,
            },
        });
        // THEN
        const manifest = app.synth();
        const consumerDeps = manifest.getStackArtifact(consumerTopLevel.artifactId).dependencies.map(d => d.id);
        test.deepEqual(consumerDeps, ['ProducerTopLevel']);
        test.done();
    },
    'another non-nested stack takes a reference on a resource within the nested stack (the parent exports)'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1');
        const stack2 = new core_1.Stack(app, 'Stack2');
        const nestedUnderStack1 = new nested_stack_1.NestedStack(stack1, 'NestedUnderStack1');
        const resourceInNestedStack = new core_1.CfnResource(nestedUnderStack1, 'ResourceInNestedStack', { type: 'MyResource' });
        // WHEN
        new core_1.CfnResource(stack2, 'ResourceInStack2', {
            type: 'JustResource',
            properties: {
                RefToSibling: resourceInNestedStack.getAtt('MyAttribute'),
            },
        });
        // THEN
        const assembly = app.synth();
        // nested stack should output this value as if it was referenced by the parent (without the export)
        assert_1.expect(nestedUnderStack1).toMatch({
            Resources: {
                ResourceInNestedStack: {
                    Type: 'MyResource',
                },
            },
            Outputs: {
                Stack1NestedUnderStack1ResourceInNestedStack6EE9DCD2MyAttribute: {
                    Value: {
                        'Fn::GetAtt': [
                            'ResourceInNestedStack',
                            'MyAttribute',
                        ],
                    },
                },
            },
        });
        // parent stack (stack1) should export this value
        test.deepEqual(assembly.getStackByName(stack1.stackName).template.Outputs, {
            ExportsOutputFnGetAttNestedUnderStack1NestedStackNestedUnderStack1NestedStackResourceF616305BOutputsStack1NestedUnderStack1ResourceInNestedStack6EE9DCD2MyAttribute564EECF3: {
                Value: { 'Fn::GetAtt': ['NestedUnderStack1NestedStackNestedUnderStack1NestedStackResourceF616305B', 'Outputs.Stack1NestedUnderStack1ResourceInNestedStack6EE9DCD2MyAttribute'] },
                Export: { Name: 'Stack1:ExportsOutputFnGetAttNestedUnderStack1NestedStackNestedUnderStack1NestedStackResourceF616305BOutputsStack1NestedUnderStack1ResourceInNestedStack6EE9DCD2MyAttribute564EECF3' },
            },
        });
        // consuming stack should use ImportValue to import the value from the parent stack
        assert_1.expect(stack2).toMatch({
            Resources: {
                ResourceInStack2: {
                    Type: 'JustResource',
                    Properties: {
                        RefToSibling: {
                            'Fn::ImportValue': 'Stack1:ExportsOutputFnGetAttNestedUnderStack1NestedStackNestedUnderStack1NestedStackResourceF616305BOutputsStack1NestedUnderStack1ResourceInNestedStack6EE9DCD2MyAttribute564EECF3',
                        },
                    },
                },
            },
        });
        test.deepEqual(assembly.stacks.length, 2);
        const stack1Artifact = assembly.getStackByName(stack1.stackName);
        const stack2Artifact = assembly.getStackByName(stack2.stackName);
        test.deepEqual(stack1Artifact.dependencies.length, 0);
        test.deepEqual(stack2Artifact.dependencies.length, 1);
        test.same(stack2Artifact.dependencies[0], stack1Artifact);
        test.done();
    },
    'references between sibling nested stacks should output from one and getAtt from the other'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'Parent');
        const nested1 = new nested_stack_1.NestedStack(parent, 'Nested1');
        const nested2 = new nested_stack_1.NestedStack(parent, 'Nested2');
        const resource1 = new core_1.CfnResource(nested1, 'Resource1', { type: 'Resource1' });
        // WHEN
        new core_1.CfnResource(nested2, 'Resource2', {
            type: 'Resource2',
            properties: {
                RefToResource1: resource1.ref,
            },
        });
        // THEN
        app.synth();
        // producing nested stack
        assert_1.expect(nested1).toMatch({
            Resources: {
                Resource1: {
                    Type: 'Resource1',
                },
            },
            Outputs: {
                ParentNested1Resource15F3F0657Ref: {
                    Value: {
                        Ref: 'Resource1',
                    },
                },
            },
        });
        // consuming nested stack
        assert_1.expect(nested2).toMatch({
            Resources: {
                Resource2: {
                    Type: 'Resource2',
                    Properties: {
                        RefToResource1: {
                            Ref: 'referencetoParentNested1NestedStackNested1NestedStackResource9C05342COutputsParentNested1Resource15F3F0657Ref',
                        },
                    },
                },
            },
            Parameters: {
                referencetoParentNested1NestedStackNested1NestedStackResource9C05342COutputsParentNested1Resource15F3F0657Ref: {
                    Type: 'String',
                },
            },
        });
        // parent
        assert_1.expect(parent).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetoParentNested1NestedStackNested1NestedStackResource9C05342COutputsParentNested1Resource15F3F0657Ref: {
                    'Fn::GetAtt': [
                        'Nested1NestedStackNested1NestedStackResourceCD0AD36B',
                        'Outputs.ParentNested1Resource15F3F0657Ref',
                    ],
                },
            },
        }));
        test.done();
    },
    'stackId returns AWS::StackId when referenced from the context of the nested stack'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // WHEN
        new core_1.CfnResource(nested, 'NestedResource', {
            type: 'Nested::Resource',
            properties: { MyStackId: nested.stackId },
        });
        // THEN
        assert_1.expect(nested).to(assert_1.haveResource('Nested::Resource', {
            MyStackId: { Ref: 'AWS::StackId' },
        }));
        test.done();
    },
    'stackId returns the REF of the CloudFormation::Stack resource when referenced from the parent stack'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // WHEN
        new core_1.CfnResource(parent, 'ParentResource', {
            type: 'Parent::Resource',
            properties: { NestedStackId: nested.stackId },
        });
        // THEN
        assert_1.expect(parent).to(assert_1.haveResource('Parent::Resource', {
            NestedStackId: { Ref: 'NestedStackNestedStackNestedStackNestedStackResourceB70834FD' },
        }));
        test.done();
    },
    'stackName returns AWS::StackName when referenced from the context of the nested stack'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // WHEN
        new core_1.CfnResource(nested, 'NestedResource', {
            type: 'Nested::Resource',
            properties: { MyStackName: nested.stackName },
        });
        // THEN
        assert_1.expect(nested).to(assert_1.haveResource('Nested::Resource', {
            MyStackName: { Ref: 'AWS::StackName' },
        }));
        test.done();
    },
    'stackName returns the REF of the CloudFormation::Stack resource when referenced from the parent stack'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // WHEN
        new core_1.CfnResource(parent, 'ParentResource', {
            type: 'Parent::Resource',
            properties: { NestedStackName: nested.stackName },
        });
        // THEN
        assert_1.expect(parent).to(assert_1.haveResource('Parent::Resource', {
            NestedStackName: {
                'Fn::Select': [
                    1,
                    {
                        'Fn::Split': [
                            '/',
                            {
                                Ref: 'NestedStackNestedStackNestedStackNestedStackResourceB70834FD',
                            },
                        ],
                    },
                ],
            },
        }));
        test.done();
    },
    '"account", "region" and "environment" are all derived from the parent'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'ParentStack', { env: { account: '1234account', region: 'us-east-44' } });
        // WHEN
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // THEN
        test.deepEqual(nested.environment, parent.environment);
        test.deepEqual(nested.account, parent.account);
        test.deepEqual(nested.region, parent.region);
        test.done();
    },
    'double-nested stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'stack');
        // WHEN
        const nested1 = new nested_stack_1.NestedStack(parent, 'Nested1');
        const nested2 = new nested_stack_1.NestedStack(nested1, 'Nested2');
        new core_1.CfnResource(nested1, 'Resource1', { type: 'Resource::1' });
        new core_1.CfnResource(nested2, 'Resource2', { type: 'Resource::2' });
        // THEN
        const assembly = app.synth();
        // nested2 is a "leaf", so it's just the resource
        assert_1.expect(nested2).toMatch({
            Resources: {
                Resource2: { Type: 'Resource::2' },
            },
        });
        // nested1 wires the nested2 template through parameters, so we expect those
        assert_1.expect(nested1).to(assert_1.haveResource('Resource::1'));
        const nested2Template = assert_1.SynthUtils.toCloudFormation(nested1);
        test.deepEqual(nested2Template.Parameters, {
            referencetostackAssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3BucketE8768F5CRef: { Type: 'String' },
            referencetostackAssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3VersionKey49DD83A2Ref: { Type: 'String' },
        });
        // parent stack should have two sets of parameters. one for the first nested stack and the second
        // for the second nested stack, passed in as parameters to the first
        const template = assert_1.SynthUtils.toCloudFormation(parent);
        test.deepEqual(template.Parameters, {
            AssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3BucketDE3B88D6: { Type: 'String', Description: 'S3 bucket for asset "8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235c"' },
            AssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3VersionKey3A62EFEA: { Type: 'String', Description: 'S3 key for asset version "8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235c"' },
            AssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cArtifactHash7DC546E0: { Type: 'String', Description: 'Artifact hash for asset "8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235c"' },
            AssetParameters8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bfS3Bucket76ACFB38: { Type: 'String', Description: 'S3 bucket for asset "8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bf"' },
            AssetParameters8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bfS3VersionKey04162EF1: { Type: 'String', Description: 'S3 key for asset version "8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bf"' },
            AssetParameters8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bfArtifactHashF227ADD3: { Type: 'String', Description: 'Artifact hash for asset "8b50795a950cca6b01352f162c45d9d274dee6bc409f2f2b2ed029ad6828b3bf"' },
        });
        // proxy asset params to nested stack
        assert_1.expect(parent).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetostackAssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3BucketE8768F5CRef: { Ref: 'AssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3BucketDE3B88D6' },
                referencetostackAssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3VersionKey49DD83A2Ref: { Ref: 'AssetParameters8169c6f8aaeaf5e2e8620f5f895ffe2099202ccb4b6889df48fe0967a894235cS3VersionKey3A62EFEA' },
            },
        }));
        // parent stack should have 2 assets
        test.deepEqual(assembly.getStackByName(parent.stackName).assets.length, 2);
        test.done();
    },
    'assets within nested stacks are proxied from the parent'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'ParentStack');
        const nested = new nested_stack_1.NestedStack(parent, 'NestedStack');
        // WHEN
        const asset = new s3_assets.Asset(nested, 'asset', {
            path: path.join(__dirname, 'asset-fixture.txt'),
        });
        new core_1.CfnResource(nested, 'NestedResource', {
            type: 'Nested::Resource',
            properties: {
                AssetBucket: asset.s3BucketName,
                AssetKey: asset.s3ObjectKey,
            },
        });
        // THEN
        const assembly = app.synth();
        const template = assert_1.SynthUtils.toCloudFormation(parent);
        // two sets of asset parameters: one for the nested stack itself and one as a proxy for the asset within the stack
        test.deepEqual(template.Parameters, {
            AssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3BucketC188F637: { Type: 'String', Description: 'S3 bucket for asset "db01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281"' },
            AssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3VersionKeyC7F4DBF2: { Type: 'String', Description: 'S3 key for asset version "db01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281"' },
            AssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281ArtifactHash373B14D2: { Type: 'String', Description: 'Artifact hash for asset "db01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281"' },
            AssetParameters46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71S3Bucket3C4265E9: { Type: 'String', Description: 'S3 bucket for asset "46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71"' },
            AssetParameters46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71S3VersionKey8E981535: { Type: 'String', Description: 'S3 key for asset version "46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71"' },
            AssetParameters46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71ArtifactHash45A28583: { Type: 'String', Description: 'Artifact hash for asset "46b107d6db798ca46046b8669d057a4debcbdbaaddb6170400748c2f9e4f9d71"' },
        });
        // asset proxy parameters are passed to the nested stack
        assert_1.expect(parent).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetoParentStackAssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3Bucket82C55B96Ref: { Ref: 'AssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3BucketC188F637' },
                referencetoParentStackAssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3VersionKeyA43C3CC6Ref: { Ref: 'AssetParametersdb01ee2eb7adc7915e364dc410d861e569543f9be3761d535a68d5c2cc181281S3VersionKeyC7F4DBF2' },
            },
        }));
        // parent stack should have 2 assets
        test.deepEqual(assembly.getStackByName(parent.stackName).assets.length, 2);
        test.done();
    },
    'docker image assets are wired through the top-level stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const parent = new core_1.Stack(app, 'my-stack');
        const nested = new nested_stack_1.NestedStack(parent, 'nested-stack');
        // WHEN
        const location = nested.addDockerImageAsset({
            directoryName: 'my-image',
            dockerBuildArgs: { key: 'value', boom: 'bam' },
            dockerBuildTarget: 'buildTarget',
            sourceHash: 'hash-of-source',
        });
        // use the asset, so the parameters will be wired.
        new sns.Topic(nested, 'MyTopic', {
            displayName: `image location is ${location.imageUri}`,
        });
        // THEN
        const asm = app.synth();
        test.deepEqual(asm.getStackArtifact(parent.artifactId).assets, [
            {
                repositoryName: 'aws-cdk/assets',
                imageTag: 'hash-of-source',
                id: 'hash-of-source',
                packaging: 'container-image',
                path: 'my-image',
                sourceHash: 'hash-of-source',
                buildArgs: { key: 'value', boom: 'bam' },
                target: 'buildTarget',
            },
            {
                path: 'mystacknestedstackFAE12FB5.nested.template.json',
                id: 'fcdaee79eb79f37eca3a9b1cc0cc9ba150e4eea8c5d6d0c343cb6cd9dc68e2e5',
                packaging: 'file',
                sourceHash: 'fcdaee79eb79f37eca3a9b1cc0cc9ba150e4eea8c5d6d0c343cb6cd9dc68e2e5',
                s3BucketParameter: 'AssetParametersfcdaee79eb79f37eca3a9b1cc0cc9ba150e4eea8c5d6d0c343cb6cd9dc68e2e5S3Bucket67A749F8',
                s3KeyParameter: 'AssetParametersfcdaee79eb79f37eca3a9b1cc0cc9ba150e4eea8c5d6d0c343cb6cd9dc68e2e5S3VersionKeyE1E6A8D4',
                artifactHashParameter: 'AssetParametersfcdaee79eb79f37eca3a9b1cc0cc9ba150e4eea8c5d6d0c343cb6cd9dc68e2e5ArtifactHash0AEDBE8A',
            },
        ]);
        test.done();
    },
    'metadata defined in nested stacks is reported at the parent stack level in the cloud assembly'(test) {
        // GIVEN
        const app = new core_1.App({ stackTraces: false });
        const parent = new core_1.Stack(app, 'parent');
        const child = new core_1.Stack(parent, 'child');
        const nested = new nested_stack_1.NestedStack(child, 'nested');
        const resource = new core_1.CfnResource(nested, 'resource', { type: 'foo' });
        // WHEN
        resource.node.addMetadata('foo', 'bar');
        // THEN: the first non-nested stack records the assembly metadata
        const asm = app.synth();
        test.deepEqual(asm.stacks.length, 2); // only one stack is defined as an artifact
        test.deepEqual(asm.getStackByName(parent.stackName).findMetadataByType('foo'), []);
        test.deepEqual(asm.getStackByName(child.stackName).findMetadataByType('foo'), [
            {
                path: '/parent/child/nested/resource',
                type: 'foo',
                data: 'bar',
            },
        ]);
        test.done();
    },
    'referencing attributes with period across stacks'(test) {
        // GIVEN
        const parent = new core_1.Stack();
        const nested = new nested_stack_1.NestedStack(parent, 'nested');
        const consumed = new core_1.CfnResource(nested, 'resource-in-nested', { type: 'CONSUMED' });
        // WHEN
        new core_1.CfnResource(parent, 'resource-in-parent', {
            type: 'CONSUMER',
            properties: {
                ConsumedAttribute: consumed.getAtt('Consumed.Attribute'),
            },
        });
        // THEN
        assert_1.expect(nested).toMatch({
            Resources: {
                resourceinnested: {
                    Type: 'CONSUMED',
                },
            },
            Outputs: {
                nestedresourceinnested59B1F01CConsumedAttribute: {
                    Value: {
                        'Fn::GetAtt': [
                            'resourceinnested',
                            'Consumed.Attribute',
                        ],
                    },
                },
            },
        });
        assert_1.expect(parent).to(assert_1.haveResource('CONSUMER', {
            ConsumedAttribute: {
                'Fn::GetAtt': [
                    'nestedNestedStacknestedNestedStackResource3DD143BF',
                    'Outputs.nestedresourceinnested59B1F01CConsumedAttribute',
                ],
            },
        }));
        test.done();
    },
    'missing context in nested stack is reported if the context is not available'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'ParentStack', { env: { account: '1234account', region: 'us-east-44' } });
        const nestedStack = new nested_stack_1.NestedStack(stack, 'nested');
        const provider = 'availability-zones';
        const expectedKey = core_1.ContextProvider.getKey(nestedStack, {
            provider,
        }).key;
        // WHEN
        core_1.ContextProvider.getValue(nestedStack, {
            provider,
            dummyValue: ['dummy1a', 'dummy1b', 'dummy1c'],
        });
        // THEN: missing context is reported in the cloud assembly
        const asm = app.synth();
        const missing = asm.manifest.missing;
        test.ok(missing && missing.find(m => {
            return (m.key === expectedKey);
        }));
        test.done();
    },
    'references to a resource from a deeply nested stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const top = new core_1.Stack(app, 'stack');
        const topLevel = new core_1.CfnResource(top, 'toplevel', { type: 'TopLevel' });
        const nested1 = new nested_stack_1.NestedStack(top, 'nested1');
        const nested2 = new nested_stack_1.NestedStack(nested1, 'nested2');
        // WHEN
        new core_1.CfnResource(nested2, 'refToTopLevel', {
            type: 'BottomLevel',
            properties: { RefToTopLevel: topLevel.ref },
        });
        // THEN
        assert_1.expect(top).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetostackAssetParameters842982bd421cce9742ba27151ef12ed699d44d22801f41e8029f63f2358a3f2fS3Bucket5DA5D2E7Ref: {
                    Ref: 'AssetParameters842982bd421cce9742ba27151ef12ed699d44d22801f41e8029f63f2358a3f2fS3BucketDD4D96B5',
                },
                referencetostackAssetParameters842982bd421cce9742ba27151ef12ed699d44d22801f41e8029f63f2358a3f2fS3VersionKey8FBE5C12Ref: {
                    Ref: 'AssetParameters842982bd421cce9742ba27151ef12ed699d44d22801f41e8029f63f2358a3f2fS3VersionKey83E381F3',
                },
                referencetostacktoplevelBB16BF13Ref: {
                    Ref: 'toplevel',
                },
            },
        }));
        assert_1.expect(nested1).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                referencetostacktoplevelBB16BF13Ref: {
                    Ref: 'referencetostacktoplevelBB16BF13Ref',
                },
            },
        }));
        assert_1.expect(nested2).to(assert_1.matchTemplate({
            Resources: {
                refToTopLevel: {
                    Type: 'BottomLevel',
                    Properties: {
                        RefToTopLevel: {
                            Ref: 'referencetostacktoplevelBB16BF13Ref',
                        },
                    },
                },
            },
            Parameters: {
                referencetostacktoplevelBB16BF13Ref: {
                    Type: 'String',
                },
            },
        }));
        test.done();
    },
    'bottom nested stack consumes value from a top-level stack through a parameter in a middle nested stack'(test) {
        // GIVEN
        const app = new core_1.App();
        const top = new core_1.Stack(app, 'Grandparent');
        const middle = new nested_stack_1.NestedStack(top, 'Parent');
        const bottom = new nested_stack_1.NestedStack(middle, 'Child');
        const resourceInGrandparent = new core_1.CfnResource(top, 'ResourceInGrandparent', { type: 'ResourceInGrandparent' });
        // WHEN
        new core_1.CfnResource(bottom, 'ResourceInChild', {
            type: 'ResourceInChild',
            properties: {
                RefToGrandparent: resourceInGrandparent.ref,
            },
        });
        // THEN
        // this is the name allocated for the parameter that's propagated through
        // the hierarchy.
        const paramName = 'referencetoGrandparentResourceInGrandparent010E997ARef';
        // child (bottom) references through a parameter.
        assert_1.expect(bottom).toMatch({
            Resources: {
                ResourceInChild: {
                    Type: 'ResourceInChild',
                    Properties: {
                        RefToGrandparent: { Ref: paramName },
                    },
                },
            },
            Parameters: {
                [paramName]: { Type: 'String' },
            },
        });
        // the parent (middle) sets the value of this parameter to be a reference to another parameter
        assert_1.expect(middle).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                [paramName]: { Ref: paramName },
            },
        }));
        // grandparent (top) assigns the actual value to the parameter
        assert_1.expect(top).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
            Parameters: {
                [paramName]: { Ref: 'ResourceInGrandparent' },
                // these are for the asset of the bottom nested stack
                referencetoGrandparentAssetParameters3208f43b793a1dbe28ca02cf31fb975489071beb42c492b22dc3d32decc3b4b7S3Bucket06EEE58DRef: {
                    Ref: 'AssetParameters3208f43b793a1dbe28ca02cf31fb975489071beb42c492b22dc3d32decc3b4b7S3Bucket01877C2E',
                },
                referencetoGrandparentAssetParameters3208f43b793a1dbe28ca02cf31fb975489071beb42c492b22dc3d32decc3b4b7S3VersionKeyD3B04909Ref: {
                    Ref: 'AssetParameters3208f43b793a1dbe28ca02cf31fb975489071beb42c492b22dc3d32decc3b4b7S3VersionKey5765F084',
                },
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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