"use strict";
const assert_1 = require("@aws-cdk/assert");
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const cdk = require("@aws-cdk/core");
const nodeunit_1 = require("nodeunit");
const lib_1 = require("../lib");
class TestCustomResource extends cdk.Construct {
    constructor(scope, id, opts = {}) {
        super(scope, id);
        const singletonLambda = new lambda.SingletonFunction(this, 'Lambda', {
            uuid: 'TestCustomResourceProvider',
            code: new lambda.InlineCode('def hello(): pass'),
            runtime: lambda.Runtime.PYTHON_2_7,
            handler: 'index.hello',
            timeout: cdk.Duration.minutes(5),
        });
        this.resource = new lib_1.CustomResource(this, 'Resource', {
            ...opts,
            provider: lib_1.CustomResourceProvider.fromLambda(singletonLambda),
        });
    }
}
module.exports = nodeunit_1.testCase({
    'custom resources honor removalPolicy': {
        'unspecified (aka .Destroy)'(test) {
            // GIVEN
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            // WHEN
            new TestCustomResource(stack, 'Custom');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {}, assert_1.ResourcePart.CompleteDefinition));
            test.equal(app.synth().tryGetArtifact(stack.stackName).findMetadataByType('aws:cdk:protected').length, 0);
            test.done();
        },
        '.Destroy'(test) {
            // GIVEN
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            // WHEN
            new TestCustomResource(stack, 'Custom', { removalPolicy: cdk.RemovalPolicy.DESTROY });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {}, assert_1.ResourcePart.CompleteDefinition));
            test.equal(app.synth().tryGetArtifact(stack.stackName).findMetadataByType('aws:cdk:protected').length, 0);
            test.done();
        },
        '.Retain'(test) {
            // GIVEN
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            // WHEN
            new TestCustomResource(stack, 'Custom', { removalPolicy: cdk.RemovalPolicy.RETAIN });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::CustomResource', {
                DeletionPolicy: 'Retain',
                UpdateReplacePolicy: 'Retain',
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
    },
    'custom resource is added twice, lambda is added once'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'Test');
        // WHEN
        new TestCustomResource(stack, 'Custom1');
        new TestCustomResource(stack, 'Custom2');
        // THEN
        assert_1.expect(stack).toMatch({
            'Resources': {
                'SingletonLambdaTestCustomResourceProviderServiceRole81FEAB5C': {
                    'Type': 'AWS::IAM::Role',
                    'Properties': {
                        'AssumeRolePolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 'sts:AssumeRole',
                                    'Effect': 'Allow',
                                    'Principal': {
                                        'Service': 'lambda.amazonaws.com',
                                    },
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                        'ManagedPolicyArns': [
                            { 'Fn::Join': ['', [
                                        'arn:', { 'Ref': 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole'
                                    ]] },
                        ],
                    },
                },
                'SingletonLambdaTestCustomResourceProviderA9255269': {
                    'Type': 'AWS::Lambda::Function',
                    'Properties': {
                        'Code': {
                            'ZipFile': 'def hello(): pass',
                        },
                        'Handler': 'index.hello',
                        'Role': {
                            'Fn::GetAtt': [
                                'SingletonLambdaTestCustomResourceProviderServiceRole81FEAB5C',
                                'Arn',
                            ],
                        },
                        'Runtime': 'python2.7',
                        'Timeout': 300,
                    },
                    'DependsOn': [
                        'SingletonLambdaTestCustomResourceProviderServiceRole81FEAB5C',
                    ],
                },
                'Custom1D319B237': {
                    'Type': 'AWS::CloudFormation::CustomResource',
                    'DeletionPolicy': 'Delete',
                    'UpdateReplacePolicy': 'Delete',
                    'Properties': {
                        'ServiceToken': {
                            'Fn::GetAtt': [
                                'SingletonLambdaTestCustomResourceProviderA9255269',
                                'Arn',
                            ],
                        },
                    },
                },
                'Custom2DD5FB44D': {
                    'Type': 'AWS::CloudFormation::CustomResource',
                    'DeletionPolicy': 'Delete',
                    'UpdateReplacePolicy': 'Delete',
                    'Properties': {
                        'ServiceToken': {
                            'Fn::GetAtt': [
                                'SingletonLambdaTestCustomResourceProviderA9255269',
                                'Arn',
                            ],
                        },
                    },
                },
            },
        });
        test.done();
    },
    'custom resources can specify a resource type that starts with Custom::'(test) {
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'Test');
        new lib_1.CustomResource(stack, 'MyCustomResource', {
            resourceType: 'Custom::MyCustomResourceType',
            provider: lib_1.CustomResourceProvider.fromTopic(new sns.Topic(stack, 'Provider')),
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::MyCustomResourceType'));
        test.done();
    },
    'fails if custom resource type is invalid': {
        'does not start with "Custom::"'(test) {
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            test.throws(() => {
                new lib_1.CustomResource(stack, 'MyCustomResource', {
                    resourceType: 'NoCustom::MyCustomResourceType',
                    provider: lib_1.CustomResourceProvider.fromTopic(new sns.Topic(stack, 'Provider')),
                });
            }, /Custom resource type must begin with "Custom::"/);
            test.done();
        },
        'has invalid characters'(test) {
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            test.throws(() => {
                new lib_1.CustomResource(stack, 'MyCustomResource', {
                    resourceType: 'Custom::My Custom?ResourceType',
                    provider: lib_1.CustomResourceProvider.fromTopic(new sns.Topic(stack, 'Provider')),
                });
            }, /Custom resource type name can only include alphanumeric characters and/);
            test.done();
        },
        'is longer than 60 characters'(test) {
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'Test');
            test.throws(() => {
                new lib_1.CustomResource(stack, 'MyCustomResource', {
                    resourceType: 'Custom::0123456789012345678901234567890123456789012345678901234567891',
                    provider: lib_1.CustomResourceProvider.fromTopic(new sns.Topic(stack, 'Provider')),
                });
            }, /Custom resource type length > 60/);
            test.done();
        },
    },
    '.ref returns the intrinsic reference (physical name)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const res = new TestCustomResource(stack, 'myResource');
        // THEN
        test.deepEqual(stack.resolve(res.resource.ref), { Ref: 'myResourceC6A188A9' });
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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