import pytest
import pandas as pd
import scanpy as sc
import scipy.sparse


@pytest.fixture(scope="session")
def create_testfolder(tmp_path_factory):
    """Create test folder."""
    fn = tmp_path_factory.mktemp("testfolder")
    return (fn)


@pytest.fixture
def database_paths():
    """Database paths for tests."""
    db = {
        'igblast_db': "database/igblast/",
        'germline': "database/germlines/imgt/human/vdj/",
        'blastdb': "database/blast/human/",
        'blastdb_fasta': "database/blast/human/human_BCR_C.fasta",
    }
    return (db)


@pytest.fixture
def processed_files():
    """Database paths for tests."""
    fl = {
        'filtered':
        "filtered_contig_igblast_db-pass_genotyped.tsv",
        'all':
        "all_contig_igblast_db-pass_genotyped.tsv",
        'germline-dmask':
        "filtered_contig_igblast_db-pass_genotyped_germline_dmask.tsv",
    }
    return (fl)


@pytest.fixture
def processed_files_tr():
    """Database paths for tests."""
    fl = {
        'filtered': "filtered_contig_igblast_db-pass.tsv",
        'all': "all_contig_igblast_db-pass.tsv",
    }
    return (fl)


@pytest.fixture
def dummy_adata():
    """Dummy anndata for tests."""
    barcodes = [
        'AAACCTGTCATATCGG-1',
        'AAACCTGTCCGTTGTC-1',
        'AAACCTGTCGAGAACG-1',
        'AAACCTGTCTTGAGAC-1',
        'AAACGGGAGCGACGTA-1',
    ]
    obs = pd.DataFrame(index=barcodes)
    n = obs.shape[0]

    # just create a random matrix
    adata = sc.AnnData(X=scipy.sparse.random(n, 100, format='csr'), obs=obs)

    # this is just to populate the neighbors slot
    sc.pp.neighbors(adata, use_rep='X', n_neighbors=3)

    return (adata)


@pytest.fixture
def dummy_adata_tr():
    """Dummy anndata for tests."""
    barcodes = [
        'AAGACCTCACTACAGT-1',
        'AACTGGTTCTTTAGTC-1',
        'AAAGATGCACCCTATC-1',
    ]
    obs = pd.DataFrame(index=barcodes)
    n = obs.shape[0]

    # just create a random matrix
    adata = sc.AnnData(X=scipy.sparse.random(n, 100, format='csr'), obs=obs)

    # this is just to populate the neighbors slot
    sc.pp.neighbors(adata, use_rep='X', n_neighbors=3)

    return (adata)


@pytest.fixture
def fasta_10x():
    """Standard cellranger fasta file to test the preprocessing."""
    seq = {
        'AAACCTGTCATATCGG-1_contig_1':
        'TGGGGAGGAGTCAGTCCCAACCAGGACACGGCCTGGACATGAGGGTCCCTGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCTCAGGTGCCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTGGGAGACAGAGTCACCATCACTTGCCAGGCGACACAAGACATTAACAATTATGTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAACTCCTGATCTACGATGCATTGAATTTAGAAATAGGGGTCCCATCAAGATTCAGTGGAAGAGGGTCTGGGACAGTCTTTATTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACATACTACTGTCAACAATATGACGAACTTCCCGTCACTTTCGGCGGAGGGACCAATGTGGAAATGAGACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
        'AAACCTGTCCGTTGTC-1_contig_1':
        'AGGAGTCAGACCCTGTCAGGACACAGCATAGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
        'AAACCTGTCCGTTGTC-1_contig_2':
        'ATCACATAACAACCACATTCCTCCTCTAAAGAAGCCCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
        'AAACCTGTCGAGAACG-1_contig_1':
        'ACTGTGGGGGTAAGAGGTTGTGTCCACCATGGCCTGGACTCCTCTCCTCCTCCTGTTCCTCTCTCACTGCACAGGTTCCCTCTCGCAGGCTGTGCTGACTCAGCCGTCTTCCCTCTCTGCATCTCCTGGAGCATCAGGCAGGCTCACCTGCACCTTACGCAGTGACATCAATGTTGGTACGTACAGGATATATTGGTACCAGCGGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGACTCAGATAAGCAGCAGGGCTCTGGAGTCCCCAGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTATTGTATGATTTGGCACAGCAGCGCTTGGGTGGTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA',
        'AAACCTGTCGAGAACG-1_contig_2':
        'GGGAGCATCACCCAGCAACCACATCTGTCCTCTAGAGAATCCCCTGAGAGCTCCGTTCCTCACCATGGACTGGACCTGGAGGATCCTCTTCTTGGTGGCAGCAGCCACAGGAGCCCACTCGCAGGTGCAACTGGTGCAGTCTGGGGGTGAGGTAAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTCACCGACTACTTCATACAGTGGCTGCGACACGCCCCTGGACAGGGGCTTGATTGGATGGGTTTAATCAACCCTAACAGTGGTGACACCAACTATGCACAGAAGTTTCAGGGCAGAGTCACCATGACCAGGGACACGTCCATCAGTACAGCCTACATGGAACTGAGCAGCCTGAGATCTGACGACACGGCCGTATATTACTGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
        'AAACCTGTCTTGAGAC-1_contig_1':
        'AGGAGTCAGACCCAGTCAGGACACAGCATGGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
        'AAACCTGTCTTGAGAC-1_contig_2':
        'GGAGTCTCCCTCACCGCCCAGCTGGGATCTCAGGGCTTCATTTTCTGTCCTCCACCATCATGGGGTCAACCGCCATCCTCGCCCTCCTCCTGGCTGTTCTCCAAGGAGTCTGTGCCGAGGTGCAGCTGGTGCAGTCTGGAGCAGAGGTGAAAAAGCCGGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTTACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGTGACTCTGATACCAGATACAGCCCGTCCTTCCAAGGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
        'AAACGGGAGCGACGTA-1_contig_1':
        'GGGGACTTTCTGAGACTCATGGACCTCCTGCACAAGAACATGAAACACCTGTGGTTCTTCCTCCTCCTGGTGGCAGCTCCCAGATGGGTCCTGTCCCAGGTGCAGCTGCAGGAGTCGGGCCCAGGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATCAGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGTGGGAGCACCAACTACAACCCCTCCCTCAAGAGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
        'AAACGGGAGCGACGTA-1_contig_2':
        'TCTGGCACCAGGGGTCCCTTCCAATATCAGCACCATGGCCTGGACTCCTCTCTTTCTGTTCCTCCTCACTTGCTGCCCAGGGTCCAATTCCCAGGCTGTGGTGACTCAGGAGCCCTCACTGACTGTGTCCCCAGGAGGGACAGTCACTCTCACCTGTGGCTCCAGCACTGGAGCTGTCACCAGTGGTCATTATCCCTACTGGTTCCAGCAGAAGCCTGGCCAAGCCCCCAGGACACTGATTTATGATACAAGCAACAAACACTCCTGGACACCTGCCCGGTTCTCAGGCTCCCTCCTTGGGGGCAAAGCTGCCCTGACCCTTTCGGGTGCGCAGCCTGAGGATGAGGCTGAGTATTACTGCTTGCTCTCCTATAGTGGTGCTAGGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA'
    }
    return (seq)


@pytest.fixture
def annotation_10x():
    """Standard cellranger annotation file to test the preprocessing."""
    annot = pd.DataFrame([
        [
            'AAACCTGTCATATCGG-1', 'True', 'AAACCTGTCATATCGG-1_contig_1',
            'True', '556', 'IGK', 'IGKV1-8', 'None', 'IGKJ4', 'IGKC', 'True',
            'True', 'CQQYDELPVTF', 'TGTCAACAATATGACGAACTTCCCGTCACTTTC', '9139',
            '68', 'clonotype9', 'clonotype9_consensus_1'
        ],
        [
            'AAACCTGTCCGTTGTC-1', 'True', 'AAACCTGTCCGTTGTC-1_contig_1',
            'True', '551', 'IGK', 'IGKV1-8', 'None', 'IGKJ1', 'IGKC', 'True',
            'True', 'CQQYYSYPRTF', 'TGTCAACAGTATTATAGTTACCCTCGGACGTTC', '5679',
            '43', 'clonotype10', 'clonotype10_consensus_1'
        ],
        [
            'AAACCTGTCCGTTGTC-1', 'True', 'AAACCTGTCCGTTGTC-1_contig_2',
            'True', '565', 'IGH', 'IGHV1-69D', 'IGHD3-22', 'IGHJ3', 'IGHM',
            'True', 'True', 'CATTYYYDSSGYYQNDAFDIW',
            'TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG',
            '4161', '51', 'clonotype10', 'clonotype10_consensus_2'
        ],
        [
            'AAACCTGTCGAGAACG-1', 'True', 'AAACCTGTCGAGAACG-1_contig_1',
            'True', '642', 'IGL', 'IGLV5-45', 'None', 'IGLJ3', 'IGLC3', 'True',
            'True', 'CMIWHSSAWVV', 'TGTATGATTTGGCACAGCAGCGCTTGGGTGGTC',
            '13160', '90', 'clonotype11', 'clonotype11_consensus_1'
        ],
        [
            'AAACCTGTCGAGAACG-1', 'True', 'AAACCTGTCGAGAACG-1_contig_2',
            'True', '550', 'IGH', 'IGHV1-2', 'None', 'IGHJ3', 'IGHM', 'True',
            'True', 'CAREIEGDGVFEIW',
            'TGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGG', '5080', '47',
            'clonotype11', 'clonotype11_consensus_2'
        ],
        [
            'AAACCTGTCTTGAGAC-1', 'True', 'AAACCTGTCTTGAGAC-1_contig_1',
            'True', '551', 'IGK', 'IGKV1D-8', 'None', 'IGKJ2', 'IGKC', 'True',
            'True', 'CQQYYSFPYTF', 'TGTCAACAGTATTATAGTTTCCCGTACACTTTT', '2813',
            '22', 'clonotype12', 'clonotype12_consensus_1'
        ],
        [
            'AAACCTGTCTTGAGAC-1', 'True', 'AAACCTGTCTTGAGAC-1_contig_2',
            'True', '557', 'IGH', 'IGHV5-51', 'None', 'IGHJ3', 'IGHM', 'True',
            'True', 'CARHIRGNRFGNDAFDIW',
            'TGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGG', '8292',
            '80', 'clonotype12', 'clonotype12_consensus_2'
        ],
        [
            'AAACGGGAGCGACGTA-1', 'True', 'AAACGGGAGCGACGTA-1_contig_1',
            'True', '534', 'IGH', 'IGHV4-59', 'None', 'IGHJ3', 'IGHM', 'True',
            'True', 'CARVGYRAAAGTDAFDIW',
            'TGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGG', '1235',
            '18', 'clonotype13', 'clonotype13_consensus_1'
        ],
        [
            'AAACGGGAGCGACGTA-1', 'True', 'AAACGGGAGCGACGTA-1_contig_2',
            'True', '631', 'IGL', 'IGLV7-46', 'None', 'IGLJ3', 'IGLC2', 'True',
            'False', 'None', 'None', '739', '8', 'clonotype13', 'None'
        ],
    ],
        columns=[
        'barcode', 'is_cell', 'contig_id',
        'high_confidence', 'length', 'chain', 'v_gene',
        'd_gene', 'j_gene', 'c_gene', 'full_length',
        'productive', 'cdr3', 'cdr3_nt', 'reads', 'umis',
        'raw_clonotype_id', 'raw_consensus_id'
    ])

    return (annot)


@pytest.fixture
def airr_10x():
    """Standard cellranger airr file to test the preprocessing."""
    airr = pd.DataFrame([
        [
            'AAACCTGTCATATCGG-1', '', 'AAACCTGTCATATCGG-1_contig_1',
            'TGGGGAGGAGTCAGTCCCAACCAGGACACGGCCTGGACATGAGGGTCCCTGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCTCAGGTGCCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTGGGAGACAGAGTCACCATCACTTGCCAGGCGACACAAGACATTAACAATTATGTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAACTCCTGATCTACGATGCATTGAATTTAGAAATAGGGGTCCCATCAAGATTCAGTGGAAGAGGGTCTGGGACAGTCTTTATTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACATACTACTGTCAACAATATGACGAACTTCCCGTCACTTTCGGCGGAGGGACCAATGTGGAAATGAGACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            'MRVPAQLLGLLLLWLSGARCDIQMTQSPSSLSASVGDRVTITCQATQDINNYVNWYQQKPGKAPKLLIYDALNLEIGVPSRFSGRGSGTVFILTISSLQPEDVATYYCQQYDELPVTFGGGTNVEMRRTVAAPSVFIFPPSDEQLKSGTASVVCLLNNFYPREAKVQWKVDN',
            'T', 'F', 'IGKV1-8', '38S314M204S', '', '', 'IGKJ4', '383S37M136S',
            'IGKC', '420S136M',
            'TGGGGAGGAGTCAGTCCCAACCAGGACACGGCCTGGACATGAGGGTCCCTGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCTCAGGTGCCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTGGGAGACAGAGTCACCATCACTTGCCAGGCGACACAAGACATTAACAATTATGTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAACTCCTGATCTACGATGCATTGAATTTAGAAATAGGGGTCCCATCAAGATTCAGTGGAAGAGGGTCTGGGACAGTCTTTATTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACATACTACTGTCAACAATATGACGAACTTCCCGTCACTTTCGGCGGAGGGACCAATGTGGAAATGAGACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            'ATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCTCACTTTCGGCGGAGGGACCAAGGTGGAGATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            'TGTCAACAATATGACGAACTTCCCGTCACTTTC', 'CQQYDELPVTF', '33', '11',
            '39', '352', '', '', '384', '420', '421', '556', '9139', '68', 'T'
        ],
        [
            'AAACCTGTCCGTTGTC-1', '', 'AAACCTGTCCGTTGTC-1_contig_1',
            'AGGAGTCAGACCCTGTCAGGACACAGCATAGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            'MRVPAQLLGLLLLWLPGARCAIRMTQSPSSFSASTGDRVTITCRASQGISSYLAWYQQKPGKAPKLLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSYPRTFGQGTKVEIKRTVAAPSVFIFPPSDEQLKSGTASVVCLLNNFYPREAKVQWKVDN',
            'T', 'F', 'IGKV1-8', '33S345M173S', '', '', 'IGKJ1', '377S38M136S',
            'IGKC', '415S136M',
            'AGGAGTCAGACCCTGTCAGGACACAGCATAGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            'AGGCTGGACACACTTCATGCAGGAGTCAGACCCTGTCAGGACACAGCATAGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTGTGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            'TGTCAACAGTATTATAGTTACCCTCGGACGTTC', 'CQQYYSYPRTF', '33', '11',
            '34', '378', '', '', '378', '415', '416', '551', '5679', '43', 'T'
        ],
        [
            'AAACCTGTCCGTTGTC-1', '', 'AAACCTGTCCGTTGTC-1_contig_2',
            'ATCACATAACAACCACATTCCTCCTCTAAAGAAGCCCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'MDWTWRFLFVVAAATGVQSQVQLVQSGAEVKKPGSSVKVSCKASGGTFSSYAISWVRQAPGQGLEWMGGIIPIFGTANYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYCATTYYYDSSGYYQNDAFDIWGQGTMVTVSSGSASAPTLFPLVSCENSPSDTSSV',
            'T', 'F', 'IGHV1-69D', '58S353M154S', 'IGHD3-22', '411S31M123S',
            'IGHJ3', '444S50M71S', 'IGHM', '494S71M',
            'ATCACATAACAACCACATTCCTCCTCTAAAGAAGCCCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'ATCACATAACAACCACATTCCTCCTCTAAAGAAGCCCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGAGAGAGTATTACTATGATAGTAGTGGTTATTACTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG',
            'CATTYYYDSSGYYQNDAFDIW', '63', '21', '59', '411', '412', '442',
            '445', '494', '495', '565', '4161', '51', 'T'
        ],
        [
            'AAACCTGTCGAGAACG-1', '', 'AAACCTGTCGAGAACG-1_contig_1',
            'ACTGTGGGGGTAAGAGGTTGTGTCCACCATGGCCTGGACTCCTCTCCTCCTCCTGTTCCTCTCTCACTGCACAGGTTCCCTCTCGCAGGCTGTGCTGACTCAGCCGTCTTCCCTCTCTGCATCTCCTGGAGCATCAGGCAGGCTCACCTGCACCTTACGCAGTGACATCAATGTTGGTACGTACAGGATATATTGGTACCAGCGGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGACTCAGATAAGCAGCAGGGCTCTGGAGTCCCCAGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTATTGTATGATTTGGCACAGCAGCGCTTGGGTGGTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA',
            'MAWTPLLLLFLSHCTGSLSQAVLTQPSSLSASPGASGRLTCTLRSDINVGTYRIYWYQRKPGSPPQYLLRYKSDSDKQQGSGVPSRFSGSKDASANAGILLISGLQSEDEADYYCMIWHSSAWVVGGGTKLTVLGQPKAAPSVTLFPPSSEELQANKATLVCLISDFYPGAVTVAWKADSSPVKAGVETTTPSKQSNNKYAASS',
            'T', 'F', 'IGLV5-45', '28S369M245S', '', '', 'IGLJ3',
            '393S38M211S', 'IGLC3', '431S211M',
            'ACTGTGGGGGTAAGAGGTTGTGTCCACCATGGCCTGGACTCCTCTCCTCCTCCTGTTCCTCTCTCACTGCACAGGTTCCCTCTCGCAGGCTGTGCTGACTCAGCCGTCTTCCCTCTCTGCATCTCCTGGAGCATCAGGCAGGCTCACCTGCACCTTACGCAGTGACATCAATGTTGGTACGTACAGGATATATTGGTACCAGCGGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGACTCAGATAAGCAGCAGGGCTCTGGAGTCCCCAGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTATTGTATGATTTGGCACAGCAGCGCTTGGGTGGTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA',
            'ACTGCGGGGGTAAGAGGTTGTGTCCACCATGGCCTGGACTCCTCTCCTCCTCCTGTTCCTCTCTCACTGCACAGGTTCCCTCTCGCAGGCTGTGCTGACTCAGCCGTCTTCCCTCTCTGCATCTCCTGGAGCATCAGCCAGTCTCACCTGCACCTTGCGCAGTGGCATCAATGTTGGTACCTACAGGATATACTGGTACCAGCAGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGACTCAGATAAGCAGCAGGGCTCTGGAGTCCCCAGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTACTGTATGATTTGGCACAGCAGCGCTTCTTTGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA',
            'TGTATGATTTGGCACAGCAGCGCTTGGGTGGTC', 'CMIWHSSAWVV', '33', '11',
            '29', '397', '', '', '394', '431', '432', '642', '13160', '90', 'T'
        ],
        [
            'AAACCTGTCGAGAACG-1', '', 'AAACCTGTCGAGAACG-1_contig_2',
            'GGGAGCATCACCCAGCAACCACATCTGTCCTCTAGAGAATCCCCTGAGAGCTCCGTTCCTCACCATGGACTGGACCTGGAGGATCCTCTTCTTGGTGGCAGCAGCCACAGGAGCCCACTCGCAGGTGCAACTGGTGCAGTCTGGGGGTGAGGTAAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTCACCGACTACTTCATACAGTGGCTGCGACACGCCCCTGGACAGGGGCTTGATTGGATGGGTTTAATCAACCCTAACAGTGGTGACACCAACTATGCACAGAAGTTTCAGGGCAGAGTCACCATGACCAGGGACACGTCCATCAGTACAGCCTACATGGAACTGAGCAGCCTGAGATCTGACGACACGGCCGTATATTACTGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'MDWTWRILFLVAAATGAHSQVQLVQSGGEVKKPGASVKVSCKASGYTFTDYFIQWLRHAPGQGLDWMGLINPNSGDTNYAQKFQGRVTMTRDTSISTAYMELSSLRSDDTAVYYCAREIEGDGVFEIWGQGTMVTVSSGSASAPTLFPLVSCENSPSDTSSV',
            'T', 'F', 'IGHV1-2', '64S353M133S', '', '', 'IGHJ3', '429S50M71S',
            'IGHM', '479S71M',
            'GGGAGCATCACCCAGCAACCACATCTGTCCTCTAGAGAATCCCCTGAGAGCTCCGTTCCTCACCATGGACTGGACCTGGAGGATCCTCTTCTTGGTGGCAGCAGCCACAGGAGCCCACTCGCAGGTGCAACTGGTGCAGTCTGGGGGTGAGGTAAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTCACCGACTACTTCATACAGTGGCTGCGACACGCCCCTGGACAGGGGCTTGATTGGATGGGTTTAATCAACCCTAACAGTGGTGACACCAACTATGCACAGAAGTTTCAGGGCAGAGTCACCATGACCAGGGACACGTCCATCAGTACAGCCTACATGGAACTGAGCAGCCTGAGATCTGACGACACGGCCGTATATTACTGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'GAGAGCATCACCCAGCAACCACATCTGTCCTCTAGAGAATCCCCTGAGAGCTCCGTTCCTCACCATGGACTGGACCTGGAGGATCCTCTTCTTGGTGGCAGCAGCCACAGGAGCCCACTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTCACCGGCTACTATATGCACTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGATGGATCAACCCTAACAGTGGTGGCACAAACTATGCACAGAAGTTTCAGGGCTGGGTCACCATGACCAGGGACACGTCCATCAGCACAGCCTACATGGAGCTGAGCAGGCTGAGATCTGACGACACGGCCGTGTATTACTGTGCGAGAGATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'TGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGG', 'CAREIEGDGVFEIW',
            '42', '14', '65', '417', '', '', '430', '479', '480', '550',
            '5080', '47', 'T'
        ],
        [
            'AAACCTGTCTTGAGAC-1', '', 'AAACCTGTCTTGAGAC-1_contig_1',
            'AGGAGTCAGACCCAGTCAGGACACAGCATGGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            'MDMRVPAQLLGLLLLWLPGARCVIWMTQSPSLLSASTGDRVTISCRMSQGISSYLAWYQQKPGKAPELLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSFPYTFGQGTKLEIKRTVAAPSVFIFPPSDEQLKSGTASVVCLLNNFYPREAKVQWKVDN',
            'T', 'F', 'IGKV1D-8', '27S353M171S', '', '', 'IGKJ2',
            '376S39M136S', 'IGKC', '415S136M',
            'AGGAGTCAGACCCAGTCAGGACACAGCATGGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            'GGGACACCTGGGGACACTGAGCTGGTGCTGAGTTACTGAGATGAGCCAGCTCTGCAGCTGTGCCCAGTCAGCCCCATCCCCTGCTCATTTGCATGTTCCCAGAGCACAACCTCCTGCACTGAAGCCTTATTAATAGGCTGGCCACACTTCATGCAGGAGTCAGACCCAGTCAGGACACAGCATGGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCTCCTGTGCAGTTTTGGCCAGGGGACCAAGCTGGAGATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            'TGTCAACAGTATTATAGTTTCCCGTACACTTTT', 'CQQYYSFPYTF', '33', '11',
            '28', '380', '', '', '377', '415', '416', '551', '2813', '22', 'T'
        ],
        [
            'AAACCTGTCTTGAGAC-1', '', 'AAACCTGTCTTGAGAC-1_contig_2',
            'GGAGTCTCCCTCACCGCCCAGCTGGGATCTCAGGGCTTCATTTTCTGTCCTCCACCATCATGGGGTCAACCGCCATCCTCGCCCTCCTCCTGGCTGTTCTCCAAGGAGTCTGTGCCGAGGTGCAGCTGGTGCAGTCTGGAGCAGAGGTGAAAAAGCCGGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTTACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGTGACTCTGATACCAGATACAGCCCGTCCTTCCAAGGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'MGSTAILALLLAVLQGVCAEVQLVQSGAEVKKPGESLKISCKGSGYSFTSYWIGWVRQMPGKGLEWMGIIYPGDSDTRYSPSFQGQVTISADKSISTAYLQWSSLKASDTAMYYCARHIRGNRFGNDAFDIWGQGTMVTVSSGSASAPTLFPLVSCENSPSDTSSV',
            'T', 'F', 'IGHV5-51', '59S353M145S', '', '', 'IGHJ3', '436S50M71S',
            'IGHM', '486S71M',
            'GGAGTCTCCCTCACCGCCCAGCTGGGATCTCAGGGCTTCATTTTCTGTCCTCCACCATCATGGGGTCAACCGCCATCCTCGCCCTCCTCCTGGCTGTTCTCCAAGGAGTCTGTGCCGAGGTGCAGCTGGTGCAGTCTGGAGCAGAGGTGAAAAAGCCGGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTTACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGTGACTCTGATACCAGATACAGCCCGTCCTTCCAAGGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'TGAGTCTCCCTCACTGCCCAGCTGGGATCTCAGGGCTTCATTTTCTGTCCTCCACCATCATGGGGTCAACCGCCATCCTCGCCCTCCTCCTGGCTGTTCTCCAAGGAGTCTGTGCCGAGGTGCAGCTGGTGCAGTCTGGAGCAGAGGTGAAAAAGCCCGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTTACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGTGACTCTGATACCAGATACAGCCCGTCCTTCCAAGGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'TGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGG',
            'CARHIRGNRFGNDAFDIW', '54', '18', '60', '412', '', '', '437',
            '486', '487', '557', '8292', '80', 'T'
        ],
        [
            'AAACGGGAGCGACGTA-1', '', 'AAACGGGAGCGACGTA-1_contig_1',
            'GGGGACTTTCTGAGACTCATGGACCTCCTGCACAAGAACATGAAACACCTGTGGTTCTTCCTCCTCCTGGTGGCAGCTCCCAGATGGGTCCTGTCCCAGGTGCAGCTGCAGGAGTCGGGCCCAGGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATCAGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGTGGGAGCACCAACTACAACCCCTCCCTCAAGAGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'MKHLWFFLLLVAAPRWVLSQVQLQESGPGLVKPSETLSLTCTVSGGSISSYYWSWIRQPAGKGLEWIGRIYTSGSTNYNPSLKSRVTMSVDTSKNQFSLKLSSVTAADTAVYYCARVGYRAAAGTDAFDIWGQGTMVTVSSGSASAPTLFPLVSCENSPSDTSSV',
            'T', 'F', 'IGHV4-59', '39S350M145S', '', '', 'IGHJ3', '413S50M71S',
            'IGHM', '463S71M',
            'GGGGACTTTCTGAGACTCATGGACCTCCTGCACAAGAACATGAAACACCTGTGGTTCTTCCTCCTCCTGGTGGCAGCTCCCAGATGGGTCCTGTCCCAGGTGCAGCTGCAGGAGTCGGGCCCAGGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATCAGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGTGGGAGCACCAACTACAACCCCTCCCTCAAGAGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'ATGAAACATCTGTGGTTCTTCCTTCTCCTGGTGGCAGCTCCCAGATGGGTCCTGTCCCAGGTGCAGCTGCAGGAGTCGGGCCCAGGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATCAGTAGTTACTACTGGAGCTGGATCCGGCAGCCCCCAGGGAAGGGACTGGAGTGGATTGGGTATATCTATTACAGTGGGAGCACCAACTACAACCCCTCCCTCAAGAGTCGAGTCACCATATCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCTGCGGACACGGCCGTGTATTACTGTGCGAGAGATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'TGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGG',
            'CARVGYRAAAGTDAFDIW', '54', '18', '40', '389', '', '', '414',
            '463', '464', '534', '1235', '18', 'T'
        ],
        [
            'AAACGGGAGCGACGTA-1', '', 'AAACGGGAGCGACGTA-1_contig_2',
            'AGCTGTGGGCTCAGAAGCAGAGTTCTGGGGTGTCTCCACCATGGCCTGGACCCCTCTCTGGCTCACTCTCCTCACTCTTTGCATAGGTTCTGTGGTTTCTTCTGAGCTGACTCAGGACCCTGCTGTGTCTGTGGCCTTGGGACAGACAGTCAGGATCACATGCCAAGGAGACAGCCTCAGAAGCTATTATGCAAGCTGGTACCAGCAGAAGCCAGGACAGGCCCCTGTACTTGTCATCTATGGTAAAAACAACCGGCCCTCAGGGATCCCAGACCGATTCTCTGGCTCCAGCTCAGGAAACACAGCTTCCTTGACCATCACTGGGGCTCAGGCGGAAGATGAGGCTGACTATTACTGTAACTCCCGGGACAGCAGTGGTAACCATGTGGTATTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA',
            'MAWTPLWLTLLTLCIGSVVSSELTQDPAVSVALGQTVRITCQGDSLRSYYASWYQQKPGQAPVLVIYGKNNRPSGIPDRFSGSSSGNTASLTITGAQAEDEADYYCNSRDSSGNHVVFGGGTKLTVLGQPKAAPSVTLFPPSSEELQANKATLVCLISDFYPGAVTVAWKADSSPVKAGVETTTPSKQSNNKYAASS',
            'T', 'F', 'IGLV3-19', '40S337M256S', '', '', 'IGLJ2',
            '384S38M211S', 'IGLC2', '422S211M',
            'AGCTGTGGGCTCAGAAGCAGAGTTCTGGGGTGTCTCCACCATGGCCTGGACCCCTCTCTGGCTCACTCTCCTCACTCTTTGCATAGGTTCTGTGGTTTCTTCTGAGCTGACTCAGGACCCTGCTGTGTCTGTGGCCTTGGGACAGACAGTCAGGATCACATGCCAAGGAGACAGCCTCAGAAGCTATTATGCAAGCTGGTACCAGCAGAAGCCAGGACAGGCCCCTGTACTTGTCATCTATGGTAAAAACAACCGGCCCTCAGGGATCCCAGACCGATTCTCTGGCTCCAGCTCAGGAAACACAGCTTCCTTGACCATCACTGGGGCTCAGGCGGAAGATGAGGCTGACTATTACTGTAACTCCCGGGACAGCAGTGGTAACCATGTGGTATTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA',
            'AGCTGTGGGCTCAGAAGCAGAGTTCTGGGGTGTCTCCACCATGGCCTGGACCCCTCTCTGGCTCACTCTCCTCACTCTTTGCATAGGTTCTGTGGTTTCTTCTGAGCTGACTCAGGACCCTGCTGTGTCTGTGGCCTTGGGACAGACAGTCAGGATCACATGCCAAGGAGACAGCCTCAGAAGCTATTATGCAAGCTGGTACCAGCAGAAGCCAGGACAGGCCCCTGTACTTGTCATCTATGGTAAAAACAACCGGCCCTCAGGGATCCCAGACCGATTCTCTGGCTCCAGCTCAGGAAACACAGCTTCCTTGACCATCACTGGGGCTCAGGCGGAAGATGAGGCTGACTATTACTGTAACTCCCGGGACAGCAGTGTGTGGTATTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA',
            'TGTAACTCCCGGGACAGCAGTGGTAACCATGTGGTATTC', 'CNSRDSSGNHVVF', '39',
            '13', '41', '377', '', '', '385', '422', '423', '633', '1344',
            '14', 'T'
        ],
    ],
        columns=[
        'cell_id', 'clone_id', 'sequence_id', 'sequence',
        'sequence_aa', 'productive', 'rev_comp', 'v_call',
        'v_cigar', 'd_call', 'd_cigar', 'j_call',
        'j_cigar', 'c_call', 'c_cigar',
        'sequence_alignment', 'germline_alignment',
        'junction', 'junction_aa', 'junction_length',
        'junction_aa_length', 'v_sequence_start',
        'v_sequence_end', 'd_sequence_start',
        'd_sequence_end', 'j_sequence_start',
        'j_sequence_end', 'c_sequence_start',
        'c_sequence_end', 'consensus_count',
        'duplicate_count', 'is_cell'
    ])
    return (airr)


@pytest.fixture
def airr_reannotated():
    annotated = pd.DataFrame([
        [
            'AAACCTGTCATATCGG-1_contig_1',
            'TGGGGAGGAGTCAGTCCCAACCAGGACACGGCCTGGACATGAGGGTCCCTGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCTCAGGTGCCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTGGGAGACAGAGTCACCATCACTTGCCAGGCGACACAAGACATTAACAATTATGTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAACTCCTGATCTACGATGCATTGAATTTAGAAATAGGGGTCCCATCAAGATTCAGTGGAAGAGGGTCTGGGACAGTCTTTATTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACATACTACTGTCAACAATATGACGAACTTCCCGTCACTTTCGGCGGAGGGACCAATGTGGAAATGAGACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            'F', 'T', 'IGKV1-33*01,IGKV1D-33*01', '', 'IGKJ4*01',
            'GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTGGGAGACAGAGTCACCATCACTTGCCAGGCGACACAAGACATT..................AACAATTATGTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAACTCCTGATCTACGATGCA.....................TTGAATTTAGAAATAGGGGTCCCA...TCAAGATTCAGTGGAAGAGGG......TCTGGGACAGTCTTTATTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACATACTACTGTCAACAATATGACGAACTTCCCGTCACTTTCGGCGGAGGGACCAATGTGGA',
            'GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTAGGAGACAGAGTCACCATCACTTGCCAGGCGAGTCAGGACATT..................AGCAACTATTTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAGCTCCTGATCTACGATGCA.....................TCCAATTTGGAAACAGGGGTCCCA...TCAAGGTTCAGTGGAAGTGGA......TCTGGGACAGATTTTACTTTCACCATCAGCAGCCTGCAGCCTGAAGATATTGCAACATATTACTGTCAACAGTATGATAATCTCCCNNTCACTTTCGGCGGAGGGACCAAGGTGGA',
            'TGTCAACAATATGACGAACTTCCCGTCACTTTC', 'CQQYDELPVTF',
            '98S44=1X31=2X2=1X7=1X3=1X3=1X37=1X19=2X5=1X4=1X15=1X11=1X2=1X10=2X4=1X1=1X29=1X10=1X11=1X5=2X1=1X2=1X2=',
            'TEMP', '384S2N22=1X5=', 'F', 'T', 'IGK', '33', '2', 'TEMP', '99',
            '382', '1', '332', 'TEMP', 'TEMP', 'TEMP', 'TEMP', '385', '412',
            '3', '30', '363', '0.90845', '3.60E-102', 'TEMP', 'TEMP', 'TEMP',
            '48.8', '0.96429', '1.23E-09',
            'GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTGGGAGACAGAGTCACCATCACTTGCCAGGCGACA',
            'GTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAACTCCTGATCTAC',
            'AATTTAGAAATAGGGGTCCCA...TCAAGATTCAGTGGAAGAGGG......TCTGGGACAGTCTTTATTCTCACCATCAGCAGCCTGCAGCCTGAAGATGTTGCAACATACTACTGT',
            'TTCGGCGGAGGGACCAATGTGGA', 'CAAGACATT..................AACAATTAT',
            'GATGCA.....................TTG', 'CAACAATATGACGAACTTCCCGTCACT',
            'AAACCTGTCATATCGG-1', 'IGKC', '9139', '68', 'IGKV1-8', 'TEMP',
            'IGKJ4', 'TGTCAACAATATGACGAACTTCCCGTCACTTTC', 'CQQYDELPVTF',
            'GACATCCAGATGACCCAGTCTCCATCCTCCCTGTCTGCATCTGTAGGAGACAGAGTCACCATCACTTGCCAGGCGAGTCAGGACATT..................AGCAACTATTTAAATTGGTATCAGCAGAAACCAGGGAAAGCCCCTAAGCTCCTGATCTACGATGCA.....................TCCAATTTGGAAACAGGGGTCCCA...TCAAGGTTCAGTGGAAGTGGA......TCTGGGACAGATTTTACTTTCACCATCAGCAGCCTGCAGCCTGAAGATATTGCAACATATTACTGTCAACAGTATGATAATCTCCCNNTCACTTTCGGCGGAGGGACCAAGGTGGA',
            'IGKV1-33*01,IGKV1D-33*01', 'sample1',
            'CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            'CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            '420', '556', '254.111', '100', '5.00E-70', 'IGKC', '11',
            'DIQMTQSPSSLSASVGDRVTITCQAT', 'VNWYQQKPGKAPKLLIY',
            'NLEIGVPSRFSGRGSGTVFILTISSLQPEDVATYYC', 'FGGGTNV', 'QDINNY', 'DAL',
            'QQYDELPVT',
            'DIQMTQSPSSLSASVGDRVTITCQATQDINNYVNWYQQKPGKAPKLLIYDALNLEIGVPSRFSGRGSGTVFILTISSLQPEDVATYYCQQYDELPVTFGGGTNV',
            'DIQMTQSPSSLSASVGDRVTITCQATQDINNYVNWYQQKPGKAPKLLIYDALNLEIGVPSRFSGRGSGTVFILTISSLQPEDVATYYCQQYDELP',
            'TEMP', 'TFGGGTNV', '0.087096774', '27'
        ],
        [
            'AAACCTGTCCGTTGTC-1_contig_2',
            'ATCACATAACAACCACATTCCTCCTCTAAAGAAGCCCCTGGGAGCACAGCTCATCACCATGGACTGGACCTGGAGGTTCCTCTTTGTGGTGGCAGCAGCTACAGGTGTCCAGTCCCAGGTGCAGCTGGTGCAGTCTGGGGCTGAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTCAGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATCTTTGGTACAGCAAACTACGCACAGAAGTTCCAGGGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'F', 'T', 'IGHV1-69*01,IGHV1-69D*01', 'IGHD3-22*01', 'IGHJ3*02',
            'CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGANNNNGTATTACTATGATAGTAGTGGTTATTACNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG',
            'CATTYYYDSSGYYQNDAFDIW', '115S292=', '411S28=', '444S50=', 'F',
            'T', 'IGH', '63', '4', '5', '116', '407', '1', '316', '412', '439',
            '1', '28', '445', '494', '1', '50', '456', '1', '2.64E-130',
            '54.5', '1', '1.35E-11', '96.8', '1', '4.25E-24',
            'CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCT',
            'ATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGG',
            'AACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGT',
            'TGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'GGAGGCACCTTC............AGCAGCTATGCT',
            'ATCATCCCTATC......TTTGGTACAGCA',
            'GCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATC',
            'AAACCTGTCCGTTGTC-1', 'IGHM', '4161', '51', 'IGHV1-69D',
            'IGHD3-22', 'IGHJ3',
            'TGTGCGACTACGTATTACTATGATAGTAGTGGTTATTACCAGAATGATGCTTTTGATATCTGG',
            'CATTYYYDSSGYYQNDAFDIW',
            'CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGTCCTCGGTGAAGGTCTCCTGCAAGGCTTCTGGAGGCACCTTC............AGCAGCTATGCTATCAGCTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGAGGGATCATCCCTATC......TTTGGTACAGCAAACTACGCACAGAAGTTCCAG...GGCAGAGTCACGATTACCGCGGACGAATCCACGAGCACAGCCTACATGGAGCTGAGCAGCCTGAGATCTGAGGACACGGCCGTGTATTACTGTGCGANNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'IGHV1-69*01,IGHV1-69D*01', 'sample1',
            'GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            '494', '565', '134.079', '100', '7.00E-34', 'IGHM', '21',
            'QVQLVQSGAEVKKPGSSVKVSCKAS', 'ISWVRQAPGQGLEWMGG',
            'NYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYC', 'WGQGTMVTVSS',
            'GGTFSSYA', 'IIPIFGTA', 'ATTYYYDSSGYYQNDAFDI',
            'QVQLVQSGAEVKKPGSSVKVSCKASGGTFSSYAISWVRQAPGQGLEWMGGIIPIFGTANYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYCATTYYYDSSGYYQNDAFDIWGQGTMVTVSS',
            'QVQLVQSGAEVKKPGSSVKVSCKASGGTFSSYAISWVRQAPGQGLEWMGGIIPIFGTANYAQKFQGRVTITADESTSTAYMELSSLRSEDTAVYYCA',
            'YYYDSSGYY', 'DAFDIWGQGTMVTVSS', '0', '0'
        ],
        [
            'AAACCTGTCCGTTGTC-1_contig_1',
            'AGGAGTCAGACCCTGTCAGGACACAGCATAGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGCGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            'F', 'T', 'IGKV1-8*01', 'TEMP', 'IGKJ1*01',
            'GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC',
            'GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC',
            'TGTCAACAGTATTATAGTTACCCTCGGACGTTC', 'CQQYYSYPRTF', '93S286=',
            'TEMP', '379S2N36=', 'F', 'T', 'IGK', '33', '0', 'TEMP', '94',
            '379', '1', '334', 'TEMP', 'TEMP', 'TEMP', 'TEMP', '380', '415',
            '3', '38', '447', '1', '1.68E-127', 'TEMP', 'TEMP', 'TEMP', '69.9',
            '1', '5.25E-16',
            'GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGT',
            'TTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTAT',
            'ACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGT',
            'TTCGGCCAAGGGACCAAGGTGGAAATCAAAC',
            'CAGGGTATT..................AGCAGTTAT',
            'GCTGCA.....................TCC', 'CAACAGTATTATAGTTACCCTCGGACG',
            'AAACCTGTCCGTTGTC-1', 'IGKC', '5679', '43', 'IGKV1-8', 'TEMP',
            'IGKJ1', 'TGTCAACAGTATTATAGTTACCCTCGGACGTTC', 'CQQYYSYPRTF',
            'GCCATCCGGATGACCCAGTCTCCATCCTCATTCTCTGCATCTACAGGAGACAGAGTCACCATCACTTGTCGGGCGAGTCAGGGTATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTAAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGCGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGCTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTACCCTCGGACGTTCGGCCAAGGGACCAAGGTGGAAATCAAAC',
            'IGKV1-8*01', 'sample1',
            'CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            'CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            '415', '551', '254.111', '100', '5.00E-70', 'IGKC', '11',
            'AIRMTQSPSSFSASTGDRVTITCRAS', 'LAWYQQKPGKAPKLLIY',
            'TLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYC', 'FGQGTKVEIK', 'QGISSY',
            'AAS', 'QQYYSYPRT',
            'AIRMTQSPSSFSASTGDRVTITCRASQGISSYLAWYQQKPGKAPKLLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSYPRTFGQGTKVEIK',
            'AIRMTQSPSSFSASTGDRVTITCRASQGISSYLAWYQQKPGKAPKLLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSYP',
            'TEMP', 'TFGQGTKVEIK', '0', '0'
        ],
        [
            'AAACCTGTCGAGAACG-1_contig_2',
            'GGGAGCATCACCCAGCAACCACATCTGTCCTCTAGAGAATCCCCTGAGAGCTCCGTTCCTCACCATGGACTGGACCTGGAGGATCCTCTTCTTGGTGGCAGCAGCCACAGGAGCCCACTCGCAGGTGCAACTGGTGCAGTCTGGGGGTGAGGTAAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTCACCGACTACTTCATACAGTGGCTGCGACACGCCCCTGGACAGGGGCTTGATTGGATGGGTTTAATCAACCCTAACAGTGGTGACACCAACTATGCACAGAAGTTTCAGGGCAGAGTCACCATGACCAGGGACACGTCCATCAGTACAGCCTACATGGAACTGAGCAGCCTGAGATCTGACGACACGGCCGTATATTACTGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'F', 'T', 'IGHV1-2*02', 'IGHD3-16*01,IGHD3-16*02,IGHD4-17*01',
            'IGHJ3*02',
            'CAGGTGCAACTGGTGCAGTCTGGGGGT...GAGGTAAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTC............ACCGACTACTTCATACAGTGGCTGCGACACGCCCCTGGACAGGGGCTTGATTGGATGGGTTTAATCAACCCTAAC......AGTGGTGACACCAACTATGCACAGAAGTTTCAG...GGCAGAGTCACCATGACCAGGGACACGTCCATCAGTACAGCCTACATGGAACTGAGCAGCCTGAGATCTGACGACACGGCCGTATATTACTGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTC............ACCGGCTACTATATGCACTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGATGGATCAACCCTAAC......AGTGGTGGCACAAACTATGCACAGAAGTTTCAG...GGCAGGGTCACCATGACCAGGGACACGTCCATCAGCACAGCCTACATGGAGCTGAGCAGGCTGAGATCTGACGACACGGCCGTGTATTACTGTGCGAGAGANNNNNNGGGGGANNNNNNTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'TGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGG', 'CAREIEGDGVFEIW',
            '121S8=1X16=1X6=1X58=1X5=2X2=1X2=1X3=1X7=1X11=1X8=1X8=1X1=2X19=1X3=1X26=1X29=1X14=1X8=1X23=1X17=',
            '423S17N6=', '435S6N6=1X37=', 'F', 'T', 'IGH', '42', '6', '6',
            '122', '417', '1', '320', '424', '429', '18', '23', '436', '479',
            '7', '50', '394', '0.92568', '1.48E-111', '12.2', '1', '71',
            '79.5', '0.97727', '6.69E-19',
            'CAGGTGCAACTGGTGCAGTCTGGGGGT...GAGGTAAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCT',
            'ATACAGTGGCTGCGACACGCCCCTGGACAGGGGCTTGATTGGATGGGTTTA',
            'AACTATGCACAGAAGTTTCAG...GGCAGAGTCACCATGACCAGGGACACGTCCATCAGTACAGCCTACATGGAACTGAGCAGCCTGAGATCTGACGACACGGCCGTATATTACTGT',
            'TGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'GGATACACCTTC............ACCGACTACTTC',
            'ATCAACCCTAAC......AGTGGTGACACC',
            'GCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATC', 'AAACCTGTCGAGAACG-1',
            'IGHM', '5080', '47', 'IGHV1-2', 'TEMP', 'IGHJ3',
            'TGTGCGAGAGAGATAGAGGGGGACGGTGTTTTTGAAATCTGG', 'CAREIEGDGVFEIW',
            'CAGGTGCAGCTGGTGCAGTCTGGGGCT...GAGGTGAAGAAGCCTGGGGCCTCAGTGAAGGTCTCCTGCAAGGCTTCTGGATACACCTTC............ACCGGCTACTATATGCACTGGGTGCGACAGGCCCCTGGACAAGGGCTTGAGTGGATGGGATGGATCAACCCTAAC......AGTGGTGGCACAAACTATGCACAGAAGTTTCAG...GGCAGGGTCACCATGACCAGGGACACGTCCATCAGCACAGCCTACATGGAGCTGAGCAGGCTGAGATCTGACGACACGGCCGTGTATTACTGTGCGAGAGANNNNNNNNNNNNNNNNNNTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'IGHV1-2*02', 'sample1',
            'GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            '479', '550', '134.079', '100', '7.00E-34', 'IGHM', '14',
            'QVQLVQSGGEVKKPGASVKVSCKAS', 'IQWLRHAPGQGLDWMGL',
            'NYAQKFQGRVTMTRDTSISTAYMELSSLRSDDTAVYYC', 'WGQGTMVTVSS',
            'GYTFTDYF', 'INPNSGDT', 'AREIEGDGVFEI',
            'QVQLVQSGGEVKKPGASVKVSCKASGYTFTDYFIQWLRHAPGQGLDWMGLINPNSGDTNYAQKFQGRVTMTRDTSISTAYMELSSLRSDDTAVYYCAREIEGDGVFEIWGQGTMVTVSS',
            'QVQLVQSGGEVKKPGASVKVSCKASGYTFTDYFIQWLRHAPGQGLDWMGLINPNSGDTNYAQKFQGRVTMTRDTSISTAYMELSSLRSDDTAVYYCAR',
            'G', 'FEIWGQGTMVTVSS', '0.064896755', '22'
        ],
        [
            'AAACCTGTCGAGAACG-1_contig_1',
            'ACTGTGGGGGTAAGAGGTTGTGTCCACCATGGCCTGGACTCCTCTCCTCCTCCTGTTCCTCTCTCACTGCACAGGTTCCCTCTCGCAGGCTGTGCTGACTCAGCCGTCTTCCCTCTCTGCATCTCCTGGAGCATCAGGCAGGCTCACCTGCACCTTACGCAGTGACATCAATGTTGGTACGTACAGGATATATTGGTACCAGCGGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGACTCAGATAAGCAGCAGGGCTCTGGAGTCCCCAGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTATTGTATGATTTGGCACAGCAGCGCTTGGGTGGTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA',
            'F', 'T', 'IGLV5-45*02', 'TEMP', 'IGLJ3*02',
            'CAGGCTGTGCTGACTCAGCCGTCTTCC...CTCTCTGCATCTCCTGGAGCATCAGGCAGGCTCACCTGCACCTTACGCAGTGACATCAATGTT.........GGTACGTACAGGATATATTGGTACCAGCGGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGAC.........TCAGATAAGCAGCAGGGCTCTGGAGTCCCC...AGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTATTGTATGATTTGGCACAGCAGCGCTTGGGTGGTCGGCGGAGGGACCAAGCTGACCGTCCTAG',
            'CAGGCTGTGCTGACTCAGCCGTCTTCC...CTCTCTGCATCTCCTGGAGCATCAGCCAGTCTCACCTGCACCTTACGCAGTGGCATCAATGTT.........GGTACCTACAGGATATACTGGTACCAGCAGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGAC.........TCAGATAAGCAGCAGGGCTCTGGAGTCCCC...AGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTACTGTATGATTTGGCACAGCAGCGCTTGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAG',
            'TGTATGATTTGGCACAGCAGCGCTTGGGTGGTC', 'CMIWHSSAWVV',
            '85S52=1X3=1X22=1X15=1X11=1X10=1X165=1X25=', 'TEMP',
            '395S2N5=1X30=', 'F', 'T', 'IGL', '33', '0', 'TEMP', '86', '395',
            '1', '334', 'TEMP', 'TEMP', 'TEMP', 'TEMP', '396', '431', '3',
            '38', '463', '0.97742', '4.02E-132', 'TEMP', 'TEMP', 'TEMP',
            '64.1', '0.97222', '3.34E-14',
            'CAGGCTGTGCTGACTCAGCCGTCTTCC...CTCTCTGCATCTCCTGGAGCATCAGGCAGGCTCACCTGCACCTTACGC',
            'ATATATTGGTACCAGCGGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGG',
            'CAGCAGGGCTCTGGAGTCCCC...AGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTATTGT',
            'GTCGGCGGAGGGACCAAGCTGACCGTCCTAG',
            'AGTGACATCAATGTT.........GGTACGTACAGG',
            'TACAAATCAGAC.........TCAGATAAG', 'ATGATTTGGCACAGCAGCGCTTGGGTG',
            'AAACCTGTCGAGAACG-1', 'IGLC3', '13160', '90', 'IGLV5-45', 'TEMP',
            'IGLJ3', 'TGTATGATTTGGCACAGCAGCGCTTGGGTGGTC', 'CMIWHSSAWVV',
            'CAGGCTGTGCTGACTCAGCCGTCTTCC...CTCTCTGCATCTCCTGGAGCATCAGCCAGTCTCACCTGCACCTTACGCAGTGGCATCAATGTT.........GGTACCTACAGGATATACTGGTACCAGCAGAAGCCAGGGAGTCCTCCCCAGTATCTCCTGAGGTACAAATCAGAC.........TCAGATAAGCAGCAGGGCTCTGGAGTCCCC...AGCCGCTTCTCTGGATCCAAAGATGCTTCGGCCAATGCAGGGATTTTACTCATCTCTGGGCTCCAGTCTGAGGATGAGGCTGACTATTACTGTATGATTTGGCACAGCAGCGCTTGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAG',
            'IGLV5-45*02', 'sample1',
            'GGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA',
            'GGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCACCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA',
            '431', '642', '392.61', '100', '1.00E-111', 'IGLC3', '11',
            'QAVLTQPSSLSASPGASGRLTCTLR', 'IYWYQRKPGSPPQYLLR',
            'QQGSGVPSRFSGSKDASANAGILLISGLQSEDEADYYC', 'VGGGTKLTVL',
            'SDINVGTYR', 'YKSDSDK', 'MIWHSSAWV',
            'QAVLTQPSSLSASPGASGRLTCTLRSDINVGTYRIYWYQRKPGSPPQYLLRYKSDSDKQQGSGVPSRFSGSKDASANAGILLISGLQSEDEADYYCMIWHSSAWVVGGGTKLTVL',
            'QAVLTQPSSLSASPGASGRLTCTLRSDINVGTYRIYWYQRKPGSPPQYLLRYKSDSDKQQGSGVPSRFSGSKDASANAGILLISGLQSEDEADYYCMIWHSSA',
            'TEMP', 'VVGGGTKLTVL', '0.023188406', '8'
        ],
        [
            'AAACCTGTCTTGAGAC-1_contig_2',
            'GGAGTCTCCCTCACCGCCCAGCTGGGATCTCAGGGCTTCATTTTCTGTCCTCCACCATCATGGGGTCAACCGCCATCCTCGCCCTCCTCCTGGCTGTTCTCCAAGGAGTCTGTGCCGAGGTGCAGCTGGTGCAGTCTGGAGCAGAGGTGAAAAAGCCGGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTTACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGTGACTCTGATACCAGATACAGCCCGTCCTTCCAAGGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'F', 'T', 'IGHV5-51*01',
            'IGHD1-26*01,IGHD1/OR15-1a*01,IGHD1/OR15-1b*01', 'IGHJ3*02',
            'GAGGTGCAGCTGGTGCAGTCTGGAGCA...GAGGTGAAAAAGCCGGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTT............ACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGT......GACTCTGATACCAGATACAGCCCGTCCTTCCAA...GGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'GAGGTGCAGCTGGTGCAGTCTGGAGCA...GAGGTGAAAAAGCCCGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTT............ACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGT......GACTCTGATACCAGATACAGCCCGTCCTTCCAA...GGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACANNNNNGTGGGANNNNNNNNNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'TGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGG',
            'CARHIRGNRFGNDAFDIW', '116S41=1X254=', '417S6N6=', '436S50=', 'F',
            'T', 'IGH', '54', '5', '13', '117', '412', '1', '320', '418',
            '423', '7', '12', '437', '486', '1', '50', '459', '0.99662',
            '3.00E-131', '12.2', '1', '72', '96.8', '1', '4.19E-24',
            'GAGGTGCAGCTGGTGCAGTCTGGAGCA...GAGGTGAAAAAGCCGGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCT',
            'ATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATC',
            'AGATACAGCCCGTCCTTCCAA...GGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGT',
            'TGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'GGATACAGCTTT............ACCAGCTACTGG',
            'ATCTATCCTGGT......GACTCTGATACC',
            'GCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATC',
            'AAACCTGTCTTGAGAC-1', 'IGHM', '8292', '80', 'IGHV5-51', 'TEMP',
            'IGHJ3', 'TGTGCGAGACATATCCGTGGGAACAGATTTGGCAATGATGCTTTTGATATCTGG',
            'CARHIRGNRFGNDAFDIW',
            'GAGGTGCAGCTGGTGCAGTCTGGAGCA...GAGGTGAAAAAGCCCGGGGAGTCTCTGAAGATCTCCTGTAAGGGTTCTGGATACAGCTTT............ACCAGCTACTGGATCGGCTGGGTGCGCCAGATGCCCGGGAAAGGCCTGGAGTGGATGGGGATCATCTATCCTGGT......GACTCTGATACCAGATACAGCCCGTCCTTCCAA...GGCCAGGTCACCATCTCAGCCGACAAGTCCATCAGCACCGCCTACCTGCAGTGGAGCAGCCTGAAGGCCTCGGACACCGCCATGTATTACTGTGCGAGACANNNNNNNNNNNNNNNNNNNNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'IGHV5-51*01', 'sample1',
            'GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            '486', '557', '134.079', '100', '7.00E-34', 'IGHM', '18',
            'EVQLVQSGAEVKKPGESLKISCKGS', 'IGWVRQMPGKGLEWMGI',
            'RYSPSFQGQVTISADKSISTAYLQWSSLKASDTAMYYC', 'WGQGTMVTVSS',
            'GYSFTSYW', 'IYPGDSDT', 'ARHIRGNRFGNDAFDI',
            'EVQLVQSGAEVKKPGESLKISCKGSGYSFTSYWIGWVRQMPGKGLEWMGIIYPGDSDTRYSPSFQGQVTISADKSISTAYLQWSSLKASDTAMYYCARHIRGNRFGNDAFDIWGQGTMVTVSS',
            'EVQLVQSGAEVKKPGESLKISCKGSGYSFTSYWIGWVRQMPGKGLEWMGIIYPGDSDTRYSPSFQGQVTISADKSISTAYLQWSSLKASDTAMYYCAR',
            'G', 'DAFDIWGQGTMVTVSS', '0.002898551', '1'
        ],
        [
            'AAACCTGTCTTGAGAC-1_contig_1',
            'AGGAGTCAGACCCAGTCAGGACACAGCATGGACATGAGGGTCCCCGCTCAGCTCCTGGGGCTCCTGCTGCTCTGGCTCCCAGGTGCCAGATGTGTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATTAGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCATCCACTTTGCAAAGTGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAACGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            'F', 'T', 'IGKV1D-8*01', 'TEMP', 'IGKJ2*01',
            'GTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGTGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAAC',
            'GTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGTGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAAC',
            'TGTCAACAGTATTATAGTTTCCCGTACACTTTT', 'CQQYYSFPYTF', '93S284=',
            'TEMP', '377S1N38=', 'F', 'T', 'IGK', '33', '0', 'TEMP', '94',
            '377', '1', '332', 'TEMP', 'TEMP', 'TEMP', 'TEMP', '378', '415',
            '2', '39', '444', '1', '1.45E-126', 'TEMP', 'TEMP', 'TEMP', '73.7',
            '1', '3.65E-17',
            'GTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGT',
            'TTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTAT',
            'ACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGTGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGT',
            'TTTGGCCAGGGGACCAAGCTGGAGATCAAAC',
            'CAGGGCATT..................AGCAGTTAT',
            'GCTGCA.....................TCC', 'CAACAGTATTATAGTTTCCCGTACACT',
            'AAACCTGTCTTGAGAC-1', 'IGKC', '2813', '22', 'IGKV1D-8', 'TEMP',
            'IGKJ2', 'TGTCAACAGTATTATAGTTTCCCGTACACTTTT', 'CQQYYSFPYTF',
            'GTCATCTGGATGACCCAGTCTCCATCCTTACTCTCTGCATCTACAGGAGACAGAGTCACCATCAGTTGTCGGATGAGTCAGGGCATT..................AGCAGTTATTTAGCCTGGTATCAGCAAAAACCAGGGAAAGCCCCTGAGCTCCTGATCTATGCTGCA.....................TCCACTTTGCAAAGTGGGGTCCCA...TCAAGGTTCAGTGGCAGTGGA......TCTGGGACAGATTTCACTCTCACCATCAGTTGCCTGCAGTCTGAAGATTTTGCAACTTATTACTGTCAACAGTATTATAGTTTCCCGTACACTTTTGGCCAGGGGACCAAGCTGGAGATCAAAC',
            'IGKV1D-8*01', 'sample1',
            'CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            'CGAACTGTGGCTGCACCATCTGTCTTCATCTTCCCGCCATCTGATGAGCAGTTGAAATCTGGAACTGCCTCTGTTGTGTGCCTGCTGAATAACTTCTATCCCAGAGAGGCCAAAGTACAGTGGAAGGTGGATAACGC',
            '415', '551', '254.111', '100', '5.00E-70', 'IGKC', '11',
            'VIWMTQSPSLLSASTGDRVTISCRMS', 'LAWYQQKPGKAPELLIY',
            'TLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYC', 'FGQGTKLEIK', 'QGISSY',
            'AAS', 'QQYYSFPYT',
            'VIWMTQSPSLLSASTGDRVTISCRMSQGISSYLAWYQQKPGKAPELLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSFPYTFGQGTKLEIK',
            'VIWMTQSPSLLSASTGDRVTISCRMSQGISSYLAWYQQKPGKAPELLIYAASTLQSGVPSRFSGSGSGTDFTLTISCLQSEDFATYYCQQYYSFP',
            'TEMP', 'YTFGQGTKLEIK', '0', '0'
        ],
        [
            'AAACGGGAGCGACGTA-1_contig_1',
            'GGGGACTTTCTGAGACTCATGGACCTCCTGCACAAGAACATGAAACACCTGTGGTTCTTCCTCCTCCTGGTGGCAGCTCCCAGATGGGTCCTGTCCCAGGTGCAGCTGCAGGAGTCGGGCCCAGGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATCAGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGTGGGAGCACCAACTACAACCCCTCCCTCAAGAGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAGGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'F', 'T', 'IGHV4-4*07', 'IGHD6-13*01', 'IGHJ3*02',
            'CAGGTGCAGCTGCAGGAGTCGGGCCCA...GGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATC............AGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGT.........GGGAGCACCAACTACAACCCCTCCCTCAAG...AGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'CAGGTGCAGCTGCAGGAGTCGGGCCCA...GGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATC............AGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGT.........GGGAGCACCAACTACAACCCCTCCCTCAAG...AGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGNNNNNNNNNNAGCAGCAGCTGGTACTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'TGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGG',
            'CARVGYRAAAGTDAFDIW', '96S292=', '398S6N15=', '413S50=', 'F', 'T',
            'IGH', '54', '10', '0', '97', '388', '1', '319', '399', '413', '7',
            '21', '414', '463', '1', '50', '456', '1', '2.49E-130', '29.5',
            '1', '0.000427', '96.8', '1', '4.01E-24',
            'CAGGTGCAGCTGCAGGAGTCGGGCCCA...GGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCT',
            'TGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGT',
            'AACTACAACCCCTCCCTCAAG...AGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGT',
            'TGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'GGTGGCTCCATC............AGTAGTTACTAC',
            'ATCTATACCAGT.........GGGAGCACC',
            'GCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATC',
            'AAACGGGAGCGACGTA-1', 'IGHM', '1235', '18', 'IGHV4-59', 'TEMP',
            'IGHJ3', 'TGTGCGAGAGTAGGCTATAGAGCAGCAGCTGGTACTGATGCTTTTGATATCTGG',
            'CARVGYRAAAGTDAFDIW',
            'CAGGTGCAGCTGCAGGAGTCGGGCCCA...GGACTGGTGAAGCCTTCGGAGACCCTGTCCCTCACCTGCACTGTCTCTGGTGGCTCCATC............AGTAGTTACTACTGGAGCTGGATCCGGCAGCCCGCCGGGAAGGGACTGGAGTGGATTGGGCGTATCTATACCAGT.........GGGAGCACCAACTACAACCCCTCCCTCAAG...AGTCGAGTCACCATGTCAGTAGACACGTCCAAGAACCAGTTCTCCCTGAAGCTGAGCTCTGTGACCGCCGCGGACACGGCCGTGTATTACTGTGCGAGAGNNNNNNNNNNNNNNNNNNNNNNNNNTGATGCTTTTGATATCTGGGGCCAAGGGACAATGGTCACCGTCTCTTCAG',
            'IGHV4-4*07', 'sample1',
            'GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            'GGGAGTGCATCCGCCCCAACCCTTTTCCCCCTCGTCTCCTGTGAGAATTCCCCGTCGGATACGAGCAGCGTG',
            '463', '534', '134.079', '100', '7.00E-34', 'IGHM', '18',
            'QVQLQESGPGLVKPSETLSLTCTVS', 'WSWIRQPAGKGLEWIGR',
            'NYNPSLKSRVTMSVDTSKNQFSLKLSSVTAADTAVYYC', 'WGQGTMVTVSS',
            'GGSISSYY', 'IYTSGST', 'ARVGYRAAAGTDAFDI',
            'QVQLQESGPGLVKPSETLSLTCTVSGGSISSYYWSWIRQPAGKGLEWIGRIYTSGSTNYNPSLKSRVTMSVDTSKNQFSLKLSSVTAADTAVYYCARVGYRAAAGTDAFDIWGQGTMVTVSS',
            'QVQLQESGPGLVKPSETLSLTCTVSGGSISSYYWSWIRQPAGKGLEWIGRIYTSGSTNYNPSLKSRVTMSVDTSKNQFSLKLSSVTAADTAVYYCAR',
            'AAAGT', 'DAFDIWGQGTMVTVSS', '0', '0'
        ],
        [
            'AAACGGGAGCGACGTA-1_contig_2',
            'TCTGGCACCAGGGGTCCCTTCCAATATCAGCACCATGGCCTGGACTCCTCTCTTTCTGTTCCTCCTCACTTGCTGCCCAGGGTCCAATTCCCAGGCTGTGGTGACTCAGGAGCCCTCACTGACTGTGTCCCCAGGAGGGACAGTCACTCTCACCTGTGGCTCCAGCACTGGAGCTGTCACCAGTGGTCATTATCCCTACTGGTTCCAGCAGAAGCCTGGCCAAGCCCCCAGGACACTGATTTATGATACAAGCAACAAACACTCCTGGACACCTGCCCGGTTCTCAGGCTCCCTCCTTGGGGGCAAAGCTGCCCTGACCCTTTCGGGTGCGCAGCCTGAGGATGAGGCTGAGTATTACTGCTTGCTCTCCTATAGTGGTGCTAGGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAGGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA',
            'F', 'F', 'IGLV7-46*01', 'TEMP', 'IGLJ3*02',
            'CAGGCTGTGGTGACTCAGGAGCCCTCA...CTGACTGTGTCCCCAGGAGGGACAGTCACTCTCACCTGTGGCTCCAGCACTGGAGCTGTCACC.........AGTGGTCATTATCCCTACTGGTTCCAGCAGAAGCCTGGCCAAGCCCCCAGGACACTGATTTATGATACA.....................AGCAACAAACACTCCTGGACACCT...GCCCGGTTCTCAGGCTCCCTC......CTTGGGGGCAAAGCTGCCCTGACCCTTTCGGGTGCGCAGCCTGAGGATGAGGCTGAGTATTACTGCTTGCTCTCCTATAGTGGTGCTAGGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAG',
            'CAGGCTGTGGTGACTCAGGAGCCCTCA...CTGACTGTGTCCCCAGGAGGGACAGTCACTCTCACCTGTGGCTCCAGCACTGGAGCTGTCACC.........AGTGGTCATTATCCCTACTGGTTCCAGCAGAAGCCTGGCCAAGCCCCCAGGACACTGATTTATGATACA.....................AGCAACAAACACTCCTGGACACCT...GCCCGGTTCTCAGGCTCCCTC......CTTGGGGGCAAAGCTGCCCTGACCCTTTCGGGTGCGCAGCCTGAGGATGAGGCTGAGTATTACTGCTTGCTCTCCTATAGTGGTGCTCGGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAG',
            'TGCTTGCTCTCCTATAGTGGTGCTAGGGGTGTTC', 'CLLSYSGARGV', '91S291=1X2=',
            'TEMP', '385S3N35=', 'F', 'F', 'IGL', '34', '0', 'TEMP', '92',
            '385', '1', '336', 'TEMP', 'TEMP', 'TEMP', 'TEMP', '386', '420',
            '4', '38', '456', '0.9966', '2.97E-130', 'TEMP', 'TEMP', 'TEMP',
            '68', '1', '2.29E-15',
            'CAGGCTGTGGTGACTCAGGAGCCCTCA...CTGACTGTGTCCCCAGGAGGGACAGTCACTCTCACCTGTGGCTCCAGC',
            'CCCTACTGGTTCCAGCAGAAGCCTGGCCAAGCCCCCAGGACACTGATTTAT',
            'AACAAACACTCCTGGACACCT...GCCCGGTTCTCAGGCTCCCTC......CTTGGGGGCAAAGCTGCCCTGACCCTTTCGGGTGCGCAGCCTGAGGATGAGGCTGAGTATTACTGC',
            'TTCGGCGGAGGGACCAAGCTGACCGTCCTAG',
            'ACTGGAGCTGTCACC.........AGTGGTCATTAT',
            'GATACA.....................AGC', 'TTGCTCTCCTATAGTGGTGCTAGGGGTG',
            'AAACGGGAGCGACGTA-1', 'IGLC', '739', '8', 'IGLV7-46', 'TEMP',
            'IGLJ3', 'TEMP', 'TEMP',
            'CAGGCTGTGGTGACTCAGGAGCCCTCA...CTGACTGTGTCCCCAGGAGGGACAGTCACTCTCACCTGTGGCTCCAGCACTGGAGCTGTCACC.........AGTGGTCATTATCCCTACTGGTTCCAGCAGAAGCCTGGCCAAGCCCCCAGGACACTGATTTATGATACA.....................AGCAACAAACACTCCTGGACACCT...GCCCGGTTCTCAGGCTCCCTC......CTTGGGGGCAAAGCTGCCCTGACCCTTTCGGGTGCGCAGCCTGAGGATGAGGCTGAGTATTACTGCTTGCTCTCCTATAGTGGTGCTCGGGGTGTTCGGCGGAGGGACCAAGCTGACCGTCCTAG',
            'IGLV7-46*01', 'sample1',
            'GGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA',
            'GGTCAGCCCAAGGCTGCCCCCTCGGTCACTCTGTTCCCGCCCTCCTCTGAGGAGCTTCAAGCCAACAAGGCCACACTGGTGTGTCTCATAAGTGACTTCTACCCGGGAGCCGTGACAGTGGCCTGGAAGGCAGATAGCAGCCCCGTCAAGGCGGGAGTGGAGACCACCACACCCTCCAAACAAAGCAACAACAAGTACGCGGCCAGCAGCTA',
            '420', '631', '392.61', '100', '1.00E-111', 'IGLC2', '11',
            'QAVVTQEPSLTVSPGGTVTLTCGSS', 'PYWFQQKPGQAPRTLIY',
            'NKHSWTPARFSGSLLGGKAALTLSGAQPEDEAEYYC', 'RRRDQADRP', 'TGAVTSGHY',
            'DTS', 'LLSYSGARG',
            'QAVVTQEPSLTVSPGGTVTLTCGSSTGAVTSGHYPYWFQQKPGQAPRTLIYDTSNKHSWTPARFSGSLLGGKAALTLSGAQPEDEAEYYCLLSYSGARGVRRRDQADRP',
            'QAVVTQEPSLTVSPGGTVTLTCGSSTGAVTSGHYPYWFQQKPGQAPRTLIYDTSNKHSWTPARFSGSLLGGKAALTLSGAQPEDEAEYYCLLSYSGAR',
            'TEMP', 'GVRRRDQADRP', '0.003058104', '1'
        ],
    ],
        columns=[
        'sequence_id', 'sequence', 'rev_comp',
        'productive', 'v_call', 'd_call', 'j_call',
        'sequence_alignment', 'germline_alignment',
        'junction', 'junction_aa', 'v_cigar',
        'd_cigar', 'j_cigar', 'stop_codon',
        'vj_in_frame', 'locus', 'junction_length',
        'np1_length', 'np2_length',
        'v_sequence_start', 'v_sequence_end',
        'v_germline_start', 'v_germline_end',
        'd_sequence_start', 'd_sequence_end',
        'd_germline_start', 'd_germline_end',
        'j_sequence_start', 'j_sequence_end',
        'j_germline_start', 'j_germline_end',
        'v_score', 'v_identity', 'v_support',
                                 'd_score', 'd_identity', 'd_support',
                                 'j_score', 'j_identity', 'j_support', 'fwr1',
                                 'fwr2', 'fwr3', 'fwr4', 'cdr1', 'cdr2',
                                 'cdr3', 'cell_id', 'c_call',
                                 'consensus_count', 'umi_count', 'v_call_10x',
                                 'd_call_10x', 'j_call_10x', 'junction_10x',
                                 'junction_10x_aa',
                                 'germline_alignment_d_mask',
                                 'v_call_genotyped', 'sample_id',
                                 'c_sequence_alignment',
                                 'c_germline_alignment', 'c_sequence_start',
                                 'c_sequence_end', 'c_score', 'c_identity',
                                 'c_support', 'c_call_10x',
                                 'junction_aa_length', 'fwr1_aa', 'fwr2_aa',
                                 'fwr3_aa', 'fwr4_aa', 'cdr1_aa', 'cdr2_aa',
                                 'cdr3_aa', 'sequence_alignment_aa',
                                 'v_sequence_alignment_aa',
                                 'd_sequence_alignment_aa',
                                 'j_sequence_alignment_aa', 'mu_freq',
                                 'mu_count'
    ])
    return (annotated)


@pytest.fixture
def fasta_10x_tr1():
    """Standard cellranger fasta file to test the preprocessing."""
    seq = {
        'AACTGGTTCTTTAGTC-1_contig_2':
        'AGGCAGAAGGTGGTTGAGAGGCAGAGCTGCCCCTGAGTGAGCCATGCAGAGGATCTCCTCCCTCATCCATCTCTCTCTCTTCTGGGCAGGAGTCATGTCAGCCATTGAGTTGGTGCCTGAACACCAAACAGTGCCTGTGTCAATAGGGGTCCCTGCCACCCTCAGGTGCTCCATGAAAGGAGAAGCGATCGGTAACTACTATATCAACTGGTACAGGAAGACCCAAGGTAACACAATGACTTTCATATACCGAGAAAAGGACATCTATGGCCCTGGTTTCAAAGACAATTTCCAAGGTGACATTGATATTGCAAAGAACCTGGCTGTACTTAAGATACTTGCACCATCAGAGAGAGATGAAGGGTCTTACTACTGTGCCTGTGACATTTTAGGGGATAAGGCCGATAAACTCATCTTTGGAAAAGGAACCCGTGTGACTGTGGAACCAAGAAGTCAGCCTCATACCAAACCATCCGTTTTTGTCATGAAAAATGGAACAAATGTCGCTTGTCTGGTGAAGGA',
        'AAGACCTCACTACAGT-1_contig_1':
        'GACTGAACTTTGAGCTTCAGGCAGCACAACTCACATTTGTGCAAAGAGCTACATGCCACATGCTGTTCTCCAGCCTGCTGTGTGTATTTGTGGCCTTCAGCTACTCTGGATCAAGTGTGGCCCAGAAGGTTACTCAAGCCCAGTCATCAGTATCCATGCCAGTGAGGAAAGCAGTCACCCTGAACTGCCTGTATGAAACAAGTTGGTGGTCATATTATATTTTTTGGTACAAGCAACTTCCCAGCAAAGAGATGATTTTCCTTATTCGCCAGGGTTCTGATGAACAGAATGCAAAAAGTGGTCGCTATTCTGTCAACTTCAAGAAAGCAGCGAAATCCGTCGCCTTAACCATTTCAGCCTTACAGCTAGAAGATTCAGCAAAGTACTTTTGTGCTCTTGGGGAACCCTTCCTCCCAGGGTGGGGCGATAAACTCATCTTTGGAAAAGGAACCCGTGTGACTGTGGAACCAAGAAGTCAGCCTCATACCAAACCATCCGTTTTTGTCATGAAAAATGGAACAAATGTCGCTTGTCTGGTGAAGGA',
    }
    return (seq)


@pytest.fixture
def fasta_10x_tr2():
    """Standard cellranger fasta file to test the preprocessing."""
    seq = {
        'AAGACCTCACTACAGT-1_contig_1':
        'GACTGAACTTTGAGCTTCAGGCAGCACAACTCACATTTGTGCAAAGAGCTACATGCCACATGCTGTTCTCCAGCCTGCTGTGTGTATTTGTGGCCTTCAGCTACTCTGGATCAAGTGTGGCCCAGAAGGTTACTCAAGCCCAGTCATCAGTATCCATGCCAGTGAGGAAAGCAGTCACCCTGAACTGCCTGTATGAAACAAGTTGGTGGTCATATTATATTTTTTGGTACAAGCAACTTCCCAGCAAAGAGATGATTTTCCTTATTCGCCAGGGTTCTGATGAACAGAATGCAAAAAGTGGTCGCTATTCTGTCAACTTCAAGAAAGCAGCGAAATCCGTCGCCTTAACCATTTCAGCCTTACAGCTAGAAGATTCAGCAAAGTACTTTTGTGCTCTTGGGGAACCCTTCCTCCCAGGGTGGGGCGATAAACTCATCTTTGGAAAAGGAACCCGTGTGACTGTGGAACCAAGAAGTCAGCCTCATACCAAACCATCCGTTTTTGTCATGAAAAATGGAACAAATGTCGCTTGTCTGGTGAAGGA',
        'AACTGGTTCTTTAGTC-1_contig_2':
        'AGGCAGAAGGTGGTTGAGAGGCAGAGCTGCCCCTGAGTGAGCCATGCAGAGGATCTCCTCCCTCATCCATCTCTCTCTCTTCTGGGCAGGAGTCATGTCAGCCATTGAGTTGGTGCCTGAACACCAAACAGTGCCTGTGTCAATAGGGGTCCCTGCCACCCTCAGGTGCTCCATGAAAGGAGAAGCGATCGGTAACTACTATATCAACTGGTACAGGAAGACCCAAGGTAACACAATGACTTTCATATACCGAGAAAAGGACATCTATGGCCCTGGTTTCAAAGACAATTTCCAAGGTGACATTGATATTGCAAAGAACCTGGCTGTACTTAAGATACTTGCACCATCAGAGAGAGATGAAGGGTCTTACTACTGTGCCTGTGACATTTTAGGGGATAAGGCCGATAAACTCATCTTTGGAAAAGGAACCCGTGTGACTGTGGAACCAAGAAGTCAGCCTCATACCAAACCATCCGTTTTTGTCATGAAAAATGGAACAAATGTCGCTTGTCTGGTGAAGGA',
        'AAAGATGCACCCTATC-1_contig_2':
        'TGGGGGAACTTTGAGCTTCAGGCAGCACAACTCACATTTGTGCAAAGAGCTACATGCCACATGCTGTTCTCCAGCCTGCTGTGTGTATTTGTGGCCTTCAGCTACTCTGGATCAAGTGTGGCCCAGAAGGTTACTCAAGCCCAGTCATCAGTATCCATGCCAGTGAGGAAAGCAGTCACCCTGAACTGCCTGTATGAAACAAGTTGGTGGTCATATTATATTTTTTGGTACAAGCAACTTCCCAGCAAAGAGATGATTTTCCTTATTCGCCAGGGTTCTGATGAACAGAATGCAAAAAGTGGTCGCTATTCTGTCAACTTCAAGAAAGCAGCGAAATCCGTCGCCTTAACCATTTCAGCCTTACAGCTAGAAGATTCAGCAAAGTACTTTTGTGCTCTTGGGGACCCCAGGCCTTCCTACAGGTACTGGGGGATACAGGGCGATAAACTCATCTTTGGAAAAGGAACCCGTGTGACTGTGGAACCAAGAAGTCAGCCTCATACCAAACCATCCGTTTTTGTCATGAAAAATGGAACAAATGTCGCTTGTCTGGTGAAGGA',
    }
    return (seq)


@pytest.fixture
def annotation_10x_tr1():
    """Standard cellranger annotation file to test the preprocessing."""
    annot = pd.DataFrame([
        [
            'AACTGGTTCTTTAGTC-1', 'false', 'AACTGGTTCTTTAGTC-1_contig_2',
            'true', '522', 'TRD', 'TRDV2', '', 'TRDJ1', 'TRDC', 'true',
            'false', '', '', '', '', '', '', '', '', '', '', 'CACDILGDKADKLIF',
            'TGTGCCTGTGACATTTTAGGGGATAAGGCCGATAAACTCATCTTT', '', '', '1176',
            '6', '', '', ''
        ],
        [
            'AAGACCTCACTACAGT-1', 'false', 'AAGACCTCACTACAGT-1_contig_1',
            'true', '544', 'TRD', 'TRDV1', '', 'TRDJ1', 'TRDC', 'true',
            'false', '', '', '', '', '', '', '', '', '', '',
            'CALGEPFLPGWGDKLIF',
            'TGTGCTCTTGGGGAACCCTTCCTCCCAGGGTGGGGCGATAAACTCATCTTT', '', '',
            '2932', '11', '', '', ''
        ],
    ],
        columns=[
        'barcode', 'is_cell', 'contig_id',
        'high_confidence', 'length', 'chain', 'v_gene',
        'd_gene', 'j_gene', 'c_gene', 'full_length',
        'productive', 'fwr1', 'fwr1_nt', 'cdr1',
        'cdr1_nt', 'fwr2', 'fwr2_nt', 'cdr2', 'cdr2_nt',
        'fwr3', 'fwr3_nt', 'cdr3', 'cdr3_nt', 'fwr4',
        'fwr4_nt', 'reads', 'umis', 'raw_clonotype_id',
        'raw_consensus_id', 'exact_subclonotype_id'
    ])

    return (annot)


@pytest.fixture
def annotation_10x_tr2():
    """Standard cellranger annotation file to test the preprocessing."""
    annot = pd.DataFrame([
        [
            'AAGACCTCACTACAGT-1', 'False', 'AAGACCTCACTACAGT-1_contig_1',
            'True', '544', 'TRD', 'TRDV1', 'None', 'TRDJ1', 'TRDC', 'True',
            'False', 'CALGEPFLPGWGDKLIF',
            'TGTGCTCTTGGGGAACCCTTCCTCCCAGGGTGGGGCGATAAACTCATCTTT', '2932',
            '11', 'None', 'None'
        ],
        [
            'AACTGGTTCTTTAGTC-1', 'False', 'AACTGGTTCTTTAGTC-1_contig_2',
            'True', '522', 'TRD', 'TRDV2', 'None', 'TRDJ1', 'TRDC', 'True',
            'False', 'CACDILGDKADKLIF',
            'TGTGCCTGTGACATTTTAGGGGATAAGGCCGATAAACTCATCTTT', '1176', '6',
            'None', 'None'
        ],
        [
            'AAAGATGCACCCTATC-1', 'False', 'AAAGATGCACCCTATC-1_contig_2',
            'True', '560', 'TRD', 'TRDV1', 'None', 'TRDJ1', 'TRDC', 'True',
            'False', 'CALGDPRPSYRYWGIQGDKLIF',
            'TGTGCTCTTGGGGACCCCAGGCCTTCCTACAGGTACTGGGGGATACAGGGCGATAAACTCATCTTT',
            '1128', '3', 'None', 'None'
        ],
    ],
        columns=[
        'barcode', 'is_cell', 'contig_id',
        'high_confidence', 'length', 'chain', 'v_gene',
        'd_gene', 'j_gene', 'c_gene', 'full_length',
        'productive', 'cdr3', 'cdr3_nt', 'reads', 'umis',
        'raw_clonotype_id', 'raw_consensus_id'
    ])

    return (annot)
