# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['gekkota']
setup_kwargs = {
    'name': 'gekkota',
    'version': '0.1.4',
    'description': 'Python code-generation for Python',
    'long_description': 'This is a Python code-generation module.    \n\n- Generates any Python statement/expression\n- Places parens to ensure expression priorities are unchanged\n- Places extra newlines before/after class/function definitions to conform with PEP 8\n- 100% coverage of type hints, passing MyPy with `--disallow-any-expr`\n- Meaningful type hierarchy inspired by Python grammar\n- Covered with ~~diamonds~~ tests completely\n\n\n## Expressions\n\n### Basic expressions\n\nYour starting points would be `Literal` and `Name`:\n\n```python\nfrom gekkota import Name, Literal\n\n# Name(self, name: str, annotation: Optional[Expression] = None)\n# Literal(self, value: Union[int, float, complex, str, bytes, bool, None])\n\na = Name("a")\nb = Name("b")\nsix = Literal(6)\n\n# prints \'a + b * 6\'\nprint(\n    (a + b * six).render_str()\n)\n\n```\n\n`Name`, as many other classes in the module, is an `Expression` object\n\nExpressions support most operations to combine with other expressions.    \nExceptions are:\n\n- Attribute reference: for that you should use `Expression.getattr(self, other: str)`\n- Indexing: `Expression.index(self, index: Expression)`\n- Slicing: `Expression.index(self, index: Union[SliceExpr, Sequence[SliceExpr]])`\n- Equality / Inequality: `Expression.eq(self, other: Expression)` and `Expression.neq(self, other: Expression)` respectively\n- `is`, `is not`, `in`, `not in`, `and`, `or`: `Expression.is_`, `Expression.is_not`, `Expression.in_`, `Expression.not_in`, `Expression.and_`, `Expression.or_`\n- `await`: `Expression.await_(self)`\n- `:=` aswsignment: `Expression.assign`\n\nFor example:\n\n```python\nfrom gekkota import Name, Literal\n\na = Name("a")\nb = Name("b")\n\nexpression = a.await_().in_(b)\n\nprint(expression.render_str()) # await a in b\n\n```\n\n\nFor any other operation on expressions you can just use familiar Python syntax:\n\n\n```python\nfrom gekkota import Name\n\n\n\na = Name("a")\nb = Name("b")\nc = Name("c")\n\n\n\nprint(\n    (a + b * c / a(b, c)).render_str() # \'a + b * c / a(b, c)\'\n)\n\n\n```\n\n### Sequences\n\nThere is a common class for all sequences, `SequenceExpr`.\nIt has 4 children: `TupleExpr`, `ListExpr`, `SetExpr` and `DictExpr`.\nAll have same signature: `SequenceExpr(self, values: Sequence[Expression])`:\n\n```python\nfrom gekkota import ListExpr, TupleExpr, DictExpr, KeyValue, SetExpr, Name\n\na = Name("a")\nb = Name("b")\n\nprint(\n    TupleExpr([a, b, Literal(6)]).render_str(), # \'(a, b, 6)\'\n    TupleExpr([a]).render_str(),                # \'(a, )\'\n    ListExpr([a, b]).render_str(),              # \'[a, b]\'\n    ListExpr([]).render_str(),                  # \'[]\'\n    DictExpr([KeyValue(a, b)]).render_str(),    # \'{a: b}\'\n    SetExpr([]).render_str()                    # \'set()\'\n)\n```\n\n\n## Statements\n\n### Small statements\n\nHere is an example of a few small statements:\n\n```python\nfrom gekkota import Name, SequenceExpr\nfrom gekkota import ReturnStmt, DelStmt, AssertStmt, BreakStmt, ContinueStmt, YieldStmt, YieldFromStmt, NonLocalStmt, GlobalStmt, PassStmt, RaiseStmt, AsyncStmt\n\na = Name("a")\nb = Name("b")\n\n\n    print(ReturnStmt(a).render_str()) # \'return a\'\n\n    print(YieldStmt(a).render_str()) # \'yield a\'\n    print(YieldFromStmt(b).render_str()) # \'yield from b\'\n    \n    print(DelStmt(a).render_str()) # \'del a\'\n    \n    print(AssertStmt(a).render_str()) # \'assert a\'\n    \n    print(BreakStmt().render_str()) # \'break\'\n    \n    print(ContinueStmt().render_str()) # \'continue\'\n    \n    print(GlobalStmt([a, b]).render_str()) # \'global a, b\'\n    print(NonLocalStmt([a, b]).render_str()) # \'nonlocal a, b\'\n    \n    print(PassStmt().render_str()) # \'pass\'\n\n    print(RaiseStmt().render_str()) # \'raise\' \n    print(RaiseStmt(a).render_str()) # \'raise a\'\n    print(RaiseStmt(a, b).render_str()) # \'raise a from b\'\n\n    print(AsyncStmt(a).render_str()) # \'async a\'\n```\n\n\n**End of documentation :(**    \nYou can check out tests for an explanation, check out code or your IDE\'s code completion.    \nThis documentation would be filled later, when I would have time and energy for that.    ',
    'author': 'Dmitry Gritsenko',
    'author_email': 'k01419q45@ya.ru',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/courage-tci/gekkota',
    'py_modules': modules,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
