"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TopicBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const policy_1 = require("./policy");
const subscription_1 = require("./subscription");
/**
 * Either a new or imported Topic.
 *
 * @stability stable
 */
class TopicBase extends core_1.Resource {
    /**
     * Subscribe some endpoint to this topic.
     *
     * @stability stable
     */
    addSubscription(subscription) {
        const subscriptionConfig = subscription.bind(this);
        const scope = subscriptionConfig.subscriberScope || this;
        let id = subscriptionConfig.subscriberId;
        if (core_1.Token.isUnresolved(subscriptionConfig.subscriberId)) {
            id = this.nextTokenId(scope);
        }
        // We use the subscriber's id as the construct id. There's no meaning
        // to subscribing the same subscriber twice on the same topic.
        if (scope.node.tryFindChild(id)) {
            throw new Error(`A subscription with id "${id}" already exists under the scope ${scope.node.path}`);
        }
        new subscription_1.Subscription(scope, id, {
            topic: this,
            ...subscriptionConfig,
        });
    }
    /**
     * Adds a statement to the IAM resource policy associated with this topic.
     *
     * If this topic was created in this stack (`new Topic`), a topic policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the topic is imported (`Topic.import`), then this is a no-op.
     *
     * @stability stable
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.TopicPolicy(this, 'Policy', { topics: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        var _b;
        const errors = super.validate();
        errors.push(...((_b = this.policy) === null || _b === void 0 ? void 0 : _b.document.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Grant topic publishing permissions to the given identity.
     *
     * @stability stable
     */
    grantPublish(grantee) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['sns:Publish'],
            resourceArns: [this.topicArn],
            resource: this,
        });
    }
    /**
     * Represents a notification target That allows SNS topic to associate with this rule target.
     *
     * @stability stable
     */
    bindAsNotificationRuleTarget(_scope) {
        // SNS topic need to grant codestar-notifications service to publish
        // @see https://docs.aws.amazon.com/dtconsole/latest/userguide/set-up-sns.html
        this.grantPublish(new iam.ServicePrincipal('codestar-notifications.amazonaws.com'));
        return {
            targetType: 'SNS',
            targetAddress: this.topicArn,
        };
    }
    nextTokenId(scope) {
        let nextSuffix = 1;
        const re = /TokenSubscription:([\d]*)/gm;
        // Search through the construct and all of its children
        // for previous subscriptions that match our regex pattern
        for (const source of scope.node.findAll()) {
            const m = re.exec(source.node.id); // Use regex to find a match
            if (m !== null) { // if we found a match
                const matchSuffix = parseInt(m[1], 10); // get the suffix for that match (as integer)
                if (matchSuffix >= nextSuffix) { // check if the match suffix is larger or equal to currently proposed suffix
                    nextSuffix = matchSuffix + 1; // increment the suffix
                }
            }
        }
        return `TokenSubscription:${nextSuffix}`;
    }
}
exports.TopicBase = TopicBase;
_a = JSII_RTTI_SYMBOL_1;
TopicBase[_a] = { fqn: "@aws-cdk/aws-sns.TopicBase", version: "1.121.0" };
//# sourceMappingURL=data:application/json;base64,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