# Copyright (c) ZOZO Technologies, Inc. All rights reserved.
# Licensed under the Apache 2.0 License.

"""Off-Policy Estimators."""
from abc import ABCMeta, abstractmethod
from dataclasses import dataclass
from typing import Dict, Optional

import numpy as np
from ..utils import estimate_confidence_interval_by_bootstrap


@dataclass
class BaseOffPolicyEstimator(metaclass=ABCMeta):
    """Base class for off-policy estimators."""

    @abstractmethod
    def _estimate_round_rewards(
        self, reward: np.ndarray, action_match: np.ndarray
    ) -> np.ndarray:
        """Estimate rewards for each round."""
        pass

    @abstractmethod
    def estimate_policy_value(self) -> float:
        """Estimate policy value of a counterfactual policy."""
        pass

    @abstractmethod
    def estimate_interval(self) -> Dict[str, float]:
        """Estimate confidence interval of policy value by nonparametric bootstrap procedure."""
        pass


@dataclass
class ReplayMethod(BaseOffPolicyEstimator):
    """Estimate the policy value by Relpay Method (RM).

    Replay Method (RM) estimates the policy value of a given counterfactual (or evaluation) policy :math:`\\pi` by

    .. math::

        \\hat{V}_{RM} (\\pi) =
        \\frac{1}{\\sum_{t=1}^T \\mathbb{I} \\{\\pi (x_t) = a_t \\}} \\sum_{t=1}^T \\mathbb{I} \\{ \\pi (x_t) = a_t \\} Y_t


    In order for this estimator to be unbiased, the actions on the logged samples
    must have been collected uniformly at random and not generated by some other policies.

    Parameters
    ----------
    estimator_name: str, default: 'rm'.
        Name of off-policy estimator.

    References
    ----------
    Lihong Li, Wei Chu, John Langford, and Xuanhui Wang.
    "Unbiased Offline Evaluation of Contextual-bandit-based News Article Recommendation Algorithms.", 2011.

    """

    estimator_name: str = "rm"

    def _estimate_round_rewards(
        self, reward: np.ndarray, action_match: np.ndarray, **kwargs
    ) -> np.ndarray:
        """Estimate rewards for each round.

        Parameters
        ----------
        reward: array-like, shape (n_rounds, )
            Observed reward (or outcome) for each round, i.e., :math:`Y_t=Y(a_t)`.

        action_match: array-like, shape (n_rounds, )
            Indicators representing wether counterfactual policy selected the same action selected by behavior policy.
            :math:`\\mathbb{I} \\{ \\pi (x_t) = a_t \\}` where :math:`\\pi (x_t)` is an action selected by counterfactual policy
            and :math:`a_t` is an action selected by behavior policy in round :math:`t`.

        Returns
        ----------
        estimated_rewards: array-like, shape (n_rounds, )
            Rewards estimated by the Replay Method for each round.

        """
        estimated_rewards = np.zeros_like(action_match)
        if np.sum(action_match) > 0.0:
            estimated_rewards = action_match * reward / np.mean(action_match)
        return estimated_rewards

    def estimate_policy_value(
        self, reward: np.ndarray, action_match: np.ndarray, **kwargs
    ) -> float:
        """Estimate policy value of a counterfactual policy.

        Parameters
        ----------
        reward: array-like, shape (n_rounds, )
            Observed reward (or outcome) for each round, i.e., :math:`Y_t = Y(a_t)`.

        action_match: array-like, shape (n_rounds, )
            Indicators representing wether counterfactual policy selected the same action selected by behavior policy.
            :math:`\\mathbb{I} \\{ \\pi (x_t) = a_t \\}` where :math:`\\pi (x_t)` is an action selected by counterfactual policy
            and :math:`a_t` is an action selected by behavior policy in round :math:`t`.

        Returns
        ----------
        V_hat: float
            Estimated policy value (performance) of a given counterfactual or evaluation policy.

        """
        estimated_rewards = self._estimate_round_rewards(
            reward=reward, action_match=action_match
        )
        return np.mean(estimated_rewards)

    def estimate_interval(
        self,
        reward: np.ndarray,
        action_match: np.ndarray,
        alpha: float = 0.05,
        n_bootstrap_samples: int = 100,
        random_state: Optional[int] = None,
        **kwargs,
    ) -> Dict[str, float]:
        """Estimate confidence interval of policy value by nonparametric bootstrap procedure.

        Parameters
        ----------
        reward: array-like, shape (n_rounds, )
            Observed reward (or outcome) for each round, i.e., :math:`Y_t=Y(a_t)`.

        action_match: array-like, shape (n_rounds, )
            Indicators representing wether counterfactual policy selected the same action selected by behavior policy.
            :math:`\\mathbb{I} \\{ \\pi (x_t) = a_t \\}` where :math:`\\pi (x_t)` is an action selected by counterfactual policy
            and :math:`a_t` is an action selected by behavior policy in round :math:`t`.

        alpha: float, default: 0.05
            P-value.

        n_bootstrap_samples: int, default: 100
            Number of resampling performed in the bootstrap procedure.

        random_state: int, default: None
            Controls the random seed in bootstrap sampling.

        Returns
        ----------
        estimated_confidence_interval: Dict[str, float]
            Dictionary storing the estimated mean and upper-lower confidence bounds.

        """
        estimated_rewards = self._estimate_round_rewards(
            reward=reward, action_match=action_match
        )
        return estimate_confidence_interval_by_bootstrap(
            samples=estimated_rewards,
            alpha=alpha,
            n_bootstrap_samples=n_bootstrap_samples,
            random_state=random_state,
        )


@dataclass
class InverseProbabilityWeighting(BaseOffPolicyEstimator):
    """Estimate the policy value by Inverse Probability Weighting (IPW).

    Inverse Probability Weighting (IPW) estimates the policy value of a given counterfactual (or evaluation) policy :math:`\\pi` by

    .. math::

        \\hat{V}_{IPW} (\\pi) =
        \\frac{1}{T} \\sum_{t=1}^T \\frac{\\mathbb{I} \\{ \\pi (x_t) = a_t \\}}{p_t} Y_t

    where :math:`p_t` is probability of an action :math:`a` was chosen by behavior policy at round :math:`t` called the *propensity score*.

    IPW re-weights the rewards by the inverse of the propensity score.
    When the behavior policy is known, the IPW estimator is unbiased and consistent for the policy value.
    However, it can have a large variance, especially when the counterfactual policy significantly deviates from the behavior policy.

    Parameters
    ----------
    min_pscore: float, default: 0.
        Minimum value used as propensity score.
        Propensity socres larger than this parameter would be clipped.

    estimator_name: str, default: 'ipw'.
        Name of off-policy estimator.

    References
    ----------
    Alex Strehl, John Langford, Lihong Li, and Sham M Kakade.
    "Learning from Logged Implicit Exploration Data"., 2010.

    Miroslav Dudík, Dumitru Erhan, John Langford, and Lihong Li.
    "Doubly Robust Policy Evaluation and Optimization.", 2014.

    """

    min_pscore: float = 0.0
    estimator_name: str = "ipw"
    assert (
        min_pscore <= 1.0
    ), f"minimum propensity score must be lower than 1, but {min_pscore} is given."

    def _estimate_round_rewards(
        self, reward: np.ndarray, pscore: np.ndarray, action_match: np.ndarray, **kwargs
    ) -> np.ndarray:
        """Estimate rewards for each round.

        Parameters
        ----------
        reward: array-like, shape (n_rounds, )
            Observed reward (or outcome) for each round, i.e., :math:`Y_t=Y(a_t)`.

        pscore: array-like, shape (n_rounds, )
            Propensity score for each round, i.e., :math:`p_t=E[A=a_t | X=x_t]`.

        action_match: array-like, shape (n_rounds, )
            Indicators representing wether counterfactual policy selected the same action selected by behavior policy.
            :math:`\\mathbb{I} \\{ \\pi (x_t) = a_t \\}` where :math:`\\pi (x_t)` is an action selected by counterfactual policy
            and :math:`a_t` is an action selected by behavior policy in round :math:`t`.

        Returns
        ----------
        estimated_rewards: array-like, shape (n_rounds, )
            Rewards estimated by the Replay Method for each round.

        """
        return (action_match * reward) / pscore

    def estimate_policy_value(
        self, reward: np.ndarray, pscore: np.ndarray, action_match: np.ndarray, **kwargs
    ) -> np.ndarray:
        """Estimate policy value of a counterfactual policy.

        Parameters
        ----------
        reward: array-like, shape (n_rounds, )
            Observed reward (or outcome) for each round, i.e., :math:`Y_t=Y(a_t)`.

        pscore: array-like, shape (n_rounds, )
            Propensity score for each round, i.e., :math:`p_t=E[A=a_t | X=x_t]`.

        action_match: array-like, shape (n_rounds, )
            Indicators representing wether counterfactual policy selected the same action selected by behavior policy.
            :math:`\\mathbb{I} \\{ \\pi (x_t) = a_t \\}` where :math:`\\pi (x_t)` is an action selected by counterfactual policy
            and :math:`a_t` is an action selected by behavior policy in round :math:`t`.

        Returns
        ----------
        V_hat: float
            Estimated policy value (performance) of a given counterfactual or evaluation policy.

        """
        estimated_rewards = self._estimate_round_rewards(
            reward=reward, pscore=pscore, action_match=action_match
        )
        return np.mean(estimated_rewards)

    def estimate_interval(
        self,
        reward: np.ndarray,
        pscore: np.ndarray,
        action_match: np.ndarray,
        alpha: float,
        n_bootstrap_samples: int,
        random_state: Optional[int] = None,
        **kwargs,
    ) -> Dict[str, float]:
        """Estimate confidence interval of policy value by nonparametric bootstrap procedure.

        Parameters
        ----------
        reward: array-like, shape (n_rounds, )
            Observed reward (or outcome) for each round, i.e., :math:`Y_t=Y(a_t)`.

        pscore: array-like, shape (n_rounds, )
            Propensity score for each round, i.e., :math:`p_t=E[A=a_t | X=x_t]`.

        action_match: array-like, shape (n_rounds, )
            Indicators representing wether counterfactual policy selected the same action selected by behavior policy.
            :math:`\\mathbb{I} \\{ \\pi (x_t) = a_t \\}` where :math:`\\pi (x_t)` is an action selected by counterfactual policy
            and :math:`a_t` is an action selected by behavior policy in round :math:`t`.

        alpha: float, default: 0.05
            P-value.

        n_bootstrap_samples: int, default: 100
            Number of resampling performed in the bootstrap procedure.

        random_state: int, default: None
            Controls the random seed in bootstrap sampling.

        Returns
        ----------
        estimated_confidence_interval: Dict[str, float]
            Dictionary storing the estimated mean and upper-lower confidence bounds.

        """
        estimated_rewards = self._estimate_round_rewards(
            reward=reward, pscore=pscore, action_match=action_match
        )
        return estimate_confidence_interval_by_bootstrap(
            samples=estimated_rewards,
            alpha=alpha,
            n_bootstrap_samples=n_bootstrap_samples,
            random_state=random_state,
        )


@dataclass
class SelfNormalizedInverseProbabilityWeighting(InverseProbabilityWeighting):
    """Estimate the policy value by Self-Normalized Inverse Probability Weighting (SNIPW).

    Self-Normalized Inverse Probability Weighting (SNIPW) estimates the policy value of a given counterfactual (or evaluation) policy :math:`\\pi` by

    .. math::

        \\hat{V}_{SNIPW} (\\pi) =
        \\frac{1}{\\sum_{t=1}^T \\frac{\\mathbb{I} \\{ \\pi (x_t) = a_t \\}}{p_t} } \\sum_{t=1}^T \\frac{\\mathbb{I} \\{ \\pi (x_t) = a_t \\}}{p_t} Y_t

    where :math:`p_t` is probability of an action :math:`a` was chosen by behavior policy at round :math:`t` called the *propensity score*.

    SNIPW re-weights the observed rewards by the self-normalized importance weihgt.
    This estimator is not unbiased even when the behavior policy is known.
    However, it is still consistent for the policy value and increase the stability in some senses.
    See the references for the detailed discussions.

    Parameters
    ----------
    min_pscore: float, default: 0.
        Minimum value used as propensity score.
        Propensity socres larger than this parameter would be clipped.

    estimator_name: str, default: 'snipw'.
        Name of off-policy estimator.

    References
    ----------
    Adith Swaminathan and Thorsten Joachims.
    "The Self-normalized Estimator for Counterfactual Learning.", 2015.

    Nathan Kallus and Masatoshi Uehara.
    "Intrinsically Efficient, Stable, and Bounded Off-Policy Evaluation for Reinforcement Learning.", 2019.

    """

    estimator_name: str = "snipw"

    def _estimate_round_rewards(
        self, reward: np.ndarray, pscore: np.ndarray, action_match: np.ndarray, **kwargs
    ) -> np.ndarray:
        """Estimate rewards for each round.

        Parameters
        ----------
        reward: array-like, shape (n_rounds, )
            Observed reward (or outcome) for each round, i.e., :math:`Y_t=Y(a_t)`.

        pscore: array-like, shape (n_rounds, )
            Propensity score for each round, i.e., :math:`p_t=E[A=a_t | X=x_t]`.

        action_match: array-like, shape (n_rounds, )
            Indicators representing wether counterfactual policy selected the same action selected by behavior policy.
            :math:`\\mathbb{I} \\{ \\pi (x_t) = a_t \\}` where :math:`\\pi (x_t)` is an action selected by counterfactual policy
            and :math:`a_t` is an action selected by behavior policy in round :math:`t`.

        Returns
        ----------
        estimated_rewards: array-like, shape (n_rounds, )
            Rewards estimated by the SNIPW estimator for each round.

        """
        return (action_match * reward) / pscore / (action_match / pscore).mean()


@dataclass
class DirectMethod(BaseOffPolicyEstimator):
    """Estimate the policy value by Direct Method (DM).

    DM first learns a supervised machine learning model, such as random
    forest, ridge regression, and gradient boosting, to predict the mean reward function (:math:`\\mu(x, a) = E[Y | X=x, A=a]`).
    It then uses it to estimate the policy value as follows.

    .. math::

        \\hat{V}_{DM} (\\pi) =
        \\frac{1}{T} \\sum_{t=1}^T \\mathbb{I} \\{ \\pi (x_t) = a_t \\} \\hat{\\mu} (x_t, a_t)

    where :math:`\\hat{\\mu}` is the regression function and :math:`\\hat{\\mu} (x_t, a_t)` is an estimated rewards for round :math:`t`.
    To estimate the mean reward function, please use `obp.ope.regression_model.RegressionModel`, which supports several fitting methods specific to OPE.

    If the regression model is a good approximation to the mean reward function,
    this estimator accurately predicts the policy value of the counterfactual policy.
    If the regression function fails to approximate the mean reward function well,
    however, the final estimator is no longer consistent.

    Parameters
    ----------
    estimator_name: str, default: 'dm'.
        Name of off-policy estimator.

    References
    ----------
    Alina Beygelzimer and John Langford.
    "The offset tree for learning with partial labels.", 2009.

    Miroslav Dudík, Dumitru Erhan, John Langford, and Lihong Li.
    "Doubly Robust Policy Evaluation and Optimization.", 2014.

    """

    estimator_name: str = "dm"

    def _estimate_round_rewards(
        self, estimated_rewards_by_reg_model: np.ndarray, **kwargs
    ) -> float:
        """Estimate policy value of a counterfactual policy.

        Parameters
        ----------
        estimated_rewards_by_reg_model: array-like, shape (n_rounds, )
            Estimated rewards by regression model.

        Returns
        ----------
        estimated_rewards: array-like, shape (n_rounds, )
            Rewards estimated by the DM estimator for each round.

        """
        return estimated_rewards_by_reg_model

    def estimate_policy_value(
        self, estimated_rewards_by_reg_model: np.ndarray, **kwargs
    ) -> float:
        """Estimate policy value of a counterfactual policy.

        Parameters
        ----------
        estimated_rewards_by_reg_model: array-like, shape (n_rounds, )
            Estimated rewards by regression model.

        Returns
        ----------
        V_hat: float
            Estimated policy value (performance) of a given counterfactual or evaluation policy.

        """
        return np.mean(estimated_rewards_by_reg_model)

    def estimate_interval(
        self,
        estimated_rewards_by_reg_model: np.ndarray,
        alpha: float,
        n_bootstrap_samples: int,
        random_state: Optional[int] = None,
        **kwargs,
    ) -> Dict[str, float]:
        """Estimate confidence interval of policy value by nonparametric bootstrap procedure.

        Parameters
        ----------
        estimated_rewards_by_reg_model: array-like, shape (n_rounds, )
            Estimated rewards by regression model.

        alpha: float, default: 0.05
            P-value.

        n_bootstrap_samples: int, default: 100
            Number of resampling performed in the bootstrap procedure.

        random_state: int, default: None
            Controls the random seed in bootstrap sampling.

        Returns
        ----------
        estimated_confidence_interval: Dict[str, float]
            Dictionary storing the estimated mean and upper-lower confidence bounds.

        """
        return estimate_confidence_interval_by_bootstrap(
            samples=estimated_rewards_by_reg_model,
            alpha=alpha,
            n_bootstrap_samples=n_bootstrap_samples,
            random_state=random_state,
        )


@dataclass
class DoublyRobust(InverseProbabilityWeighting):
    """Estimate the policy value by Doubly Robust (DR).

    Similar to DM, DR first learns a supervised machine learning model, such as random
    forest, ridge regression, and gradient boosting, to predict the mean reward function (:math:`\\mu(x, a) = E[Y | X=x, A=a]`).
    It then uses it to estimate the policy value as follows.

    .. math::

            \\hat{V}_{DR} (\\pi) =
            \\frac{1}{T} \\sum_{t=1}^T \\frac{\\mathbb{I} \\{ \\pi (x_t) = a_t \\}}{p_t} (Y_t - \\hat{\\mu} (x_t, a_t)) + \\hat{\\mu} (x_t, a_t)

    where :math:`p_t` is the probability of an action :math:`a` was chosen by behavior policy at round :math:`t` called the *propensity score*.
    :math:`\\hat{\\mu}` is the regression function and :math:`\\hat{\\mu} (x_t, a_t)` is an estimated rewards for round :math:`t`.
    To estimate the mean reward function, please use `obp.ope.regression_model.RegressionModel`,
    which supports several fitting methods specific to OPE such as *more robust doubly robust*.

    DR mimics IPW to use a weighted version of rewards, but DR also uses the estimated mean reward
    function (the regression model) as a control variate to decrease the variance.
    It preserves the consistency of IPW if either the importance weight or
    the mean reward estimator is accurate (a property called double robustness).
    Moreover, DR is semiparametric efficient when the mean reward estimator is correctly specified.

    Parameters
    ----------
    estimator_name: str, default: 'dr'.
        Name of off-policy estimator.

    References
    ----------
    Miroslav Dudík, Dumitru Erhan, John Langford, and Lihong Li.
    "Doubly Robust Policy Evaluation and Optimization.", 2014.

    Mehrdad Farajtabar, Yinlam Chow, and Mohammad Ghavamzadeh.
    "More Robust Doubly Robust Off-policy Evaluation.", 2018.

    """

    estimator_name: str = "dr"

    def _estimate_round_rewards(
        self,
        reward: np.ndarray,
        pscore: np.ndarray,
        estimated_rewards_by_reg_model: np.ndarray,
        action_match: np.ndarray,
        **kwargs,
    ) -> np.ndarray:
        """Estimate rewards for each round.

        Parameters
        ----------
        reward: array-like, shape (n_rounds, )
            Observed reward (or outcome) for each round, i.e., :math:`Y_t=Y(a_t)`.

        pscore: array-like, shape (n_rounds, )
            Propensity score for each round, i.e., :math:`p_t=E[A=a_t | X=x_t]`.

        estimated_rewards: array-like, shape (n_rounds, )
            Estimated rewards by regression model.

        action_match: array
            Indicators representing wether counterfactual policy selected the same action selected by behavior policy.
            :math:`\\mathbb{I} \\{ \\pi (x_t) = a_t \\}` where :math:`\\pi (x_t)` is an action selected by counterfactual policy
            and :math:`a_t` is an action selected by behavior policy in round :math:`t`.

        Returns
        ----------
        estimated_rewards: array-like, shape (n_rounds, )
            Rewards estimated by the DR estimator for each round.

        """
        return (
            (action_match * (reward - estimated_rewards_by_reg_model)) / pscore
        ) + estimated_rewards_by_reg_model

    def estimate_policy_value(
        self,
        reward: np.ndarray,
        pscore: np.ndarray,
        action_match: np.ndarray,
        estimated_rewards_by_reg_model: np.ndarray,
    ) -> float:
        """Estimate policy value of a counterfactual policy.

        Parameters
        ----------
        reward: array-like, shape (n_rounds, )
            Observed reward (or outcome) for each round, i.e., :math:`Y_t=Y(a_t)`.

        pscore: array-like, shape (n_rounds, )
            Propensity score for each round, i.e., :math:`p_t=E[A=a_t | X=x_t]`.

        action_match: array-like, shape (n_rounds, )
            Indicators representing wether counterfactual policy selected the same action selected by behavior policy.
            :math:`\\mathbb{I} \\{ \\pi (x_t) = a_t \\}` where :math:`\\pi (x_t)` is an action selected by counterfactual policy
            and :math:`a_t` is an action selected by behavior policy in round :math:`t`.

        estimated_rewards_by_reg_model: array-like, shape (n_rounds, )
            Estimated policy value (performance) of a given counterfactual or evaluation policy.

        Returns
        ----------
        V_hat: float
            Estimated policy value by the DR estimator.

        """
        estimated_rewards = self._estimate_round_rewards(
            reward=reward,
            pscore=pscore,
            action_match=action_match,
            estimated_rewards_by_reg_model=estimated_rewards_by_reg_model,
        )
        return np.mean(estimated_rewards)

    def estimate_interval(
        self,
        reward: np.ndarray,
        pscore: np.ndarray,
        action_match: np.ndarray,
        estimated_rewards_by_reg_model: np.ndarray,
        alpha: float,
        n_bootstrap_samples: int,
        random_state: Optional[int] = None,
        **kwargs,
    ) -> Dict[str, float]:
        """Estimate confidence interval of policy value by nonparametric bootstrap procedure.

        Parameters
        ----------
        reward: array-like, shape (n_rounds, )
            Observed reward (or outcome) for each round, i.e., :math:`Y_t=Y(a_t)`.

        pscore: array-like, shape (n_rounds, )
            Propensity score for each round, i.e., :math:`p_t=E[A=a_t | X=x_t]`.

        action_match: array-like, shape (n_rounds, )
            Indicators representing wether counterfactual policy selected the same action selected by behavior policy.
            :math:`\\mathbb{I} \\{ \\pi (x_t) = a_t \\}` where :math:`\\pi (x_t)` is an action selected by counterfactual policy
            and :math:`a_t` is an action selected by behavior policy in round :math:`t`.

        estimated_rewards_by_reg_model: array-like, shape (n_rounds, )
            Estimated rewards by regression model.

        alpha: float, default: 0.05
            P-value.

        n_bootstrap_samples: int, default: 100
            Number of resampling performed in the bootstrap procedure.

        random_state: int, default: None
            Controls the random seed in bootstrap sampling.

        Returns
        ----------
        estimated_confidence_interval: Dict[str, float]
            Dictionary storing the estimated mean and upper-lower confidence bounds.

        """
        estimated_rewards = self._estimate_round_rewards(
            reward=reward,
            pscore=pscore,
            action_match=action_match,
            estimated_rewards_by_reg_model=estimated_rewards_by_reg_model,
        )
        return estimate_confidence_interval_by_bootstrap(
            samples=estimated_rewards,
            alpha=alpha,
            n_bootstrap_samples=n_bootstrap_samples,
            random_state=random_state,
        )


@dataclass
class SelfNormalizedDoublyRobust(DoublyRobust):
    """Estimate the policy value by Self-Normalized Doubly Robust (SNDR).

    Self-Normalized Doubly Robust estimates the policy value of a given counterfactual (or evaluation) policy :math:`\\pi` by

    .. math::

            \\hat{V}_{SNDR} (\\pi) =
            \\frac{1}{\\sum_{t=1}^T \\frac{\\mathbb{I} \\{ \\pi (x_t) = a_t \\}}{p_t}} \\sum_{t=1}^T \\frac{\\mathbb{I} \\{ \\pi (x_t) = a_t \\}}{p_t} (Y_t - \\hat{\\mu} (x_t, a_t)) + \\hat{\\mu} (x_t, a_t)

    where :math:`p_t` is the probability of an action :math:`a` was chosen by behavior policy at round :math:`t` called the *propensity score*.
    :math:`\\hat{\\mu}` is the regression function and :math:`\\hat{\\mu} (x_t, a_t)` is an estimated rewards for round :math:`t`.
    To estimate the mean reward function, please use `obp.ope.regression_model.RegressionModel`,
    which supports several fitting methods specific to OPE such as *more robust doubly robust*.

    Similar to Self-Normalized Inverse Probability Weighting, SNDR estimator applies the self-normalized importance weightinng technique to
    increase the stability of the original Doubly Robust estimator.
    See also the description of `obp.ope.SelfNormalizedInverseProbabilityWeighting` for details.

    Parameters
    ----------
    estimator_name: str, default: 'sndr'.
        Name of off-policy estimator.

    References
    ----------
    Miroslav Dudík, Dumitru Erhan, John Langford, and Lihong Li.
    "Doubly Robust Policy Evaluation and Optimization.", 2014.

    Nathan Kallus and Masatoshi Uehara.
    "Intrinsically Efficient, Stable, and Bounded Off-Policy Evaluation for Reinforcement Learning.", 2019.

    """

    estimator_name: str = "sndr"

    def _estimate_round_rewards(
        self,
        reward: np.ndarray,
        pscore: np.ndarray,
        estimated_rewards_by_reg_model: np.ndarray,
        action_match: np.ndarray,
        **kwargs,
    ) -> np.ndarray:
        """Estimate rewards for each round.

        Parameters
        ----------
        reward: array-like, shape (n_rounds, )
            Observed reward (or outcome) for each round, i.e., :math:`Y_t=Y(a_t)`.

        pscore: array-like, shape (n_rounds, )
            Propensity score for each round, i.e., :math:`p_t=E[A=a_t | X=x_t]`.

        estimated_rewards_by_reg_model: array-like, shape (n_rounds, )
            Estimated rewards by regression model.

        action_match: array
            Indicators representing wether counterfactual policy selected the same action selected by behavior policy.
            :math:`\\mathbb{I} \\{ \\pi (x_t) = a_t \\}` where :math:`\\pi (x_t)` is an action selected by counterfactual policy
            and :math:`a_t` is an action selected by behavior policy in round :math:`t`.

        Returns
        ----------
        estimated_rewards: array-like, shape (n_rounds, )
            Rewards estimated by the SNDR estimator for each round.

        """
        round_rewards = (
            action_match * (reward - estimated_rewards_by_reg_model) / pscore
        )
        round_rewards += estimated_rewards_by_reg_model
        round_rewards /= (action_match / pscore).mean()
        return round_rewards


@dataclass
class SwitchDoublyRobust(DoublyRobust):
    """Estimate the policy value by Switch Doubly Robust (Switch-DR).

    Switch Doubly Robust aims to reduce the variance of the Doubly Robust esitmator by using the direct method insted of doubly robust
    when the inverse of the propensity score (or the density ratio) is large.
    This estimator estimates the policy value of a given counterfactual (or evaluation) policy :math:`\\pi` by

    .. math::

            \\hat{V}_{Switch-DR} (\\pi) =
            \\frac{1}{T} \\sum_{t=1}^T v_t \\mathbb{I} \\{ (p_t)^{-1} \\le \\tau \\} + \{ (p_t)^{-1} > \\tau \\} \\hat{\\mu} (x_t, a_t)

    where :math:`v_t =  \\frac{\\mathbb{I} \\{ \\pi (x_t) = a_t \\}}{p_t} (Y_t - \\hat{\\mu} (x_t, a_t)) + \\hat{\\mu} (x_t, a_t)` is the doubly robust estimates for the rewards of each round
    and :math:`\\tau (\\ge 1)` is the *switching hyperparameter*, which decides the *threshold* for the inverse of the propensity score.
    :math:`p_t` is the probability of an action :math:`a` was chosen by behavior policy at round :math:`t` called the *propensity score*.
    :math:`\\hat{\\mu}` is the regression function and :math:`\\hat{\\mu} (x_t, a_t)` is an estimated rewards for round :math:`t`.
    To estimate the mean reward function, please use `obp.ope.regression_model.RegressionModel`,
    which supports several fitting methods specific to OPE such as *more robust doubly robust*.

    Parameters
    ----------
    tau: float, default: 1000
        Switching hyperparameter. When the density ratio is larger than this parameter
        the DM estimator is applied, otherwise the DR esitmator is applied.
        This hyperparameter should be larger than 1., otherwise it is meaningless.

    estimator_name: str, default: 'switch-dr'.
        Name of off-policy estimator.

    References
    ----------
    Miroslav Dudík, Dumitru Erhan, John Langford, and Lihong Li.
    "Doubly Robust Policy Evaluation and Optimization.", 2014.

    Yu-Xiang Wang, Alekh Agarwal, and Miroslav Dudík.
    "Optimal and Adaptive Off-policy Evaluation in Contextual Bandits", 2016.

    """

    tau: float = 1000
    estimator_name: str = "switch-dr"
    assert (
        tau >= 1.0
    ), f"switching hyperparameter should be larger than 1. but {tau} is given."

    def _estimate_round_rewards(
        self,
        reward: np.ndarray,
        pscore: np.ndarray,
        estimated_rewards_by_reg_model: np.ndarray,
        action_match: np.ndarray,
        **kwargs,
    ) -> float:
        """Estimate rewards for each round.

        Parameters
        ----------
        reward: array-like, shape (n_rounds, )
            Observed reward (or outcome) for each round, i.e., :math:`Y_t=Y(a_t)`.

        pscore: array-like, shape (n_rounds, )
            Propensity score for each round, i.e., :math:`p_t=E[A=a_t | X=x_t]`.

        estimated_rewards_by_reg_model: array-like, shape (n_rounds, )
            Estimated rewards by regression model.

        action_match: array-like, shape (n_rounds, )
            Indicators representing wether counterfactual policy selected the same action selected by behavior policy.
            :math:`\\mathbb{I} \\{ \\pi (x_t) = a_t \\}` where :math:`\\pi (x_t)` is an action selected by counterfactual policy
            and :math:`a_t` is an action selected by behavior policy in round :math:`t`.

        Returns
        ----------
        estimated_rewards: array-like, shape (n_rounds, )
            Rewards estimated by the Switch-DR estimator for each round.

        """
        switch_indicator = np.array((1.0 / pscore) <= self.tau, dtype=int)
        dr_estimated_rewards = (
            action_match * (reward - estimated_rewards_by_reg_model)
        ) / pscore
        dr_estimated_rewards += estimated_rewards_by_reg_model
        switch_dr_estimated_rewards = switch_indicator * dr_estimated_rewards
        switch_dr_estimated_rewards += (
            1 - switch_indicator
        ) * estimated_rewards_by_reg_model

        return switch_dr_estimated_rewards

