import logging

from .jobs import (
  get_status, Instance, Subway
)
from ..utils import nbox_session

logger = logging.getLogger()
web_server_subway = Subway("https://nimblebox.ai", nbox_session)

def start(
  name: str,
  loc = None,
  cpu_only: bool = True,
  gpu_count: int = 1,
):
  logger.info(f"Starting {name} on {loc}")
  Instance(name, loc).start(cpu_only, gpu_count)


def stop(
  name: str = "all",
  loc = None
):
  if name == "all":
    logger.info(f"Stopping {all} on {loc}")
    instances_to_stop = []
    money, data = get_status(loc)
    logger.info(f"Money: {money}")
    for item in data:
      if item["state"] == "RUNNING":
        instances_to_stop.append(item)
  else:
    instances_to_stop = [name]

  if not instances_to_stop:
    logger.info("No instances to stop")
    return

  for item in instances_to_stop:
    logger.info(f"Stopping {item['instance_id']}")
    web_server_subway.stop(item["instance_id"])


def init(project_name):
  print("-" * 69)
  import os, re
  from datetime import datetime
  created_time = datetime.now().strftime("%Y-%m-%d %H:%M:%S")

  out = re.findall("^[a-zA-Z_]+$", project_name)
  if not out:
    raise ValueError("Project name can only contain letters and underscore")

  if os.path.exists(project_name):
    raise ValueError(f"Project {project_name} already exists")
  
  print(f"Creating a folder: {project_name}")
  os.mkdir(project_name)
  os.chdir(project_name)

  print("Creating: exe.py")
  with open("exe.py", "w") as f:
    code_ = f'''
#!/usr/bin/env python3
# auto generated by 'nbox jobs init' command
# project_name: {project_name}
# created_time: {created_time}

from nbox import Operator
from nbox.utils import folder

def get_operator() -> Operator:
  # since initialising your operator might require passing a bunch of arguments
  # you can use this function to get the operator by manually defining things here

  return None

def deploy():
  # get operator and deploy it
  op: Operator = get_operator()
  op.deploy(
    job_id = "if job_id is None, new job will be created",
    init_folder = folder(__file__) # no need to change this
  )

def run():
  # run your operator
  op: Operator = get_operator()
  op(
    # if your operator is scheduled, avoid passing anything here
    # if your operator is run one time, pass the input here
  )
'''

    code_ += '''
if __name__ == "__main__":
  from fire import Fire
  Fire({"deploy": deploy, "run": run})

# end of auto generated code'''
    f.write(code_.strip())

  with open("README.md", "w") as f:
    f.write(f'''
# '{project_name}' NBX-Jobs

This is an autogenerated README file for [NBX-Jobs](www.nimblebox.ai/jobs) which was:
* created at: {created_time}
* name: {project_name}

Steps to complete:

1. Add all your code module here in this folder
2. Add in requirements.txt file if needed
3. Fill The Three functions (`get_operator`, `deploy`, `run`) in exe.py
4. Run `nbox jobs deploy <folder_path> or `nbox jobs deploy -h` for more help

Rule of Thumb:

1. If you are scheduling, your `op` forward should not takes arguments, otherwise
you are executing the same process over and over, which is not what you want.
If you still want to do so, try moving as much as possible to `__init__`.
2. If you are running this one-time, then you should define arguments in `run` function.

    '''.strip() + "\n")

  open("requirements.txt", "w").close()

  print("Completed")
  print("-" * 69)

def deploy(folder: str = "./"):
  """Deploy a job on NimbleBox.ai's NBX-Jobs. This convinience function will
  just run ``./exe.py deploy``

  Args:
    folder (str, optional): Folder to deploy. Defaults to "./".
  """

  import os
  import sys
  import subprocess
  
  if not os.path.exists(folder) or not os.path.isdir(folder):
    raise ValueError(f"Incorrect project at path: '{folder}'! nbox jobs init <name>")
  if os.path.isdir(folder):
    os.chdir(folder)
    if not os.path.exists("./exe.py"):
      raise ValueError(f"Incorrect project at path: '{folder}'! nbox jobs init <name>")
  else:
    raise ValueError(f"Incorrect project at path: '{folder}'! nbox jobs init <name>")

  sys.path.append(folder)
  os.chdir(folder)
  subprocess.call(["./exe.py", "deploy"])

def status(
  id_or_name: str
):
  raise NotImplementedError


# -------------------------

