# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['zmq_tubes']

package_data = \
{'': ['*']}

install_requires = \
['pyzmq']

setup_kwargs = {
    'name': 'zmq-tubes',
    'version': '1.5.0',
    'description': 'Wrapper for ZMQ comunication.',
    'long_description': '[![PyPI](https://img.shields.io/pypi/v/zmq_tubes?color=green&style=plastic)](https://pypi.org/project/zmq-tubes/)\n![PyPI - Python Version](https://img.shields.io/pypi/pyversions/zmq_tubes?style=plastic)\n![License](https://img.shields.io/github/license/calcite/zmq_tubes?style=plastic)\n# ZMQ Tubes\n\nZMQ Tubes is a managing system for ZMQ communication. \nIt can manage many ZMQ sockets by one interface. \nThe whole system is hierarchical, based on topics \n(look at [MQTT topics](https://www.hivemq.com/blog/mqtt-essentials-part-5-mqtt-topics-best-practices/)).\n\n## Classes\n- **TubeMessage** - This class represents a request/response message. \n  Some types of tubes require a response in this format.\n- **Tube** - This class wraps a ZMQ socket. \n  It represents a connection between client and server.\n- **TubeNode** - This represents an application interface for communication via tubes.\n\n\n## Asyncion / Threading\nThe library support bot method. \n\n```python\nfrom zmq_tubes import TubeNode, Tube            # Asyncio classes\nfrom zmq_tubes.threads import TubeNode, Tube    # Threads classes\n```\n\n\n## Usage:\n\n### Node definitions in yml file \nWe can define all tubes for one TubeNode by yml file. \n\n```yaml\n# test.yml\ntubes:\n  - name: Client REQ\n    addr:  ipc:///tmp/req.pipe      \n    tube_type: REQ\n    topics:\n      - foo/#\n      - +/bar\n  \n  - name: Client PUB\n    addr:  ipc:///tmp/pub.pipe      \n    tube_type: PUB\n    topics:\n      - foo/pub/#\n\n  - name: Server ROUTER\n    addr:  ipc:///tmp/router.pipe      \n    tube_type: ROUTER\n    server: yes\n    sockopts:\n      LINGER: 0\n    topics:\n      - server/#\n```\n\n```python\nimport asyncio\nimport yaml\nfrom zmq_tubes import TubeNode, TubeMessage\n\n\nasync def handler(request: TubeMessage):\n  print(request.payload)\n  return request.create_response(\'response\')\n\n\nwith open(\'test.yml\', \'r+\') as fd:    \n    schema = yaml.safe_load(fd)\n\nnode = TubeNode(schema=schema)\nnode.register_handler(\'server/#\', handler)\nasyncio.current_task(node.start(), name="Server")\n\nnode.publish(\'foo/pub/test\', \'message 1\')\nprint(await node.request(\'foo/xxx\', \'message 2\'))\n```\n\n\n\n\n### Request / Response\nThis is a simple scenario, the server processes the requests serially.\n#### Server:\n\n```python\nfrom zmq_tubes import Tube, TubeNode, TubeMessage\n\n\nasync def handler(request: TubeMessage):\n  print(request.payload)\n  return \'answer\'\n  # or return request.create_response(\'response\')\n\n\ntube = Tube(\n  name=\'Server\',\n  addr=\'ipc:///tmp/req_resp.pipe\',\n  server=True,\n  tube_type=\'REP\'\n)\n\nnode = TubeNode()\nnode.register_tube(tube, \'test/#\')\nnode.register_handler(\'test/#\', handler)\nawait node.start()\n\n# output: \'question\'\n```\n\n#### Client:\n\n```python\nfrom zmq_tubes import Tube, TubeNode\n\ntube = Tube(\n  name=\'Client\',\n  addr=\'ipc:///tmp/req_resp.pipe\',\n  tube_type=\'REQ\'\n)\n\nnode = TubeNode()\nnode.register_tube(tube, \'test/#\')\nresponse = await node.request(\'test/xxx\', \'question\')\nprint(response.payload)\n# output: \'answer\'\n```\n\n\n\n\n### Subscribe / Publisher\n#### Server:\n\n```python\nfrom zmq_tubes import Tube, TubeNode, TubeMessage\n\n\nasync def handler(request: TubeMessage):\n  print(request.payload)\n\n\ntube = Tube(\n  name=\'Server\',\n  addr=\'ipc:///tmp/sub_pub.pipe\',\n  server=True,\n  tube_type=\'SUB\'\n)\n\nnode = TubeNode()\nnode.register_tube(tube, \'test/#\')\nnode.register_handler(\'test/#\', handler)\nawait node.start()\n# output: \'message\'\n```\n\n#### Client:\n\n```python\nfrom zmq_tubes import Tube, TubeNode\n\ntube = Tube(\n  name=\'Client\',\n  addr=\'ipc:///tmp/sub_pub.pipe\',\n  tube_type=\'PUB\'\n)\n# In the case of publishing, the first message is very often\n# lost. The workaround is to connect the tube manually as soon as possible.\ntube.connect()\n\nnode = TubeNode()\nnode.register_tube(tube, \'test/#\')\nnode.publish(\'test/xxx\', \'message\')        \n```\n\n\n\n\n### Request / Router\nThe server is asynchronous. It means it is able to process \nmore requests at the same time.\n\n#### Server:\n\n```python\nimport asyncio\nfrom zmq_tubes import Tube, TubeNode, TubeMessage\n\n\nasync def handler(request: TubeMessage):\n  print(request.payload)\n  if request.payload == \'wait\':\n    await asyncio.sleep(10)\n  return request.create_response(request.payload)\n\n\ntube = Tube(\n  name=\'Server\',\n  addr=\'ipc:///tmp/req_router.pipe\',\n  server=True,\n  tube_type=\'ROUTER\'\n)\n\nnode = TubeNode()\nnode.register_tube(tube, \'test/#\')\nnode.register_handler(\'test/#\', handler)\nawait node.start()\n# output: \'wait\'\n# output: \'message\'\n```\n\n#### Client:\n\n```python\nimport asyncio\nfrom zmq_tubes import Tube, TubeNode\n\ntube = Tube(\n  name=\'Client\',\n  addr=\'ipc:///tmp/req_router.pipe\',\n  tube_type=\'REQ\'\n)\n\n\nasync def task(node, text):\n  print(await node.request(\'test/xxx\', text))\n\n\nnode = TubeNode()\nnode.register_tube(tube, \'test/#\')\nasyncio.create_task(task(node, \'wait\'))\nasyncio.create_task(task(node, \'message\'))\n# output: \'message\'\n# output: \'wait\'\n```\n\n\n\n\n### Dealer / Response\nThe client is asynchronous. It means it is able to send \nmore requests at the same time.\n\n#### Server:\n\n```python\nfrom zmq_tubes import Tube, TubeNode, TubeMessage\n\n\nasync def handler(request: TubeMessage):\n  print(request.payload)\n  return \'response\'\n  # or return requset.create_response(\'response\')\n\n\ntube = Tube(\n  name=\'Server\',\n  addr=\'ipc:///tmp/dealer_resp.pipe\',\n  server=True,\n  tube_type=\'REP\'\n)\n\nnode = TubeNode()\nnode.register_tube(tube, \'test/#\')\nnode.register_handler(\'test/#\', handler)\nawait node.start()\n# output: \'message\'\n```\n\n#### Client:\n\n```python\nfrom zmq_tubes import Tube, TubeNode, TubeMessage\n\ntube = Tube(\n  name=\'Client\',\n  addr=\'ipc:///tmp/dealer_resp.pipe\',\n  tube_type=\'DEALER\'\n)\n\n\nasync def handler(response: TubeMessage):\n  print(response.payload)\n\n\nnode = TubeNode()\nnode.register_tube(tube, \'test/#\')\nnode.register_handler(\'test/#\', handler)\n\nnode.send(\'test/xxx\', \'message\')\n\n# output: \'response\'\n```\n\n\n\n### Dealer / Router\nThe client and server are asynchronous. It means it is able to send and process \nmore requests/responses at the same time.\n\n#### Server:\n\n```python\nimport asyncio\nfrom zmq_tubes import Tube, TubeNode, TubeMessage\n\n\nasync def handler(request: TubeMessage):\n  print(request.payload)\n  if request.payload == \'wait\':\n    await asyncio.sleep(10)\n  return request.create_response(request.payload)\n\n\ntube = Tube(\n  name=\'Server\',\n  addr=\'ipc:///tmp/dealer_router.pipe\',\n  server=True,\n  tube_type=\'ROUTER\'\n)\n\nnode = TubeNode()\nnode.register_tube(tube, \'test/#\')\nnode.register_handler(\'test/#\', handler)\nawait node.start()\n# output: \'wait\'\n# output: \'message\'\n```\n\n#### Client:\n\n```python\nfrom zmq_tubes import Tube, TubeNode, TubeMessage\n\ntube = Tube(\n  name=\'Client\',\n  addr=\'ipc:///tmp/dealer_router.pipe\',\n  tube_type=\'DEALER\'\n)\n\n\nasync def handler(response: TubeMessage):\n  print(response.payload)\n\n\nnode = TubeNode()\nnode.register_tube(tube, \'test/#\')\nnode.register_handler(\'test/#\', handler)\n\nnode.send(\'test/xxx\', \'wait\')\nnode.send(\'test/xxx\', \'message\')\n\n# output: \'message\'\n# output: \'wait\'\n```\n\n\n\n### Dealer / Dealer\nThe client and server are asynchronous. It means it is able to send and process \nmore requests/responses at the same time.\n\n#### Server:\n\n```python\nfrom zmq_tubes import Tube, TubeNode, TubeMessage\n\ntube = Tube(\n  name=\'Server\',\n  addr=\'ipc:///tmp/dealer_dealer.pipe\',\n  server=True,\n  tube_type=\'DEALER\'\n)\n\n\nasync def handler(response: TubeMessage):\n  print(response.payload)\n\n\nnode = TubeNode()\nnode.register_tube(tube, \'test/#\')\nnode.register_handler(\'test/#\', handler)\n\nnode.send(\'test/xxx\', \'message from server\')\n# output: \'message from client\'\n```\n\n#### Client:\n\n```python\nfrom zmq_tubes import Tube, TubeNode, TubeMessage\n\ntube = Tube(\n  name=\'Client\',\n  addr=\'ipc:///tmp/dealer_dealer.pipe\',\n  tube_type=\'DEALER\'\n)\n\n\nasync def handler(response: TubeMessage):\n  print(response.payload)\n\n\nnode = TubeNode()\nnode.register_tube(tube, \'test/#\')\nnode.register_handler(\'test/#\', handler)\n\nnode.send(\'test/xxx\', \'message from client\')\n# output: \'message from server\'\n```\n',
    'author': 'Martin Korbel',
    'author_email': 'mkorbel@alps.cz',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/calcite/zmq_tubes',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
