"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventField = exports.RuleTargetInput = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * The input to send to the event target
 */
class RuleTargetInput {
    /**
     * Pass text to the event target
     *
     * May contain strings returned by EventField.from() to substitute in parts of the
     * matched event.
     */
    static fromText(text) {
        return new FieldAwareEventInput(text, InputType.Text);
    }
    /**
     * Pass text to the event target, splitting on newlines.
     *
     * This is only useful when passing to a target that does not
     * take a single argument.
     *
     * May contain strings returned by EventField.from() to substitute in parts
     * of the matched event.
     */
    static fromMultilineText(text) {
        return new FieldAwareEventInput(text, InputType.Multiline);
    }
    /**
     * Pass a JSON object to the event target
     *
     * May contain strings returned by EventField.from() to substitute in parts of the
     * matched event.
     */
    static fromObject(obj) {
        return new FieldAwareEventInput(obj, InputType.Object);
    }
    /**
     * Take the event target input from a path in the event JSON
     */
    static fromEventPath(path) {
        return new LiteralEventInput({ inputPath: path });
    }
    constructor() {
    }
}
exports.RuleTargetInput = RuleTargetInput;
/**
 * Event Input that is directly derived from the construct
 */
class LiteralEventInput extends RuleTargetInput {
    constructor(props) {
        super();
        this.props = props;
    }
    /**
     * Return the input properties for this input object
     */
    bind(_rule) {
        return this.props;
    }
}
/**
 * Input object that can contain field replacements
 *
 * Evaluation is done in the bind() method because token resolution
 * requires access to the construct tree.
 *
 * Multiple tokens that use the same path will use the same substitution
 * key.
 *
 * One weird exception: if we're in object context, we MUST skip the quotes
 * around the placeholder. I assume this is so once a trivial string replace is
 * done later on by EventBridge, numbers are still numbers.
 *
 * So in string context:
 *
 *    "this is a string with a <field>"
 *
 * But in object context:
 *
 *    "{ \"this is the\": <field> }"
 *
 * To achieve the latter, we postprocess the JSON string to remove the surrounding
 * quotes by using a string replace.
 */
class FieldAwareEventInput extends RuleTargetInput {
    constructor(input, inputType) {
        super();
        this.input = input;
        this.inputType = inputType;
    }
    bind(rule) {
        let fieldCounter = 0;
        const pathToKey = new Map();
        const inputPathsMap = {};
        function keyForField(f) {
            const existing = pathToKey.get(f.path);
            if (existing !== undefined) {
                return existing;
            }
            fieldCounter += 1;
            const key = f.displayHint || `f${fieldCounter}`;
            pathToKey.set(f.path, key);
            return key;
        }
        const self = this;
        class EventFieldReplacer extends core_1.DefaultTokenResolver {
            constructor() {
                super(new core_1.StringConcat());
            }
            resolveToken(t, _context) {
                if (!isEventField(t)) {
                    return core_1.Token.asString(t);
                }
                const key = keyForField(t);
                if (inputPathsMap[key] && inputPathsMap[key] !== t.path) {
                    throw new Error(`Single key '${key}' is used for two different JSON paths: '${t.path}' and '${inputPathsMap[key]}'`);
                }
                inputPathsMap[key] = t.path;
                return self.keyPlaceholder(key);
            }
        }
        const stack = core_1.Stack.of(rule);
        let resolved;
        if (this.inputType === InputType.Multiline) {
            // JSONify individual lines
            resolved = core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer(),
            });
            resolved = resolved.split('\n').map(stack.toJsonString).join('\n');
        }
        else {
            resolved = stack.toJsonString(core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer(),
            }));
        }
        if (Object.keys(inputPathsMap).length === 0) {
            // Nothing special, just return 'input'
            return { input: resolved };
        }
        return {
            inputTemplate: this.unquoteKeyPlaceholders(resolved),
            inputPathsMap,
        };
    }
    /**
     * Return a template placeholder for the given key
     *
     * In object scope we'll need to get rid of surrounding quotes later on, so
     * return a bracing that's unlikely to occur naturally (like tokens).
     */
    keyPlaceholder(key) {
        if (this.inputType !== InputType.Object) {
            return `<${key}>`;
        }
        return UNLIKELY_OPENING_STRING + key + UNLIKELY_CLOSING_STRING;
    }
    /**
     * Removing surrounding quotes from any object placeholders
     *
     * Those have been put there by JSON.stringify(), but we need to
     * remove them.
     */
    unquoteKeyPlaceholders(sub) {
        if (this.inputType !== InputType.Object) {
            return sub;
        }
        return core_1.Lazy.stringValue({ produce: (ctx) => core_1.Token.asString(deepUnquote(ctx.resolve(sub))) });
        function deepUnquote(resolved) {
            if (Array.isArray(resolved)) {
                return resolved.map(deepUnquote);
            }
            else if (typeof (resolved) === 'object' && resolved !== null) {
                for (const [key, value] of Object.entries(resolved)) {
                    resolved[key] = deepUnquote(value);
                }
                return resolved;
            }
            else if (typeof (resolved) === 'string') {
                return resolved.replace(OPENING_STRING_REGEX, '<').replace(CLOSING_STRING_REGEX, '>');
            }
            return resolved;
        }
    }
}
const UNLIKELY_OPENING_STRING = '<<${';
const UNLIKELY_CLOSING_STRING = '}>>';
const OPENING_STRING_REGEX = new RegExp(regexQuote('"' + UNLIKELY_OPENING_STRING), 'g');
const CLOSING_STRING_REGEX = new RegExp(regexQuote(UNLIKELY_CLOSING_STRING + '"'), 'g');
/**
 * Represents a field in the event pattern
 */
class EventField {
    /**
     *
     * @param path the path to a field in the event pattern
     */
    constructor(path) {
        this.path = path;
        this.displayHint = this.path.replace(/^[^a-zA-Z0-9_-]+/, '').replace(/[^a-zA-Z0-9_-]/g, '-');
        Object.defineProperty(this, EVENT_FIELD_SYMBOL, { value: true });
        this.creationStack = core_1.captureStackTrace();
    }
    /**
     * Extract the event ID from the event
     */
    static get eventId() {
        return this.fromPath('$.id');
    }
    /**
     * Extract the detail type from the event
     */
    static get detailType() {
        return this.fromPath('$.detail-type');
    }
    /**
     * Extract the source from the event
     */
    static get source() {
        return this.fromPath('$.source');
    }
    /**
     * Extract the account from the event
     */
    static get account() {
        return this.fromPath('$.account');
    }
    /**
     * Extract the time from the event
     */
    static get time() {
        return this.fromPath('$.time');
    }
    /**
     * Extract the region from the event
     */
    static get region() {
        return this.fromPath('$.region');
    }
    /**
     * Extract a custom JSON path from the event
     */
    static fromPath(path) {
        return new EventField(path).toString();
    }
    resolve(_ctx) {
        return this.path;
    }
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    /**
     * Convert the path to the field in the event pattern to JSON
     */
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.EventField = EventField;
var InputType;
(function (InputType) {
    InputType[InputType["Object"] = 0] = "Object";
    InputType[InputType["Text"] = 1] = "Text";
    InputType[InputType["Multiline"] = 2] = "Multiline";
})(InputType || (InputType = {}));
function isEventField(x) {
    return EVENT_FIELD_SYMBOL in x;
}
const EVENT_FIELD_SYMBOL = Symbol.for('@aws-cdk/aws-events.EventField');
/**
 * Quote a string for use in a regex
 */
function regexQuote(s) {
    return s.replace(/[.?*+^$[\]\\(){}|-]/g, '\\$&');
}
//# sourceMappingURL=data:application/json;base64,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