"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateHostedZone = exports.PublicHostedZone = exports.HostedZone = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const record_set_1 = require("./record-set");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
/**
 * Container for records, and records contain information about how to route traffic for a
 * specific domain, such as example.com and its subdomains (acme.example.com, zenith.example.com)
 */
class HostedZone extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * VPCs to which this hosted zone will be added
         */
        this.vpcs = new Array();
        jsiiDeprecationWarnings._aws_cdk_aws_route53_HostedZoneProps(props);
        util_1.validateZoneName(props.zoneName);
        const resource = new route53_generated_1.CfnHostedZone(this, 'Resource', {
            name: props.zoneName + '.',
            hostedZoneConfig: props.comment ? { comment: props.comment } : undefined,
            queryLoggingConfig: props.queryLogsLogGroupArn ? { cloudWatchLogsLogGroupArn: props.queryLogsLogGroupArn } : undefined,
            vpcs: core_1.Lazy.any({ produce: () => this.vpcs.length === 0 ? undefined : this.vpcs }),
        });
        this.hostedZoneId = resource.ref;
        this.hostedZoneNameServers = resource.attrNameServers;
        this.zoneName = props.zoneName;
        for (const vpc of props.vpcs || []) {
            this.addVpc(vpc);
        }
    }
    get hostedZoneArn() {
        return util_1.makeHostedZoneArn(this, this.hostedZoneId);
    }
    /**
     * Import a Route 53 hosted zone defined either outside the CDK, or in a different CDK stack
     *
     * Use when hosted zone ID is known. Hosted zone name becomes unavailable through this query.
     *
     * @param scope the parent Construct for this Construct
     * @param id  the logical name of this Construct
     * @param hostedZoneId the ID of the hosted zone to import
     */
    static fromHostedZoneId(scope, id, hostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = hostedZoneId;
            }
            get zoneName() {
                throw new Error('HostedZone.fromHostedZoneId doesn\'t support "zoneName"');
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports a hosted zone from another stack.
     *
     * Use when both hosted zone ID and hosted zone name are known.
     *
     * @param scope the parent Construct for this Construct
     * @param id  the logical name of this Construct
     * @param attrs the HostedZoneAttributes (hosted zone ID and hosted zone name)
     */
    static fromHostedZoneAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_route53_HostedZoneAttributes(attrs);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = attrs.hostedZoneId;
                this.zoneName = attrs.zoneName;
            }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Lookup a hosted zone in the current account/region based on query parameters.
     * Requires environment, you must specify env for the stack.
     *
     * Use to easily query hosted zones.
     *
     * @see https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     */
    static fromLookup(scope, id, query) {
        jsiiDeprecationWarnings._aws_cdk_aws_route53_HostedZoneProviderProps(query);
        const DEFAULT_HOSTED_ZONE = {
            Id: 'DUMMY',
            Name: query.domainName,
        };
        const response = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.HOSTED_ZONE_PROVIDER,
            dummyValue: DEFAULT_HOSTED_ZONE,
            props: query,
        }).value;
        // CDK handles the '.' at the end, so remove it here
        if (response.Name.endsWith('.')) {
            response.Name = response.Name.substring(0, response.Name.length - 1);
        }
        response.Id = response.Id.replace('/hostedzone/', '');
        return HostedZone.fromHostedZoneAttributes(scope, id, {
            hostedZoneId: response.Id,
            zoneName: response.Name,
        });
    }
    /**
     * Add another VPC to this private hosted zone.
     *
     * @param vpc the other VPC to add.
     */
    addVpc(vpc) {
        this.vpcs.push({ vpcId: vpc.vpcId, vpcRegion: core_1.Stack.of(vpc).region });
    }
}
exports.HostedZone = HostedZone;
_a = JSII_RTTI_SYMBOL_1;
HostedZone[_a] = { fqn: "@aws-cdk/aws-route53.HostedZone", version: "1.149.0" };
/**
 * Create a Route53 public hosted zone.
 *
 * @resource AWS::Route53::HostedZone
 */
class PublicHostedZone extends HostedZone {
    constructor(scope, id, props) {
        super(scope, id, props);
        jsiiDeprecationWarnings._aws_cdk_aws_route53_PublicHostedZoneProps(props);
        if (props.caaAmazon) {
            new record_set_1.CaaAmazonRecord(this, 'CaaAmazon', {
                zone: this,
            });
        }
        if (!props.crossAccountZoneDelegationPrincipal && props.crossAccountZoneDelegationRoleName) {
            throw Error('crossAccountZoneDelegationRoleName property is not supported without crossAccountZoneDelegationPrincipal');
        }
        if (props.crossAccountZoneDelegationPrincipal) {
            this.crossAccountZoneDelegationRole = new iam.Role(this, 'CrossAccountZoneDelegationRole', {
                roleName: props.crossAccountZoneDelegationRoleName,
                assumedBy: props.crossAccountZoneDelegationPrincipal,
                inlinePolicies: {
                    delegation: new iam.PolicyDocument({
                        statements: [
                            new iam.PolicyStatement({
                                actions: ['route53:ChangeResourceRecordSets'],
                                resources: [this.hostedZoneArn],
                            }),
                            new iam.PolicyStatement({
                                actions: ['route53:ListHostedZonesByName'],
                                resources: ['*'],
                            }),
                        ],
                    }),
                },
            });
        }
    }
    /**
     * Import a Route 53 public hosted zone defined either outside the CDK, or in a different CDK stack
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param publicHostedZoneId the ID of the public hosted zone to import
     */
    static fromPublicHostedZoneId(scope, id, publicHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = publicHostedZoneId;
            }
            get zoneName() { throw new Error('cannot retrieve "zoneName" from an an imported hosted zone'); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
    addVpc(_vpc) {
        throw new Error('Cannot associate public hosted zones with a VPC');
    }
    /**
     * Adds a delegation from this zone to a designated zone.
     *
     * @param delegate the zone being delegated to.
     * @param opts     options for creating the DNS record, if any.
     */
    addDelegation(delegate, opts = {}) {
        jsiiDeprecationWarnings._aws_cdk_aws_route53_IPublicHostedZone(delegate);
        jsiiDeprecationWarnings._aws_cdk_aws_route53_ZoneDelegationOptions(opts);
        new record_set_1.ZoneDelegationRecord(this, `${this.zoneName} -> ${delegate.zoneName}`, {
            zone: this,
            recordName: delegate.zoneName,
            nameServers: delegate.hostedZoneNameServers,
            comment: opts.comment,
            ttl: opts.ttl,
        });
    }
}
exports.PublicHostedZone = PublicHostedZone;
_b = JSII_RTTI_SYMBOL_1;
PublicHostedZone[_b] = { fqn: "@aws-cdk/aws-route53.PublicHostedZone", version: "1.149.0" };
/**
 * Create a Route53 private hosted zone for use in one or more VPCs.
 *
 * Note that `enableDnsHostnames` and `enableDnsSupport` must have been enabled
 * for the VPC you're configuring for private hosted zones.
 *
 * @resource AWS::Route53::HostedZone
 */
class PrivateHostedZone extends HostedZone {
    constructor(scope, id, props) {
        super(scope, id, props);
        jsiiDeprecationWarnings._aws_cdk_aws_route53_PrivateHostedZoneProps(props);
        this.addVpc(props.vpc);
    }
    /**
     * Import a Route 53 private hosted zone defined either outside the CDK, or in a different CDK stack
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param privateHostedZoneId the ID of the private hosted zone to import
     */
    static fromPrivateHostedZoneId(scope, id, privateHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = privateHostedZoneId;
            }
            get zoneName() { throw new Error('cannot retrieve "zoneName" from an an imported hosted zone'); }
            get hostedZoneArn() {
                return util_1.makeHostedZoneArn(this, this.hostedZoneId);
            }
        }
        return new Import(scope, id);
    }
}
exports.PrivateHostedZone = PrivateHostedZone;
_c = JSII_RTTI_SYMBOL_1;
PrivateHostedZone[_c] = { fqn: "@aws-cdk/aws-route53.PrivateHostedZone", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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