# -*- coding: utf-8 -*-
#
# Copyright 2020 - Swiss Data Science Center (SDSC)
# A partnership between École Polytechnique Fédérale de Lausanne (EPFL) and
# Eidgenössische Technische Hochschule Zürich (ETHZ).
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Renku service controller mixin."""

from abc import ABCMeta, abstractmethod

from git import Repo

from renku.core.commands.save import repo_sync
from renku.core.errors import RenkuException
from renku.core.utils.contexts import chdir
from renku.service.controllers.utils.remote_project import RemoteProject


class ReadOperationMixin(metaclass=ABCMeta):
    """Read operation mixin."""

    def __init__(self, cache, user_data, request_data):
        """Read operation mixin for controllers."""
        self.user = cache.ensure_user(user_data)

        self.cache = cache
        self.user_data = user_data
        self.request_data = request_data

        self.project_path = None

    @property
    @abstractmethod
    def context(self):
        """Operation context."""
        raise NotImplementedError

    @abstractmethod
    def renku_op(self):
        """Implements operation for the controller."""
        raise NotImplementedError

    def execute_op(self):
        """Execute renku operation which controller implements."""
        if "project_id" in self.context:
            return self.local()
        elif "git_url" in self.context:
            return self.remote()
        else:
            raise RenkuException("context does not contain `project_id` or `git_url`")

    def local(self):
        """Execute operation against service cache."""
        project = self.cache.get_project(self.user, self.context["project_id"])
        self.project_path = project.abs_path

        with chdir(self.project_path):
            return self.renku_op()

    def remote(self):
        """Execute operation against remote project."""
        project = RemoteProject(self.user_data, self.request_data)

        with project.remote() as path:
            self.project_path = path
            return self.renku_op()


class ReadWithSyncOperation(ReadOperationMixin, metaclass=ABCMeta):
    """Sync operation mixin."""

    def sync(self, remote="origin"):
        """Sync with remote."""
        if self.project_path is None:
            raise RenkuException("unable to sync with remote since no operation has been executed")

        _, remote_branch = repo_sync(Repo(self.project_path), remote=remote)
        return remote_branch

    def execute_and_sync(self, remote="origin"):
        """Execute operation which controller implements and sync with the remote."""
        # NOTE: This will return the operation result as well as name of the branch to which it has been pushed.
        return self.execute_op(), self.sync(remote=remote)
